# Copyright (C) 2006,2007 Daiki Ueno <ueno@unixuser.org>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'treil/color'
require 'rexml/document'
require 'date'

module Treil
  class Reader
    def initialize(input, config)
      @input = input
      @config = config
    end

    @@reader = Array.new
    def self.instance(input, config, class_name = nil)
      if class_name
	klass = Treil.const_get(class_name)
      else
	klass = @@reader.detect {|klass| klass.accept?(input)}
      end
      unless klass
        raise "Can't find reader for #{input}"
      end
      unless klass.accept?(input)
        raise "#{klass} can't read from #{input}"
      end
      klass.new(input, config)
    end

    def self.register(reader)
      @@reader.push(reader)
    end

    def self.list
      @@reader.collect {|klass| klass.name.sub(/\ATreil::/, '')}
    end
  end

  class DirReader < Reader
    def initialize(input, config)
      super
      @input = Pathname.new(@input).realpath
    end

    def self.accept?(input)
      Pathname.new(input).directory?
    end

    class DirTree
      def initialize(name, parent = nil, size = 0, children = Array.new)
        @name = name
        @parent = parent
        @size = size
        @children = children
      end
      attr_reader :name, :children
      attr_accessor :size
    end

    def _read(path, parent = nil, &block)
      yield path if block_given?
      tree = DirTree.new(path.basename.to_s, parent)
      if path.directory?
        path.opendir do |dir|
          dir.each do |name|
            next if name == '.' || name == '..'
            next if self.excludes.detect {|exclude| File.fnmatch(exclude, name)}
            _read(path + name, tree, &block)
          end
        end
      elsif path.file?
        tree.size = path.size
      end
      if parent
        parent.size += tree.size
        parent.children.push(tree)
      end
      tree
    end

    def read
      tree = _read(@input)
      Treil::Tree.new(tree)
    end

    NewerRGB = [1.0, 0.0, 0.0]
    OlderRGB = [0.0, 1.0, 0.0]

    def set_background_by_timestamp(timestamp)
      newer_rgb = @config['']['background_color_newer'] || NewerRGB
      older_rgb = @config['']['background_color_older'] || OlderRGB
      newer_hsv = Treil::Color.rgb2hsv(*newer_rgb)
      older_hsv = Treil::Color.rgb2hsv(*older_rgb)
      values = timestamp.values.sort
      median = values[values.length / 2]
      lambda = values.inject(0) {|accu, value| accu + (values[-1] - value)} / values.length
      timestamp.each do |key, val|
        key = key.relative_path_from(@input.parent).to_s
        t = Math.exp((val - values[-1]) / lambda)
        if val > median
          hsv = newer_hsv
        else
          hsv = older_hsv
          t = 1.0 - t
        end
        @config[key] ||= Hash.new
        @config[key]['background_color'] =
          Treil::Color.hsv2rgb(hsv[0], hsv[1], hsv[2] * t)
      end
    end

    def excludes
      @config['']['excludes']
    end
  end

  class CVSReader < DirReader
    def self.accept?(input)
      (Pathname.new(input) + 'CVS').directory?
    end

    def run_cvs_log(path, branch = nil)
      args = Array.new
      args.push('-l', '-N')
      if branch
        args.push("-r#{branch}.")
      else
        args.push('-rHEAD')
      end
      args.push(path.relative_path_from(@input).to_s)
      pr, pw = IO.pipe
      pid = fork do
        pr.close
        $stdout.reopen(pw)
	Dir.chdir(@input.to_s)
        exec('cvs', 'log', *args)
      end
      pw.close
      pr.read
    end

    def read
      timestamp = Hash.new
      tree = _read(@input) do |path|
	next unless path.directory?
        entries = path + 'CVS/Entries'
        next unless entries.exist?
        tag = path + 'CVS/Tag'
        if tag.exist?
          branch = tag.read.sub!(/\AT/, '')
        else
          branch = nil
        end
        log = run_cvs_log(path, branch)
        log.split(/^=+\n\n/).each do |s|
          if s =~ /^Working file: (.*)/
            file = $1
          else
            next
          end
          if s =~ /^date: (.*?);/
            date = $1
          else
            next
          end
          timestamp[@input + file] = Date.parse($1)
        end
      end
      set_background_by_timestamp(timestamp)
      Treil::Tree.new(tree)
    end

    def excludes
      super + ['CVS']
    end
  end

  class SVNReader < DirReader
    def self.accept?(input)
      (Pathname.new(input) + '.svn').directory?
    end

    def read
      pr, pw = IO.pipe
      pid = fork do
        pr.close
        $stdout.reopen(pw)
        exec('svn', 'info', '--xml', '-R', @input)
      end
      pw.close
      timestamp = Hash.new
      doc = REXML::Document.new(pr.read)
      doc.each_element('/info/entry') do |entry|
        path = entry.attributes['path']
        date = entry.get_text('commit/date')
        timestamp[Pathname.new(path)] = Date.parse(date.to_s)
      end
      set_background_by_timestamp(timestamp)
      super
    end

    def excludes
      super + ['.svn']
    end
  end

  class GitReader < DirReader
    def self.accept?(input)
      (Pathname.new(input) + '.git').directory?
    end

    def read_timestamp
      timestamp = Hash.new
      pr, pw = IO.pipe
      pid = fork do
        pr.close
        $stdout.reopen(pw)
	Dir.chdir(@input.to_s)
        exec('git-whatchanged', '-m', '--pretty=medium')
      end
      pw.close
      date = nil
      loop do
        line = pr.gets
        break unless line
        case line
        when /\ADate:\s*(.*)/
          date = Date.parse($1)
        when /\A:\S+ \S+ \S+ \S+ [AM]\t/
          path = @input + Pathname.new($'.chomp)
          if !timestamp[path] || timestamp[path] < date
            timestamp[path] = date
          end
        end
      end
      timestamp
    end

    def read
      timestamp = read_timestamp
      set_background_by_timestamp(timestamp)
      super
    end

    def excludes
      super + ['.git']
    end
  end

  class HgReader < DirReader
    def self.accept?(input)
      (Pathname.new(input) + '.hg').directory?
    end

    def run_hg_history(path)
      pr, pw = IO.pipe
      pid = fork do
        pr.close
        $stdout.reopen(pw)
	Dir.chdir(@input.to_s)
        exec('hg', 'history', '-l', '1', path.relative_path_from(@input).to_s)
      end
      pw.close
      pr.read
    end

    def read
      timestamp = Hash.new
      tree = _read(@input) do |path|
	next if path.directory?
	str = run_hg_history(path)
	if str =~ /^date:\s*(.*)/
          timestamp[path] = Date.parse($1)
	end
      end
      set_background_by_timestamp(timestamp)
      Treil::Tree.new(tree)
    end

    def excludes
      super + ['.hg']
    end
  end

  class ChangeLogReader < DirReader
    def self.accept?(input)
      !Pathname.glob("#{input}/**/ChangeLog*").empty?
    end

    def read
      timestamp = Hash.new
      Pathname.glob("#{@input}/**/ChangeLog*") do |changelog|
	time = nil
        changelog.each_line do |line|
	  case line
	  when /\A(\d{4})-(\d{2})-(\d{2})/
	    begin
	      time = Time.utc($1.to_i, $2.to_i, $3.to_i)
	    rescue
	      next
	    end
	  when /\A\t\* ([^ \t\n:,]+)/
	    next unless time
            key = changelog.dirname + $1
	    next unless key.exist?
	    if !timestamp[key] || timestamp[key] < time
	      timestamp[key] = time
	    end
	  end
        end
      end
      set_background_by_timestamp(timestamp)
      super
    end
  end

  class CVSReader
    register(self)
  end

  class SVNReader
    register(self)
  end

  class GitReader
    register(self)
  end

  class HgReader
    register(self)
  end

  class ChangeLogReader
    register(self)
  end

  class DirReader
    register(self)
  end
end
