/*
    Tucnak - VHF contest log
    Copyright (C) 2002-2006  Ladislav Vaiz <ok1zia@nagano.cz>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "header.h"


struct dw *dw;
static gchar *dw_next;  /* wwl1 */
static gchar *wd_next;  /* dxc1 */

struct dw *init_dw(void){
    struct dw *dw;

    dw = g_new0(struct dw,1 );

    dw->pd = g_hash_table_new(g_str_hash, g_str_equal);
    dw->dw = g_hash_table_new(g_str_hash, g_str_equal);
    dw->wd = g_hash_table_new(g_str_hash, g_str_equal);

    dw_next = NULL;
    wd_next = NULL;

    return dw;
}



gboolean free_gpointer_item(gpointer key, gpointer value, gpointer user_data){
    if (key) g_free(key);
    if (value) g_free(value);
    return TRUE;
}

gboolean free_dw_item(gpointer key, gpointer value, gpointer user_data){
    struct dw_item *dwi;
    
    g_free(key);
    dwi = (struct dw_item *)value;
    if (dwi->wwl0) g_free(dwi->wwl0);
    if (dwi->wwl1) g_free(dwi->wwl1);

    g_hash_table_foreach_remove(dwi->wwls, free_gpointer_item, NULL);
    g_hash_table_destroy(dwi->wwls);
    
    g_free(value);
    return FALSE;
}

gboolean free_wd_item(gpointer key, gpointer value, gpointer user_data){
    struct wd_item *wdi;
    
    g_free(key);
    wdi = (struct wd_item *)value;
    if (wdi->dxc0) g_free(wdi->dxc0);
    if (wdi->dxc1) g_free(wdi->dxc1);
    g_free(value);
    return FALSE;
}


void free_dw(struct dw *dw){
    g_hash_table_foreach_remove(dw->pd, free_gpointer_item, NULL);
    g_hash_table_destroy(dw->pd);
    g_hash_table_foreach_remove(dw->dw, free_dw_item, NULL);
    g_hash_table_destroy(dw->dw);
    g_hash_table_foreach_remove(dw->wd, free_wd_item, NULL);
    g_hash_table_destroy(dw->wd);
}

gint get_pd_size(struct dw *dw){
    return g_hash_table_size(dw->pd);
}

gint get_dw_size(struct dw *dw){
    return g_hash_table_size(dw->dw);
}

gint get_wd_size(struct dw *dw){
    return g_hash_table_size(dw->wd);
}


#define DW_DELIM " \t\r\n"

gchar *safe_fgets(GString *gs, FILE *f, int stripcomment){
    char s[102], *c;
    int len;
    
    
    g_string_truncate(gs, 0);

    while (1){
        c = fgets(s, 100, f);
        if (!c) return NULL;
        
        len = strlen(s);
        if (len>0 && s[len-1]=='\n'){ /* MAC ?*/
            s[len-1]='\0';
            if (len>1 && s[len-2]=='\r') s[len-2]='\0';
            g_string_append(gs, s);
            break;
        }
        g_string_append(gs, s);
    }
    
    if (stripcomment){
        char *c;

        c = index(gs->str, '#');
        if (c) g_string_truncate(gs, c - gs->str);
    }
        
     /*dbg("%3d '%s'\n",gs->len, gs->str);*/
    return gs->str;    
}


void add_pd(struct dw *dw, gchar *prefix, gchar *dxc){
    gpointer orig_prefix, orig_dxc;

/*    dbg("add_pd(prefix='%s', dxc='%s'\n", prefix, dxc );*/
    
    if (g_hash_table_lookup_extended(dw->pd, (gpointer) prefix,
               &orig_prefix, &orig_dxc)){
        
        g_hash_table_remove(dw->pd, orig_prefix);
        g_free(orig_prefix);
        g_free(orig_dxc);
    } 
    g_hash_table_insert(dw->pd, g_strdup(prefix), g_strdup(dxc) );
}


struct dw_item *add_dxc(struct dw *dw, gchar *dxc, gchar *wwl0, gchar *wwl1){
    struct dw_item *dxci;

    /*dbg("add_dxc(dxc='%s', wwl='%s' '%s')\n",dxc, wwl0, wwl1);*/
    

    dxci = g_hash_table_lookup(dw->dw, dxc);
    if (!dxci) {
       dxci = g_new0(struct dw_item, 1); 
       
       dxci->wwls = g_hash_table_new(g_str_hash, g_str_equal);
       g_hash_table_insert(dw->dw, g_strdup(dxc), dxci);
    }
    
    if (dxci->wwl0 && wwl0) { g_free(dxci->wwl0); dxci->wwl0 = NULL; }
    if (dxci->wwl1 && wwl1) { g_free(dxci->wwl1); dxci->wwl1 = NULL; }

    dxci->wwl0 = g_strdup(wwl0);
    dxci->wwl1 = g_strdup(wwl1);
    
    return dxci;
}

void add_dw(struct dw_item *dxci, gchar *dxc, gchar *wwl){
    gpointer orig_wwl, orig_dxc;
    char wwl1[6];

    if (!wwl) return;
    
    
    
    if (g_hash_table_lookup_extended(dxci->wwls, (gpointer) wwl,
               &orig_wwl, &orig_dxc)){
        
        g_hash_table_remove(dxci->wwls, orig_wwl);
        g_free(orig_wwl);
        g_free(orig_dxc);
    } 
    safe_strncpy(wwl1, wwl, 5);
    /*dbg("add_dw(dxc='%s', wwl='%s')\n", dxc, wwl1);*/
    g_hash_table_insert(dxci->wwls, g_strdup(wwl1), g_strdup(dxc) );
    
}


void add_wd(struct dw *dw, gchar *wwl, gchar *dxc){
    struct wd_item *wdi;
    
    if (!wwl || !dxc) return;
   /*dbg("add_wd(wwl='%s', dxc='%s')  size=%d \n",wwl, dxc, g_hash_table_size(dw->wd));*/
    
    wdi = g_hash_table_lookup(dw->wd, wwl);
    if (!wdi){
        wdi = g_new0(struct wd_item, 1);

        g_hash_table_insert(dw->wd, g_strdup(wwl), wdi);
      /*  dbg("   adding wwl '%s' \n", wwl);*/
    }
    
    if (!wdi->dxc0) {
        wdi->dxc0 = g_strdup(dxc);
        /*dbg("      adding dxc0 '%s' \n", wdi->dxc0);*/
        return;
    }
    
    if (!wdi->dxc1) {
        wdi->dxc1 = g_strdup(dxc);       
       /* dbg("      adding dxc1 '%s' \n", wdi->dxc1);*/
        return;
    }
    
    return;
}



int load_dw_from_file(struct dw *dw, gchar *filename){
    FILE *f;
    GString *pref, *wwls, *centers; 
    gchar *dxc, *prefix, *wwl, *c0, *c1;
    struct dw_item *dxci;
    char *token_ptr;
    
    f = fopen(filename, "rt");
    if (!f) {
        dbg("Can't open '%s'\n",filename);
        return -1;
    }

    pref    = g_string_sized_new(100);
    wwls    = g_string_sized_new(100);
    centers = g_string_sized_new(100);
    

    while(1){
        do {
           if (!safe_fgets(pref,  f, 1)) goto abort;
        }while (pref->len==0);
        
        dxc = strtok_r(pref->str, DW_DELIM, &token_ptr);
        if (!dxc) continue;
        add_pd(dw, dxc, dxc);
        
        while ((prefix=strtok_r(NULL, DW_DELIM, &token_ptr))!=NULL){
            add_pd(dw, prefix, dxc);
        } 
        
        if (!safe_fgets(wwls,    f, 1)) break;
        if (!wwls->len) continue;

        if (!safe_fgets(centers, f, 1)) break;

        if (centers->len)  c0 = strtok_r(centers->str, DW_DELIM, &token_ptr);
        else               c0 = strtok_r(wwls->str,    DW_DELIM, &token_ptr);
        
        c1 = strtok_r(NULL, DW_DELIM, &token_ptr);
            

        dxci = add_dxc(dw, dxc, c0, c1); 
        
        add_dw(dxci, dxc, c0);
        add_dw(dxci, dxc, c1);
        add_wd(dw, c0, dxc);
        add_wd(dw, c1, dxc);

        if (centers->len){  
            c0 = strtok_r(wwls->str, DW_DELIM, &token_ptr);
            add_dw(dxci, dxc, c0);
            add_wd(dw, c0, dxc);
        }
        
        while((wwl = strtok_r(NULL, DW_DELIM, &token_ptr))!=NULL){
            add_dw(dxci, dxc, wwl);
            add_wd(dw, wwl, dxc);
        }  
        
    }
abort:;
    fclose(f);
    g_string_free(pref, 1);
    g_string_free(wwls, 1);
    g_string_free(centers, 1);
    return 0;
}

#define CTY_DELIM ":"
#define PREF_DELIM ","

int load_cty_from_file(struct dw *dw, gchar *filename){
    FILE *f;
    GString *gs, *gs_next; 
    gchar **line, **prefs;
    int i;
    gchar *dxc;
    char wwl4[6];

/*        , *prefix, *wwl, *c0, *c1;*/
    struct dw_item *dxci;
    
    
    f = fopen(filename, "rt");
    if (!f) {
        dbg("Can't open '%s'\n", filename);
        return -1;
    }

    gs      = g_string_sized_new(100);
    gs_next = g_string_sized_new(100);
    

    while(1){
        if (gs_next->len>0){
            g_string_assign(gs, gs_next->str);
        }else{
            if (!safe_fgets(gs,  f, 0)) break;
        }
        
        while(1){
            if (!safe_fgets(gs_next,  f, 0)){
                g_string_truncate(gs_next,0); /* next safe_fgets breaks loop */
                break;
            }
            
            if (gs_next->str[0]==' '){ 
                g_string_append(gs, gs_next->str);
            }else{
                break;
            }
        }

        
        line = g_strsplit(gs->str, CTY_DELIM, 9);
        for (i=0;i<9;i++) if (line[i] ==NULL) break;
        if (i!=9) continue;
        
        g_strstrip(line[4]);
        g_strstrip(line[5]);
        g_strstrip(line[7]);
        g_strstrip(line[8]);
        
        
        dxc=line[7];
        if (index(dxc, '/')) {
            g_strfreev(line);
            continue;
        }
        
        if (!dxc) continue;
        add_pd(dw, dxc, dxc);
        
/*        dbg("h=%s, w=%s, d=%s, p=%s\n", line[5], line[6], line[7], line[8]);*/
        
        prefs = g_strsplit(line[8], PREF_DELIM, 0);
        for (i=0; prefs[i]!=NULL; i++){
            char *c;

            g_strstrip(prefs[i]);
            if (index(prefs[i],'/')) continue;
            if ((c=index(prefs[i],'('))!=NULL) *c='\0';
            if ((c=index(prefs[i],'['))!=NULL) *c='\0';
            if ((c=index(prefs[i],';'))!=NULL) *c='\0';
            
            if (strlen(prefs[i])>2 && !isdigit(prefs[i][ strlen(prefs[i])-1])) continue;
            
            /*dbg("\t'%s'", prefs[i]);  */
            add_pd(dw, prefs[i], dxc);
            
        }
        g_strfreev(prefs);
        /*dbg("\n");*/
        compute_wwl4(wwl4, -atof(line[5]), atof(line[4]));

        /*dbg("h=%s, w=%s, d=%s, wwl4=%s \n", line[4], line[5], line[7], wwl4);*/
        dxci = add_dxc(dw, dxc, wwl4, NULL); 
        
        g_strfreev(line);
        
    }
    fclose(f);
    g_string_free(gs, 1);
    g_string_free(gs_next, 1);
    return 0;
}


void read_dw_files(struct dw *dw){
    gchar *s;
    
    load_cty_from_file(dw, "/etc/cty.dat");
    s = g_strconcat(getenv("HOME"), "/tucnak/cty.dat", NULL);
    load_cty_from_file(dw, s);
    g_free(s);
    
    load_dw_from_file(dw, "/etc/tucnakdw");
    s = g_strconcat(getenv("HOME"), "/tucnak/tucnakdw", NULL);
    load_dw_from_file(dw, s);
    g_free(s);
}




int wwl_is_ok_by_call(struct dw *dw, gchar *wwl, gchar *call){
    char dxcc[25];
    struct dw_item *dxci;
    char swwl[5];
    
    if (!wwl || !call) return 1;
    
    get_dxcc(dw, dxcc, call);
    /*dbg("dxcc=%s\n", dxcc);*/
            
    dxci = g_hash_table_lookup(dw->dw, dxcc);
    if (!dxci) {
/*        dbg("can't find dxcc (%s) for call %s \n", dxcc, call);*/
        return 0;
    }

    safe_strncpy(swwl, wwl, 5);
    if (g_hash_table_lookup(dxci->wwls, swwl)){
/*       dbg("is OK\n"); */
       return 1; 
    }
/*    dbg("loc not found\n");*/
    return 0;
}


/* 0=OK
   1=WARN, known call from unknown locator
   2=ERR,  prefix and big wwl dont match
*/ 
int get_susp(struct cw *cw, struct dw *dw, gchar *call, gchar *wwl){
    int susp=0;
    
    if (!call || !wwl || !*call || !*wwl) return 0;
    
    if (!wwl_is_ok_by_call(dw, wwl, call)) 
        susp=2; /* bad country */
    else{
        gchar *wwl0, *wwl1;

        wwl0=find_wwl_by_call(cw, call);
        wwl1=find_wwl_by_call(cw, NULL); 

        if (wwl0!=NULL){ /* we don't know this call, cannot be suspicious */
            susp = 1;
            if (wwl0 && strcasecmp(wwl0, wwl)==0) susp = 0; 
            if (wwl1 && strcasecmp(wwl1, wwl)==0) susp = 0; 
        }
    }
    return susp;
}


gchar *find_wwl_by_dxc(struct dw *dw,  gchar *dxc){
    struct dw_item *dxci;

    if (!dxc) return dw_next;
    dw_next = NULL;
    
    dxci = g_hash_table_lookup(dw->dw, dxc);
    if (!dxci) return NULL;

    dw_next = dxci->wwl1;

    return dxci->wwl0;
    
}


gchar *find_dxc_by_wwl(struct dw *dw, gchar *wwl){
    struct wd_item *wdi;
    char s[10];

    if (!wwl) return wd_next;
    wd_next = NULL;

    safe_strncpy(s, wwl, 10);
    if (strlen(s)>4) s[4]='\0';

    wdi = g_hash_table_lookup(dw->wd, s);
    if (!wdi) return NULL;
    
    wd_next = wdi->dxc1;
    return wdi->dxc0;
    
}


char *get_dxcc(struct dw *dw, char *buf, char *call){ 
    char *c0, *c1, *dxc;
    
    c0=c1=NULL;
    safe_strncpy0(buf, call, 20);
    if (regmatch(call, "^(.*[0-9])[A-Z]*", &c0, &c1, NULL)==0){
        
        safe_strncpy0(buf, c1, 20);
        while (*c1!='\0'){
           dxc = g_hash_table_lookup(dw->pd, c1);
           if (dxc) {
      /*        dbg("get_dxcc(%s)='%s'\n", call, c1); */
              safe_strncpy0(buf, dxc, 20);
              break;
           }
           c1[strlen(c1)-1]='\0'; 
        }
    }
    if (c0) mem_free(c0);
    if (c1) mem_free(c1);
/*    dbg("get_dxcc='%s'\n", dxc);*/
    return buf;  
}

void check_susp_one(gpointer key, gpointer value, gpointer user_data){
    gchar *call;
    struct cw_item *cwi;

    call = (gchar *) key;
    cwi  = (struct cw_item *) value;
    if (cwi->wwl0) {
        if (get_susp(cw,dw,call,cwi->wwl0)==2){
            dbg("suspicious %-14s %-6s %08d\n", call, cwi->wwl0, cwi->stamp0);
        } 
    }
    if (cwi->wwl1) {
        if (get_susp(cw,dw,call,cwi->wwl1)==2){
            dbg("suspicious %-14s %-6s %08d\n", call, cwi->wwl1, cwi->stamp1);
        } 
    }
}
    

void check_susp(void){
    t_hash_table_foreach(cw->cw, check_susp_one, NULL);
}
