/*
    Tucnak - VHF contest log
    Copyright (C) 2002-2006  Ladislav Vaiz <ok1zia@nagano.cz>
    and authors of web browser Links 0.96

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "header.h"

extern int starting_sbrk;

void menu_set_language(struct terminal *term, void *pcp, struct session *ses);

void menu_about(struct terminal *term, void *d, struct session *ses)
{
    msg_box(term, NULL, TEXT(T_ABOUT), AL_CENTER, TEXT(T_TUCNAK__TACLOG_LIKE), NULL, 1, TEXT(T_OK), NULL, B_ENTER | B_ESC);
}

void menu_copying(struct terminal *term, void *d, struct session *ses)
{
    msg_box(term, NULL, TEXT(T_COPYING), AL_CENTER, TEXT(T_COPYING_VER), NULL, 1, TEXT(T_OK), NULL, B_ENTER | B_ESC);
}

void menu_contest_new(struct terminal *term, void *d, struct session *ses)
{
    if (ctest){
        msg_box(term, NULL, TEXT(T_WARNING), AL_CENTER,TEXT(T_CTEST_ACTIVE), NULL,1, TEXT(T_CANCEL), NULL, B_ENTER |B_ESC); 
        return;
    }
    contest_options1_from_menu(term,d,ses);
}

void menu_contest_new_wizz(struct terminal *term, void *d, struct session *ses)
{
    if (ctest){
        msg_box(term, NULL, TEXT(T_WARNING),AL_CENTER, TEXT(T_CTEST_ACTIVE), NULL,1, TEXT(T_CANCEL), NULL, B_ENTER |B_ESC); 
        return;
    }
    menu_wizz(term,d,ses);
}


void menu_contest1_options_from_ctest(struct terminal *term, void *d, struct session *ses)
{
    contest_options1_from_ctest(term,d,ses);
}




int select_dir_func(const struct dirent *de){
    struct stat st;
    gchar *f;
    
    
    if (regcmp((char *)de->d_name, "^[0-9]{8}(\\.[0-9]+)?$")) return 0;

    f = g_strconcat(getenv("HOME"), "/tucnak/", de->d_name, NULL); 
    if (stat(f, &st)) {
        g_free(f);
        return 0;
    }
    g_free(f);
    if (!S_ISDIR(st.st_mode)) return 0;

    return 1;
}

int compare_dir_func(const struct dirent **pde1, const struct dirent **pde2){
    char *c1, *c2;
    int r, n1, n2;

    r = strncmp( (*pde1)->d_name, (*pde2)->d_name, 8);
    if (r!=0) return -r;
        
    c1 = index( (*pde1)->d_name , '.');
    c2 = index( (*pde2)->d_name , '.');

    n1 = c1 ? atoi(c1+1) : 0;
    n2 = c2 ? atoi(c2+1) : 0;
    
    return n2 - n1;
}

struct menu_item no_contests[] = {
    {CTEXT(T_NO_CTESTS), "", M_BAR, NULL, NULL, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};

void menu_load_contest(struct terminal *term, char *dir, struct session *ses){
    /*dbg("menu_load_contest('%s')\n", dir);*/
  
    load_contest_edi(dir, 0);
    redraw_later(term);
} 

static struct dirent **namelist=NULL;
static int namelistlen = 0;
    
void free_namelist(void){
    int i;

    if (!namelistlen) return;
    
    for (i=0; i<namelistlen; i++)     
        free(namelist[i]);
    free(namelist);
    namelist = NULL;
    namelistlen = 0;
}

void menu_contest_open(struct terminal *term, void *ddd, struct session *ses)
{
    int i, n, max=0;
    gchar *dir;
    struct menu_item *mi = NULL;
    char s[1025], *d;
    
    /*dbg("open contest \n");*/

    free_namelist();
        
    dir = g_strdup_printf("%s/tucnak", getenv("HOME"));

#ifdef HAVE_SCANDIR_PROTOTYPE_1    
     /* glibc 2.2.4 */   
    n = scandir(dir, &namelist, select_dir_func,(int (*)(const void*, const void*))compare_dir_func);
#elif HAVE_SCANDIR_PROTOTYPE_2
    /*  cygwin */
    n = scandir(dir, &namelist, select_dir_func,(int (*)(const struct dirent **, const struct dirent **)) compare_dir_func);
#else    
    /*  libc5  */
    n = scandir(dir, &namelist, select_dir_func,(int (*)(__const struct dirent * __const *, __const struct dirent * __const *)) compare_dir_func);
#endif    
 
    if (n<=0){
        do_menu_(term, no_contests, ses);
        return;
    }
    
    if (!(mi = new_menu(3))) return;
    for (i=0; i<n; i++){
        FILE *f;
        gchar *c;
        int locked=0;
/*      
 *      dbg("'%s'\n", namelist[i]->d_name);*/
        
        c=g_strdup_printf("%s/%s/desc", dir, namelist[i]->d_name);
                
        memset(s, 0, sizeof(s));
        d = NULL;
        f = fopen(c, "rt");
        if (f){
            locked=0;
            if (lockf(fileno(f), F_TEST, 0)) locked=1;
            if (fgets(s,1024,f)!=NULL){
                if (strlen(s)>0 && s[strlen(s)-1]=='\n') s[strlen(s)-1]='\0';
                if (strlen(s)>0 && s[strlen(s)-1]=='\r') s[strlen(s)-1]='\0';
                d = index(s,' ');
                if (d) d++;
            }
            fclose(f);
        }
        g_free(c);

        
        if (!d) d=s;
        if (strlen(d)>50) d[50]='\0';
        
        c = g_strdup_printf("%-12s%s", namelist[i]->d_name, d);
        
        if (strlen(c)>max) max=strlen(c);
        add_to_menu(&mi, 
                stracpy(c), 
                locked? "!":"", "", 
                MENU_FUNC menu_load_contest, namelist[i]->d_name, 0);    
        g_free(c);
    }
    g_free(dir);
    
    set_window_ptr(ses->win, (term->x-6-max)/2,(term->y-2-i)/2);
    do_menu_(term, mi, ses);
}


void really_close_ctest(struct session *ses)
{
    rel_write_all("CL\n");
    save_all_bands_txt(0);    
    free_ctest();
}

void menu_save_all(struct terminal *term, void *d, struct session *ses){
    save_all_bands_txt(0);    
}

void menu_export_edi(struct terminal *term, void *d, struct session *ses){
    export_all_bands_edi();    
}

void menu_export_adif(struct terminal *term, void *d, struct session *ses){
    export_all_bands_adif();    
}

void menu_export_report(struct terminal *term, void *d, struct session *ses){
    export_all_bands_report();    
}

void menu_export_html(struct terminal *term, void *d, struct session *ses){
    export_all_bands_html();    
}

void menu_export_stats(struct terminal *term, void *d, struct session *ses) {
	export_stats_fifo();
}

void menu_export_titlpage(struct terminal *term, void *d, struct session *ses) {
	export_all_bands_titlpage();
}



void menu_import_edi(struct terminal *term, void *d, struct session *ses){
    char *pwd;
    
    pwd=getcwd(NULL, 0);
    input_field(ses->term, NULL, TEXT(T_IMPORT_EDI), TEXT(T_FILENAME), 
               TEXT(T_OK), TEXT(T_CANCEL), ses, 
               NULL, 50, pwd, 0, 0, NULL,
               (void (*)(void *, char *)) import_edi, NULL);
    if (pwd) free(pwd);
}

void menu_import_adif(struct terminal *term, void *d, struct session *ses){
    char *pwd;
    
    pwd=getcwd(NULL, 0);
    input_field(ses->term, NULL, TEXT(T_IMPORT_ADIF), TEXT(T_FILENAME), 
               TEXT(T_OK), TEXT(T_CANCEL), ses, 
               NULL, 50, pwd, 0, 0, NULL,
               (void (*)(void *, char *)) import_adifx, NULL);
    if (pwd) free(pwd);
}

void menu_import_swap(struct terminal *term, void *d, struct session *ses){
    char *pwd;
    
    pwd=getcwd(NULL, 0);
    input_field(ses->term, NULL, TEXT(T_IMPORT_SWAP), TEXT(T_FILENAME), 
               TEXT(T_OK), TEXT(T_CANCEL), ses, 
               NULL, 50, pwd, 0, 0, NULL,
               (void (*)(void *, char *)) import_swap, NULL);
    if (pwd) free(pwd);
}





void menu_contest_close(struct terminal *term, void *d, struct session *ses)
{
    if (!ctest){
        msg_box(term, NULL, TEXT(T_WARNING),AL_CENTER,TEXT(T_NO_CCLOSE),NULL,1, TEXT(T_CANCEL), NULL, B_ENTER |B_ESC); 
        return;
    }
    msg_box(ses->term, NULL, TEXT(T_REALLY), AL_CENTER, 
            TEXT(T_CLOSE_CTEST),ses, 2, 
            TEXT(T_YES), (void (*)(void *))really_close_ctest, B_ENTER, 
            TEXT(T_NO), NULL, B_ESC);
}






   /****************************************************************************/
    
void really_exit_prog(struct session *ses)
{
    register_bottom_half((void (*)(void *))destroy_terminal, ses->term);
}

void dont_exit_prog(struct session *ses)
{
    ses->exit_query = 0;
}

void query_exit(struct session *ses)
{
    ses->exit_query = 1;
   msg_box(ses->term, NULL, TEXT(T_EXIT_TUCNAK), AL_CENTER, 
           TEXT(T_DO_YOU_REALLY_WANT_TO_EXIT_TUCNAK), ses, 2, TEXT(T_YES), (void (*)(void *))really_exit_prog, B_ENTER, TEXT(T_NO), dont_exit_prog, B_ESC);
}

void exit_prog(struct terminal *term, void *d, struct session *ses)
{
    if (!ses) {
        register_bottom_half((void (*)(void *))destroy_terminal, term);
        return;
    }

    if (1){
        query_exit(ses);
        return;
    }
    really_exit_prog(ses);
}

struct refresh {
    struct terminal *term;
    struct window *win;
    struct session *ses;
    void (*fn)(struct terminal *term, void *d, struct session *ses);
    void *data;
    int timer;
};

void refresh(struct refresh *r)
{
    struct refresh rr;
    r->timer = -1;
    memcpy(&rr, r, sizeof(struct refresh));
    delete_window(r->win);
    rr.fn(rr.term, rr.data, rr.ses);
}

void end_refresh(struct refresh *r)
{
    if (r->timer != -1) kill_timer(r->timer);
    mem_free(r);
}

void refresh_abort(struct dialog_data *dlg)
{
    end_refresh(dlg->dlg->udata2);
}


#ifdef LEAK_DEBUG

void memory_cld(struct terminal *term, void *d)
{
    last_mem_amount = mem_amount;
}

/* FIXME! overflow */
#define MSG_BUF 4000
#define MSG_W   100

void memory_info(struct terminal *term, void *d, struct session *ses)
{
    char message[MSG_BUF];
    char *p;
    struct refresh *r;
    FILE *f;
    char s[1026], *c1, *c2, *c3;
    int cnt=1;
    
    if (!(r = mem_alloc(sizeof(struct refresh)))) return;
    r->term = term;
    r->win = NULL;
    r->ses = ses;
    r->fn = memory_info;
    r->data = d;
    r->timer = -1;
    p = message;
    p += sprintf(p, TEXT(T_GUI_LD), mem_amount);
    if (last_mem_amount != -1) p += sprintf(p, ", %s %ld, %s %ld", TEXT(T_LAST), last_mem_amount, TEXT(T_DIFFERENCE), mem_amount - last_mem_amount);
    p += sprintf(p, TEXT(T_SBRK), (vint)sbrk(0)-starting_sbrk);
    p += sprintf(p, ".\n");

    p += sprintf(p, TEXT(T_CWI), get_cw_size(cw), get_wc_size(cw), cw->latest); 
    p += sprintf(p, TEXT(T_DWI), get_pd_size(dw), get_dw_size(dw), get_wd_size(dw)); 
    p += sprintf(p, TEXT(T_NAMES), get_namedb_size(namedb)); 

    f = fopen("/proc/self/status", "rt");
    if (f){
        p+=sprintf(p,TEXT(T_MEMS));
        while (fgets(s, 1024, f)!=NULL){
/*            dbg("%s", s);*/
            if (regmatch(s, "Vm(.*):. *([0-9 a-zA-Z]+)", &c1, &c2, &c3, NULL)==0){
/*                dbg("match '%s' '%s' \n", c2, c3);*/
                p+=sprintf(p, " %s=%s ", c2, c3);
                cnt++;
                if (cnt==3){
                    cnt=0;
                    p+=sprintf(p, "\n");
                }
                mem_free(c1);mem_free(c2);mem_free(c3);
            }
            
        }
        
        fclose(f);
    }

    if (!(p = stracpy(message))) {
        mem_free(r);
        return;
    }
    msg_box(term, getml(p, NULL), TEXT(T_MEMORY_INFO), AL_CENTER, p, r, 1, TEXT(T_OK), NULL, B_ENTER | B_ESC);
    r->win = term->windows.next;
    ((struct dialog_data *)r->win->data)->dlg->abort = refresh_abort;
    r->timer = install_timer_(RESOURCE_INFO_REFRESH, (void (*)(void *))refresh, r);
}

#endif

void net_info(struct terminal *term, void *d, struct session *ses)
{
    char message[MSG_BUF];
    char *p;
    struct refresh *r;
    int i;
    
    if (!(r = mem_alloc(sizeof(struct refresh)))) return;
    r->term = term;
    r->win = NULL;
    r->ses = ses;
    r->fn = net_info;
    r->data = d;
    r->timer = -1;
    p = message;
    p += sprintf(p,TEXT(T_NMY), inet_ntoa(net->my.sin_addr), ntohs(net->my.sin_port)); 
    p += sprintf(p,TEXT(T_NGLOBAL), inet_ntoa(net->global.sin_addr), ntohs(net->global.sin_port)); 
    if (net->master){
        p += sprintf(p,TEXT(T_NMASTER), ns_desc[net->master->state], 
                                                net->master->operator?net->master->operator:"",
                                                inet_ntoa(net->master->sin.sin_addr), 
                                                ntohs(net->master->sin.sin_port),
                                                net->master->is_same_ctest?"OK":"--",
                                                get_timer_str(net->master)); 
    }
    if (net->peers->len>0){
        p += sprintf(p, TEXT(T_NSLAVES)); 
    }
    for (i=0;i<net->peers->len; i++){
        struct conn *conn;
        
        conn = g_ptr_array_index(net->peers, i);
        /*p += sprintf(p, "%s %s:%d\n", ns_desc[conn->state], inet_ntoa(conn->sin.sin_addr), ntohs(conn->sin.sin_port)); */
        p += sprintf(p, "%s %s %s %s %s\n", 
                ns_desc[conn->state], 
                conn->operator?conn->operator:"",
                conn->remote_id, 
                conn->is_same_ctest?"OK":"--",
                get_timer_str(conn)); 
    }

    if (!(p = stracpy(message))) {
        mem_free(r);
        return;
    }
    msg_box(term, getml(p, NULL), TEXT(T_NINFO), AL_LEFT, p, r, 1, TEXT(T_OK), NULL, B_ENTER | B_ESC);
    r->win = term->windows.next;
    ((struct dialog_data *)r->win->data)->dlg->abort = refresh_abort;
    r->timer = install_timer_(RESOURCE_INFO_REFRESH, (void (*)(void *))refresh, r);
}



struct menu_item no_bands_qrv_menu[] = {
    {CTEXT(T_NO_BANDS_QRV), "", M_BAR, NULL, NULL, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};

struct menu_item no_bands_configured_menu[] = {
    {CTEXT(T_NO_BANDS_CONF), "", M_BAR, NULL, NULL, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0},
};

void menu_activate_band(struct terminal *term, char *pband, struct session *ses){
    struct band *b;
    
    /*dbg("menu_activate_band(term=%p, pband='%s', ses=%p)\n",term,pband,ses);*/
    
    b=find_band_by_pband(pband);
    activate_band(ses,b);
    
}


    
void menu_setup_band(struct terminal *term, gchar *pband, struct session *ses){
    band_settings(term, pband, NULL);
}

void bands_menu(struct terminal *term, void *ddd, struct session *ses)
{
    struct menu_item *mi = NULL;
    int i;
    
    if (ddd==menu_activate_band){
        if (!ctest || !ctest->bands->len) {
            do_menu_(term, no_bands_qrv_menu, ses);
            return;
        }

        for (i=0;i<ctest->bands->len;i++){
            struct band *b;
            static char *bandstrs[]={"A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","X","Y","Z"};

            b = g_ptr_array_index(ctest->bands,i);

            if (!mi) if (!(mi = new_menu(3))) return;
            add_to_menu(&mi, 
                    stracpy(b->pband), 
                    bandstrs[upcase(b->bandchar)-'A'],
                    bandstrs[upcase(b->bandchar)-'A'],
                    MENU_FUNC ddd, b->pband, 0);
        }


        
    }else{
        if (!cfg || !cfg->bands->len) {
            do_menu_(term, no_bands_configured_menu, ses);
            return;
        }
        
        for (i=0;i<cfg->bands->len;i++){
            struct config_band *b;
            static char *bandstrs[]={"A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","X","Y","Z"};

            b = g_ptr_array_index(cfg->bands,i);
            if (!b->qrv) continue;

            if (!mi) if (!(mi = new_menu(3))) return;
            add_to_menu(&mi, 
                    stracpy(b->pband), 
                    bandstrs[upcase(b->bandchar)-'A'],
                    bandstrs[upcase(b->bandchar)-'A'],
                    MENU_FUNC ddd, b->pband, 0);
        }  
        
        if (mi) add_to_menu(&mi, stracpy(""), "", M_BAR, NULL, NULL, 0);
        
        for (i=0;i<cfg->bands->len;i++){
            struct config_band *b;
            static char *bandstrs[]={"A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","X","Y","Z"};

            b = g_ptr_array_index(cfg->bands,i);
            if (b->qrv) continue;

            if (!mi) if (!(mi = new_menu(3))) return;
            add_to_menu(&mi, 
                    stracpy(b->pband), 
                    bandstrs[upcase(b->bandchar)-'A'],
                    bandstrs[upcase(b->bandchar)-'A'],
                    MENU_FUNC ddd, b->pband, 0);
        }  
    }
    do_menu_(term, mi, ses);
}

struct menu_item not_available[] = {
    {CTEXT(T_NOT_AVAILABLE), "", M_BAR, NULL, NULL, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};



void menu_activate_subwin(struct terminal *term, void *ddd, struct session *ses){
    /*dbg("menu_activate_subwin %d\n", (int)ddd);*/

    if (sw_set_ontop(ses, (vint)ddd)){
        if (ses->focused){
            sw_set_focus(ses);
            il_unset_focus(aband->il);
        }else{
            sw_unset_focus(ses);
            il_set_focus(aband->il);
        }
        redraw_later(term);
    }
}


void subwins_menu(struct terminal *term, void *ddd, struct session *ses)
{
    struct menu_item *mi = NULL;
    int i;
    char s[16];
    
    if (!ctest) {
        do_menu_(term, not_available, ses);
        return;
    }

    if (!(mi = new_menu(3))) return;
    add_to_menu(&mi, stracpy(TEXT(T_ADD_SUBWIN)), "", TEXT(T_HK_ADD_SUBWIN), MENU_FUNC menu_add_subwin, (void *)0, 0);
    add_to_menu(&mi, stracpy(TEXT(T_CLOSE_SUBWIN)), "", TEXT(T_HK_CLOSE_SUBWIN), MENU_FUNC menu_close_subwin, (void *)0, 0);
    add_to_menu(&mi, stracpy(""), "", M_BAR, NULL, NULL, 0);

    for (i=0;i<ses->subwins->len;i++){
        struct subwin *sw;

        sw = g_ptr_array_index(ses->subwins,i);

        sprintf(s,"%d", i);
        add_to_menu(&mi, 
                stracpy(sw->title), 
                "",
                "",
                MENU_FUNC menu_activate_subwin, (void *)(vint)i, 0);
    }
    do_menu_(term, mi, ses);
}


/****************** NETWORK ********************************************/

void display_codepage(struct terminal *term, void *pcp, struct session *ses)
{
    int cp = (vint)pcp;
    struct term_spec *t = new_term_spec(term->term);
    if (t) t->charset = cp;
    cls_redraw_all_terminals();
}

void charset_list(struct terminal *term, void *xxx, struct session *ses)
{
    int i, sel;
    char *n;
    struct menu_item *mi;
    if (!(mi = new_menu(1))) return;
    for (i = 0; (n = get_cp_name(i)); i++) {
        if (is_cp_special(i)) continue;
        add_to_menu(&mi, get_cp_name(i), "", "", MENU_FUNC display_codepage, (void *)(vint)i, 0);
    }
    sel = ses->term->spec->charset;
    if (sel < 0) sel = 0;
    do_menu_selected(term, mi, ses, sel);
}

void set_val(struct terminal *term, void *ip, int *d)
{
    *d = (vint)ip;
}

void charset_sel_list(struct terminal *term, struct session *ses, int *ptr)
{
    int i, sel;
    char *n;
    struct menu_item *mi;
    if (!(mi = new_menu(1))) return;
    for (i = 0; (n = get_cp_name(i)); i++) {
        add_to_menu(&mi, get_cp_name(i), "", "", MENU_FUNC set_val, (void *)(vint)i, 0);
    }
    sel = *ptr;
    if (sel < 0) sel = 0;
    do_menu_selected(term, mi, ptr, sel);
}

void terminal_options_ok(void *p)
{
    cls_redraw_all_terminals();
}

char *td_labels[] = { CTEXT(T_NO_FRAMES), CTEXT(T_VT_100_FRAMES), CTEXT(T_LINUX_OR_OS2_FRAMES), CTEXT(T_KOI8R_FRAMES), CTEXT(T_USE_11M), CTEXT(T_RESTRICT_FRAMES_IN_CP850_852), CTEXT(T_BLOCK_CURSOR), CTEXT(T_COLOR), CTEXT(T_UTF_8_IO), NULL };

void terminal_options(struct terminal *term, void *xxx, struct session *ses)
{
    struct dialog *d;
    struct term_spec *ts = new_term_spec(term->term);
    if (!ts) return;
    if (!(d = mem_alloc(sizeof(struct dialog) + 12 * sizeof(struct dialog_item)))) return;
    memset(d, 0, sizeof(struct dialog) + 12 * sizeof(struct dialog_item));
    d->title = TEXT(T_TERMINAL_OPTIONS);
    d->fn = checkbox_list_fn;
    d->udata = td_labels;
    d->refresh = (void (*)(void *))terminal_options_ok;
    d->items[0].type = D_CHECKBOX;
    d->items[0].gid = 1;
    d->items[0].gnum = TERM_DUMB;
    d->items[0].dlen = sizeof(int);
    d->items[0].data = (void *)&ts->mode;
    d->items[1].type = D_CHECKBOX;
    d->items[1].gid = 1;
    d->items[1].gnum = TERM_VT100;
    d->items[1].dlen = sizeof(int);
    d->items[1].data = (void *)&ts->mode;
    d->items[2].type = D_CHECKBOX;
    d->items[2].gid = 1;
    d->items[2].gnum = TERM_LINUX;
    d->items[2].dlen = sizeof(int);
    d->items[2].data = (void *)&ts->mode;
    d->items[3].type = D_CHECKBOX;
    d->items[3].gid = 1;
    d->items[3].gnum = TERM_KOI8;
    d->items[3].dlen = sizeof(int);
    d->items[3].data = (void *)&ts->mode;
    d->items[4].type = D_CHECKBOX;
    d->items[4].gid = 0;
    d->items[4].dlen = sizeof(int);
    d->items[4].data = (void *)&ts->m11_hack;
    d->items[5].type = D_CHECKBOX;
    d->items[5].gid = 0;
    d->items[5].dlen = sizeof(int);
    d->items[5].data = (void *)&ts->restrict_852;
    d->items[6].type = D_CHECKBOX;
    d->items[6].gid = 0;
    d->items[6].dlen = sizeof(int);
    d->items[6].data = (void *)&ts->block_cursor;
    d->items[7].type = D_CHECKBOX;
    d->items[7].gid = 0;
    d->items[7].dlen = sizeof(int);
    d->items[7].data = (void *)&ts->col;
    d->items[8].type = D_CHECKBOX;
    d->items[8].gid = 0;
    d->items[8].dlen = sizeof(int);
    d->items[8].data = (void *)&ts->utf_8_io;
    d->items[9].type = D_BUTTON;
    d->items[9].gid = B_ENTER;
    d->items[9].fn = ok_dialog;
    d->items[9].text = TEXT(T_OK);
    d->items[10].type = D_BUTTON;
    d->items[10].gid = B_ESC;
    d->items[10].fn = cancel_dialog;
    d->items[10].text = TEXT(T_CANCEL);
    d->items[11].type = D_END;
    do_dialog_(term, d, getml(d, NULL));
}


void menu_shell(struct terminal *term, void *xxx, void *yyy)
{
    char *sh;
    if (!(sh = GETSHELL)) sh = DEFAULT_SHELL;
    exec_on_terminal(term, sh, "", 1);
}

void menu_set_language(struct terminal *term, void *pcp, struct session *ses)
{
    set_language((vint)pcp);
    cls_redraw_all_terminals();
}

void menu_language_list(struct terminal *term, void *xxx, struct session *ses)
{
    int i, sel;
    char *n;
    struct menu_item *mi;
    if (!(mi = new_menu(1))) return;
    for (i = 0; i < n_languages(); i++) {
        n = language_name(i);
        add_to_menu(&mi, n, "", "", MENU_FUNC menu_set_language, (void *)(vint)i, 0);
    }
    sel = current_language;
    do_menu_selected(term, mi, ses, sel);
}

void menu_open_gfx(void){
#ifdef HAVE_SDL
    if (!gfx){
        gfx = init_gfx();
    }
#endif
}

void menu_close_gfx(void){
#ifdef HAVE_SDL
    free_gfx();
    gfx=NULL;
#endif
}

struct menu_item file_menu11[] = {
    {CTEXT(T_OPEN_GFX), "", CTEXT(T_HK_OPEN_GFX), MENU_FUNC menu_open_gfx, NULL, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0}
};

struct menu_item file_menu12[] = {
    {CTEXT(T_CLOSE_GFX), "", CTEXT(T_HK_CLOSE_SUBWIN), MENU_FUNC menu_close_gfx, NULL, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0}
};

struct menu_item file_menu22[] = {
/*  "", "", M_BAR, NULL, NULL, 0, 0,*/
/*#ifdef LEAK_DEBUG*/
    {CTEXT(T_MEMORY_INFO), "", CTEXT(T_HK_MEMORY_INFO), MENU_FUNC memory_info, (void *)0, 0, 0},
    {CTEXT(T_NINFO), "", CTEXT(T_HK_NINFO), MENU_FUNC net_info, (void *)0, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0}
/*#endif*/
};

struct menu_item file_menu3[] = {
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_EXIT), "", CTEXT(T_HK_EXIT), MENU_FUNC exit_prog, (void *)0, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};

void do_file_menu(struct terminal *term, void *xxx, struct session *ses)
{
    int x,file_menu1_size;
    struct menu_item *file_menu, *e, *f;
    
#ifdef HAVE_SDL                    
	struct menu_item *file_menu1;

    file_menu1=gfx?file_menu12:file_menu11;
    file_menu1_size=sizeof(file_menu1);
#else
    file_menu1_size=0;    
#endif                    
    

    if (!(file_menu = mem_alloc(file_menu1_size +
                                sizeof(file_menu22) + 
                                sizeof(file_menu3) + 
                                3 * sizeof(struct menu_item)))) return;
    e = file_menu;
#ifdef HAVE_SDL
    memcpy(e, file_menu1, sizeof(file_menu12));
    e += sizeof(file_menu12) / sizeof(struct menu_item);
#endif    
    memcpy(e, file_menu22, sizeof(file_menu22));
    e += sizeof(file_menu22) / sizeof(struct menu_item);
    x = 1;
    if (can_open_os_shell(term->environment)) {
        e->text = TEXT(T_OS_SHELL);
        e->rtext = "";
        e->hotkey = TEXT(T_HK_OS_SHELL);
        e->func = MENU_FUNC menu_shell;
        e->data = NULL;
        e->in_m = 0;
        e->free_i = 0;
        e++;
        x = 0;
    }
    memcpy(e, file_menu3 + x, sizeof(file_menu3) - x * sizeof(struct menu_item));
    e += sizeof(file_menu3) / sizeof(struct menu_item);
    for (f = file_menu; f < e; f++) f->free_i = 1;
    do_menu_(term, file_menu, ses);
}

struct menu_item help_menu[] = {
    {CTEXT(T_ABOUT), "", CTEXT(T_HK_ABOUT), MENU_FUNC menu_about, (void *)0, 0, 0},
    {CTEXT(T_COPYING), "", CTEXT(T_HK_COPYING), MENU_FUNC menu_copying, (void *)0, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};



/************************* CONTEST MENU *********************************/

struct menu_item ctest_menu1[] = {
    {CTEXT(T_NEW),        "", CTEXT(T_HK_NEW), MENU_FUNC menu_contest_new, NULL, 0, 0},
    {CTEXT(T_NEW_WIZZ),   ">", CTEXT(T_HK_NEW_WIZZ), MENU_FUNC menu_contest_new_wizz, NULL, 1, 0},
    {CTEXT(T_OPEN),       "F3 >", CTEXT(T_HK_OPEN), MENU_FUNC menu_contest_open, NULL, 1, 0},
/*    {CTEXT(T_IMPORT_EDI), ">", CTEXT(T_HK_IMPORT_EDI), MENU_FUNC menu_contest_new, NULL, 1, 0},*/
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};
    
struct menu_item ctest_menu2[] = {
    {CTEXT(T_SAVE),           "F2", CTEXT(T_HK_SAVE), MENU_FUNC menu_save_all, NULL, 0, 0},
    {CTEXT(T_EXPORT_EDI),     "",  CTEXT(T_HK_EXPORT_EDI), MENU_FUNC menu_export_edi, NULL, 0, 0},
    {CTEXT(T_EXPORT_REPORT),  "",  CTEXT(T_HK_EXPORT_REPORT), MENU_FUNC menu_export_report, NULL, 0, 0},
    {CTEXT(T_EXPORT_ADIF),    "",  CTEXT(T_HK_EXPORT_ADIF), MENU_FUNC menu_export_adif, NULL, 0, 0},
    {CTEXT(T_EXPORT_HTML),    "",  CTEXT(T_HK_EXPORT_HTML), MENU_FUNC menu_export_html, NULL, 0, 0},
    {CTEXT(T_EXPORT_STATS),   "",  CTEXT(T_HK_EXPORT_STATS), MENU_FUNC menu_export_stats, NULL, 0 , 0},
    {CTEXT(T_EXPORT_TITLPAGE),"",  CTEXT(T_HK_EXPORT_TITLPAGE), MENU_FUNC menu_export_titlpage, NULL, 0 , 0},
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_IMPORT_EDI),     "",  CTEXT(T_HK_IMPORT_EDI),  MENU_FUNC menu_import_edi,  NULL, 0, 0},
    {CTEXT(T_IMPORT_ADIF),    "",  CTEXT(T_HK_IMPORT_ADIF), MENU_FUNC menu_import_adif, NULL, 0, 0},
    {CTEXT(T_IMPORT_SWAP),    "",  CTEXT(T_HK_IMPORT_SWAP), MENU_FUNC menu_import_swap, NULL, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_CONTEST_OP),  "",  CTEXT(T_HK_CONTEST_OP), MENU_FUNC menu_contest1_options_from_ctest, NULL, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_CLOSE_CONTEST),     "",  CTEXT(T_HK_CLOSE_CONTEST), MENU_FUNC menu_contest_close, NULL, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};

void do_contest_menu(struct terminal *term, void *xxx, struct session *ses)
{
    if (!ctest) do_menu_(term, ctest_menu1, ses);
    else        do_menu_(term, ctest_menu2, ses);
}
/************************* EDIT MENU *********************************/

struct menu_item edit_menu[] = {
    {CTEXT(T_ROTAR), "Alt+R", CTEXT(T_HK_ROTAR), MENU_FUNC menu_rotar, NULL, 0, 0},
    {CTEXT(T_ADD_ERROR), "", CTEXT(T_HK_ADD_ERROR),  MENU_FUNC menu_add_error, NULL, 0, 0},
    {CTEXT(T_CHOP), "Alt+O", CTEXT(T_HK_CHOP),  MENU_FUNC menu_chop, NULL, 0, 0},
    {CTEXT(T_SKED_QRG), "Alt+F", CTEXT(T_HK_SKED_QRG),  MENU_FUNC menu_skedqrg, NULL, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_GRAB_BAND), "Alt+G", CTEXT(T_HK_GRAB_BAND),  MENU_FUNC menu_grabband, NULL, 0, 0},
    {CTEXT(T_FORCE_RUN), "", CTEXT(T_HK_FORCE_RUN),  MENU_FUNC menu_forcerun, NULL, 0, 0},
    {CTEXT(T_CH_SPY), "", CTEXT(T_HK_CH_SPY), MENU_FUNC menu_load_from_peer, send_spypeer_request, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_FILLOP), "", CTEXT(T_HK_FILLOP), MENU_FUNC menu_fillop, NULL, 0, 0},
    {CTEXT(T_RECAQRB), "", CTEXT(T_HK_RECAQRB),  MENU_FUNC menu_recalc_qrb, NULL, 0, 0},
    {CTEXT(T_CW_CTEST), "", CTEXT(T_HK_CW_CTEST),  MENU_FUNC menu_cw_update_contest, NULL, 0, 0},
    {CTEXT(T_CW_BAND), "", CTEXT(T_HK_CW_BAND),  MENU_FUNC menu_cw_update_band, NULL, 0, 0},
    {CTEXT(T_IMPORT_EBW), "", CTEXT(T_HK_IMPORT_EBW),  MENU_FUNC menu_import_ebw, NULL, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};
    
struct menu_item no_edit_menu[] = {
    {CTEXT(T_NO_CTEST), "", M_BAR, NULL, NULL, 0, 0},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};

void do_edit_menu(struct terminal *term, void *xxx, struct session *ses)
{
    if (!ctest || !aband) do_menu_(term, no_edit_menu, ses);
    else                  do_menu_(term, edit_menu, ses);
}

/************************** SETUP MENU ************************************/

void menu_save_rc(struct terminal *term, void *xxx, void *yyy){
    gchar *filename;
    gchar *c;
    int ret;
    
    filename = g_strconcat(getenv("HOME"), "/tucnak/tucnakrc", NULL);
    ret=save_rc_file(filename);        
    if (ret) {
        c = g_strdup_printf(TEXT(T_CANT_WRITE_S), filename);
        errbox(c, ret);
        g_free(c);
    }else{
        log_addf(TEXT(T_SAVED_S), filename);
    }
    
    g_free(filename);

}

void do_peer_menu(struct terminal *term, void (* func)(struct terminal*, int no, struct session *ses) , struct session *ses){
    int i,max;
    struct menu_item *mi = NULL;
    gchar **items;

    items = g_strsplit(net->allpeers,";",0);
    max=0;
    for (i=0; items[i]!=NULL;i++){
        
        if (strcmp(items[i],net->myid)==0) continue;
        if (strlen(items[i])==0) continue;

        if (!mi) if (!(mi = new_menu(3))) return;
        if (strlen(items[i])>max) max=strlen(items[i]);
        add_to_menu(&mi,stracpy(items[i]),"", "", MENU_FUNC func, (void *)(vint)i, 0);    
    }
    g_strfreev(items);
    
    if (mi){
        set_window_ptr(gses->win, (term->x-max)/2,(term->y-2-i)/2);
        do_menu_(term, mi, ses);
    }else
        errbox(TEXT(T_NO_PEERS),0);
    return;
    
}

void do_peer_operators_menu(struct terminal *term, void (* func)(struct terminal*, int no, struct session *ses) , struct session *ses){
    int i,max;
    struct menu_item *mi = NULL;
    gchar **items,*c;

    items = g_strsplit(net->allpeers,";",0);
    max=0;
    for (i=0; items[i]!=NULL && items[i+1]!=NULL;i+=2){
        
        if (strcmp(items[i],net->myid)==0) continue;
        if (strlen(items[i])==0) continue;

        if (!mi) if (!(mi = new_menu(3))) return;
        c=g_strdup_printf("%-8s %s", items[i+1], items[i]);
        if (strlen(c)>max) max=strlen(c);
        add_to_menu(&mi,stracpy(c),"", "", MENU_FUNC func, (void *)(vint)i, 0);    
        g_free(c);
    }
    g_strfreev(items);
    
    if (mi){
        set_window_ptr(gses->win, (term->x-max)/2,(term->y-2-i)/2);
        do_menu_(term, mi, ses);
    }else
        errbox(TEXT(T_NO_PEERS),0);
    return;
}

void menu_load_from_peer(struct terminal *term, void *fn, struct session *ses){
    GString *gs;
    char *op;

    /*dbg("menu_load_from_peer\n"); */
    
    if (cmp_sin(&net->global, &net->my)==0) { /* i'm master */
        int i;
        struct conn *conn;
        
        /* LOOK ALSO net.c rel_write, dommand DO */
        
        CONDGFREE(net->allpeers);
        gs=g_string_sized_new(100);
        for (i=0;i<net->peers->len;i++){
            conn = (struct conn *)g_ptr_array_index(net->peers,i);
        
            if (!conn_prod_state(conn)) continue;
            op="---";
            if (ctest && aband && conn->operator) op=conn->operator;
            g_string_sprintfa(gs,"%s;%s;", conn->remote_id, op);
        }
        net->allpeers=gs->str;
        g_string_free(gs,FALSE);
        do_peer_operators_menu(term, fn, ses);
        return;
    }
    
    /* i'm slave */
    if (!conn_prod_state(net->master)){
        errbox(TEXT(T_NO_MASTER),0);  
        return;
    }

    net->peerfunc = fn;
    rel_write(net->master, "DO\n");
}

                                                                      
void contest_def(struct terminal *term, void *xxx, void *yyy);

struct menu_item setup_menu[] = {
    {CTEXT(T_CTEST_DEF), "", CTEXT(T_HK_CTEST_DEF), MENU_FUNC contest_def, NULL, 0, 0},
    {CTEXT(T_BAND_DEF), ">", CTEXT(T_HK_BAND_DEF), MENU_FUNC bands_menu, menu_setup_band, 1, 0},
    {CTEXT(T_RESPOP), "", CTEXT(T_HK_RESPOP), MENU_FUNC menu_responsible_op, NULL, 0, 0},
    {CTEXT(T_CW_DAEMON), "", CTEXT(T_HK_CW_DAEMON), MENU_FUNC menu_cwda, NULL, 0, 0},
    {CTEXT(T_SSBD), "", CTEXT(T_HK_SSBD), MENU_FUNC menu_ssbd, NULL, 0, 0},
    {CTEXT(T_CW_CQ), ">", CTEXT(T_HK_CW_CQ), MENU_FUNC menu_cq_cw, NULL, 1, 0},
    {CTEXT(T_SSB_CQ), ">", CTEXT(T_HK_SSB_CQ), MENU_FUNC menu_cq_ssb, NULL, 1, 0},
    {CTEXT(T_AUTOSAVE), "", CTEXT(T_HK_AUTOSAVE), MENU_FUNC menu_autosave, NULL, 0, 0},
    {CTEXT(T_NETWORK), "", CTEXT(T_HK_NETWORK), MENU_FUNC menu_network, NULL, 0, 0},
    {CTEXT(T_TERMINAL_OPTIONS), "", CTEXT(T_HK_TERMINAL_OPTIONS), MENU_FUNC terminal_options, NULL, 0, 0},
    {CTEXT(T_LANGUAGE), ">", CTEXT(T_HK_LANGUAGE), MENU_FUNC menu_language_list, NULL, 1, 0},
    {CTEXT(T_CHARACTER_SET), ">", CTEXT(T_HK_CHARACTER_SET), MENU_FUNC charset_list, (void *)1, 1, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_LOAD_CFG_NET), "", CTEXT(T_HK_LOAD_CFG_NET), MENU_FUNC menu_load_from_peer, send_config_request, 0, 0},
    {CTEXT(T_LOAD_CW_NET), "", CTEXT(T_HK_LOAD_CW_NET), MENU_FUNC menu_load_from_peer, send_cwdb_request, 0, 0},
    {"", "", M_BAR, NULL, NULL, 0, 0},
    {CTEXT(T_SAVE_CFG), "", CTEXT(T_HK_SAVE_CFG), MENU_FUNC menu_save_rc, NULL, 0, 0},
    /*{   CTEXT(T_SAVE_OPTIONS), "", CTEXT(T_HK_SAVE_OPTIONS), MENU_FUNC write_config, NULL, 0, 0},*/
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};

void do_setup_menu(struct terminal *term, void *xxx, struct session *ses)
{
    do_menu_(term, setup_menu, ses);
}

struct menu_item main_menu[] = {
    {CTEXT(T_CONTEST), "", CTEXT(T_HK_CONTEST), MENU_FUNC do_contest_menu, NULL, 1, 1},
    {CTEXT(T_FILE), "", CTEXT(T_HK_FILE), MENU_FUNC do_file_menu, NULL, 1, 1},
    {CTEXT(T_EDIT), "", CTEXT(T_HK_EDIT), MENU_FUNC do_edit_menu, NULL, 1, 1},
    {CTEXT(T_BANDS), "", CTEXT(T_HK_BANDS), MENU_FUNC bands_menu, menu_activate_band, 1, 1},
    {CTEXT(T_SUBWINS), "", CTEXT(T_HK_SUBWINS), MENU_FUNC subwins_menu, NULL, 1, 1},
    {CTEXT(T_SETUP), "", CTEXT(T_HK_SETUP), MENU_FUNC do_setup_menu, NULL, 1, 1},
    {CTEXT(T_HELP), "", CTEXT(T_HK_HELP), MENU_FUNC do_menu_, help_menu, 1, 1},
    {NULL, NULL, 0, NULL, NULL, 0, 0}
};

void activate_bfu_technology(struct session *ses, int item)
{
   struct terminal *term = ses->term;
    do_mainmenu(term, main_menu, ses, item);
}

struct history file_history = { 0, {&file_history.items, &file_history.items} };
struct history search_history = { 0, {&search_history.items, &search_history.items} };

struct history load_swap_history = { 0, {&load_swap_history.items, &load_swap_history.items }};


void free_history_lists()
{
    free_list(file_history.items);
    free_list(search_history.items);
    
    free_list(load_swap_history.items);
}


/***************************** LOAD SWAP ***********************************/

#if 0

void load_swap(struct session *ses, char *text){
    FILE *f;

    f = fopen(text, "rt");
    if (!f){
        msg_box(term, NULL, TEXT(T_ERROR), AL_CENTER, TEXT(T_CANT_READ), 
                NULL, 1, TEXT(T_OK), NULL, B_ENTER | B_ESC);
        return;
    }
    
    add_qsos_from_swap(aband, f);
    fclose(f);
    
}


void menu_load_swap(struct terminal *term, void *d, struct session *ses){
    input_field(ses->term, NULL, "Load swap", "Enter filename", 
            TEXT(T_OK), TEXT(T_CANCEL), ses, &load_swap_history, 
            MAX_INPUT_URL_LEN, "", 0, 0, NULL, 
            (void (*)(void *, char *)) load_swap, NULL);
}
#endif


