//-*-c++-*-
/**
 Authors: David Auber, Romain Bourqui, Patrick Mary
 from the LaBRI Visualization Team
 Email : auber@tulip-software.org
 Last modification : 13/07/2007 
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by  
 the Free Software Foundation; either version 2 of the License, or     
 (at your option) any later version.
*/
#include <GL/gl.h>
#include <cmath>

#include <tulip/StringProperty.h>
#include <tulip/ColorProperty.h>
#include <tulip/DoubleProperty.h>

#include <tulip/Glyph.h>
#include <tulip/Graph.h>
#include <tulip/GlGraph.h>
#include <tulip/GlTools.h>

using namespace std;
using namespace tlp;

/** \addtogroup glyph */
/*@{*/
/// A 3D glyph.
/** This glyph draws a transparent cube using the "viewBorderColor" node
 * property value to draw its edges.
 */
class CubeOutLinedTransparent : public Glyph {
public:
  CubeOutLinedTransparent(GlyphContext *gc=NULL);
  virtual ~CubeOutLinedTransparent();
  virtual void draw(node n);
  virtual Coord getAnchor(const Coord & vector) const;

private:
  GLuint LList;
  bool listOk;
  void drawCubeSimple(GLenum);
};

GLYPHPLUGIN(CubeOutLinedTransparent, "3D - Cube OutLined Transparent", "David Auber", "09/07/2002", "Textured cubeOutLined", "1.0", 9);

//===================================================================================
CubeOutLinedTransparent::CubeOutLinedTransparent(GlyphContext *gc): Glyph(gc),listOk(false) {
}

CubeOutLinedTransparent::~CubeOutLinedTransparent() {
  if (listOk) {
    if (glIsList(LList)) glDeleteLists(LList, 2);
  }
}

void CubeOutLinedTransparent::draw(node n) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (!listOk) {
    LList = glGenLists( 2 );
    glNewList( LList, GL_COMPILE );
    //    drawCube(GL_QUADS);
    glEndList();
    glNewList( LList + 1, GL_COMPILE );
    //    setMaterial(Color(0,0,0,255));
    drawCubeSimple(GL_LINE_LOOP);
    glEndList();
    GLenum error = glGetError();
    if ( error != GL_NO_ERROR)
      cerr << "Open GL Error : " << error << " in " << __PRETTY_FUNCTION__ << endl;
    listOk = true;
  }
  assert(glIsList(LList));
  tlp::setMaterial(glGraph->elementColor->getNodeValue(n));
  string texFile = glGraph->elementTexture->getNodeValue(n);
  if (texFile != "") {
    if (glGraph->activateTexture(texFile))
      setMaterial(Color(255,255,255,0));
  }
  glCallList(LList);
  ColorProperty *borderColor = glGraph->getGraph()->getProperty<ColorProperty>("viewBorderColor");
  DoubleProperty *borderWidth = 0;
  if (glGraph->getRenderingParameters().getGraph()->existProperty ("viewBorderWidth"))
    borderWidth = glGraph->getRenderingParameters().getGraph()->getProperty<DoubleProperty>("viewBorderWidth");
  glGraph->desactivateTexture();
  Color c = borderColor->getNodeValue(n);
  //  setMaterial(c);
  if (borderWidth == 0) glLineWidth(2);
  else {
    double lineWidth = borderWidth->getNodeValue (n);
    if (lineWidth < 1e-6) glLineWidth (1e-6); //no negative borders
    else glLineWidth (lineWidth);
  }
  glDisable(GL_LIGHTING);
  glColor3ub(c[0],c[1],c[2]);
  glCallList(LList + 1);  
  glEnable(GL_LIGHTING);
}


Coord CubeOutLinedTransparent::getAnchor(const Coord & vector ) const {
  float x,y,z, fmax;
  vector.get(x,y,z);
  fmax = std::max(std::max(fabsf(x), fabsf(y)), fabsf(z));
  if( fmax > 0.0f )
    return vector * (0.5f/fmax);
  else
    return vector;
}

void CubeOutLinedTransparent::drawCubeSimple(GLenum type) {
  /* front face */
  glBegin(type);
  glVertex3f(-0.5f, -0.5f, 0.5f); 
  glVertex3f(0.5f, -0.5f, 0.5f);
  glVertex3f(0.5f, 0.5f, 0.5f);
  glVertex3f(-0.5f, 0.5f, 0.5f);
  glEnd();
  /* back face */
  glBegin(type);
  glVertex3f(-0.5f, -0.5f, -0.5f); 
  glVertex3f(-0.5f, 0.5f, -0.5f);
  glVertex3f(0.5f, 0.5f, -0.5f);
  glVertex3f(0.5f, -0.5f, -0.5f);
  glEnd();
  /* right face */
  glBegin(type);
  glVertex3f(0.5f, -0.5f, -0.5f); 
  glVertex3f(0.5f, 0.5f, -0.5f);
  glVertex3f(0.5f, 0.5f, 0.5f);
  glVertex3f(0.5f, -0.5f, 0.5f);
  glEnd();
  /* left face */
  glBegin(type);
  glVertex3f(-0.5f, -0.5f, 0.5f); 
  glVertex3f(-0.5f, 0.5f, 0.5f);
  glVertex3f(-0.5f, 0.5f, -0.5f);
  glVertex3f(-0.5f, -0.5f, -0.5f);
  glEnd();
  /* top face */
  glBegin(type);
  glVertex3f(0.5f, 0.5f, 0.5f); 
  glVertex3f(0.5f, 0.5f, -0.5f);
  glVertex3f(-0.5f, 0.5f, -0.5f);
  glVertex3f(-0.5f, 0.5f, 0.5f);
  glEnd();
  /* bottom face */
  glBegin(type);
  glVertex3f(0.5f, -0.5f, -0.5f); 
  glVertex3f(0.5f, -0.5f, 0.5f);
  glVertex3f(-0.5f, -0.5f, 0.5f);
  glVertex3f(-0.5f, -0.5f, -0.5f);
  glEnd();
}
/*@}*/
