# Copyright (c) 2007 Andrew Price
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import getpass
import twitter
import user
import data
from getopt import getopt, GetoptError

class CommandException(Exception):

	"""A generic Exception class for when things go wrong wrt commands"""

	def __init__(self, value):
		self.parameter = value

	def __str__(self):
		return str(self.parameter)


class Command:

	"""Provides a convenient interface to the twyt commands"""

	def __init__(self, display_hook=None):

		self.cmd_map = {
			"tweet":        self.tweet,
			"direct":       self.direct,
			"publictl":     self.publictl,
			"friendstl":    self.friendstl,
			"usertl":       self.usertl,
			"directtl":     self.directtl,
			"directsent":   self.directsent,
			"show":         self.show,
			"replies":	self.replies,
			"help":         self.help,
			"delete":       self.delete,
			"directdel":    self.directdel,
			"savepass":     self.savepass
		}


		self.opts = []
		self.displayer = display_hook
		self.usr = user.User()
		self.twitter = twitter.Twitter()
		if self.usr.getpassword() and self.usr.getusername():
			self.twitter.setauth(
				self.usr.getusername(),
				self.usr.getpassword())


	def doargv(self, argv=[]):

		try:
			opts, args = getopt(argv[2:], 
				"u:p:s:c:t",
				["user=","pass=","status=","since=","count="])
		except GetoptError, e:
			raise CommandException("Whoops, " + str(e))

		self.opts = opts
		for opt, arg in opts:
			if opt in ("-p", "--pass"):
				self.usr.setpassword(arg)
			elif opt in ("-u", "--user"):
				self.usr.setusername(arg)

		try:
			self.docommand(argv[1], args)
		except IndexError:
			raise CommandException("No command specified. Use 'help'.")


	def docommand(self, command, args=[]):

		result = None
		try:
			result = self.cmd_map[command](args)
		except KeyError, e:
			raise CommandException("No such command: " + str(e))

		self.usr.synctimes()
		return result


	def _callwithauth(self, func, data):

		answer = ""

		while not answer:
			try:
				answer = func(*data)
				break
			except twitter.TwitterAuthException:
				user = self.usr.getusername()
				if not user:
					raise CommandException('No username specified. Use -u')
				self.twitter.setauth(user, self._passwdprompt(user))
			except twitter.TwitterException, e:
				raise CommandException(str(e))

		return answer


	def tweet(self, args):

		if len(args) < 1:
			raise CommandException("Malformed command. Check options")

		tweetmsg = " ".join(args)

		answer = self._callwithauth(
			self.twitter.status_update, [tweetmsg])

		result = data.Status()
		result.loads(answer)

		if callable(self.displayer):
			self.displayer(unicode(result))

		return result
	
	def direct(self, args):
		
		if len(args) < 1:
			raise CommandException("Malformed command. Check options")

		if len(args) < 2:
			raise CommandException("Message missing.")

		touser = args[0]
		msg = " ".join(args[1:])
		answer = self._callwithauth(
			self.twitter.direct_new, [touser, msg])

		result = data.DirectMsg()
		result.loads(answer)
		
		if not result.id:
			raise CommandException("Sending direct message failed")
		if callable(self.displayer):
			self.displayer(unicode(result))
	
	def directtl(self, args):

		since = ""
		since_id = ""
		for opt, arg in self.opts:
			if opt in ("-s", "--since"):
				# since can be an id or a date
				try:
					since_id = int(arg)
				except ValueError:
					since = arg

		page = 1
		if len(args) > 0:
			try:
				page = int(args[0])
			except ValueError:
				pass

		answer = self._callwithauth(
			self.twitter.direct_messages, [since, since_id, page])

		results = data.DirectList(answer)
		
		if callable(self.displayer):
			for result in results:
				self.displayer(unicode(result))

		return results

	def directsent(self, args):

		since = ""
		since_id = ""
		for opt, arg in self.opts:
			if opt in ("-s", "--since"):
				# since can be an id or a date
				try:
					since_id = int(arg)
				except ValueError:
					since = arg

		page = 1
		if len(args) > 0:
			try:
				page = int(args[0])
			except ValueError:
				pass

		answer = self._callwithauth(
			self.twitter.direct_sent, [since, since_id, page])

		results = data.DirectList(answer)
		
		if callable(self.displayer):
			for result in results:
				self.displayer(unicode(result))

		return results

	def directdel(self, args):

		sid = -1
		if len(args) > 0:
			try:
				sid = int(args[0])
			except ValueError:
				pass

		if sid < 0:
			raise CommandException("Invalid ID specified")

		answer = self._callwithauth(
			self.twitter.direct_destroy, [sid])
		print answer
		if answer:
			result = data.DirectMsg()
			result.loads(answer)

			if callable(self.displayer):
				self.displayer(u'Deleted: ' + unicode(result))

			return result
		else:
			if callable(self.displayer):
				self.displayer(u'No messages deleted. Are you authorised?')

		

	def publictl(self, args):

		since_id = self.usr.getlastread('publictl', 'lastid')
		for opt, arg in self.opts:
			if opt in ("-s", "--since"):
				since_id = arg
		answer = ""
		try:
			answer = self.twitter.status_public_timeline(since_id)
		except twitter.TwitterException, e:
			raise CommandException(str(e))

		results = data.StatusList(answer)

		if callable(self.displayer):
			if since_id:
				self.displayer("Since ID %s:" % since_id)
			for result in results:
				self.displayer(unicode(result))
		if len(results) > 0:
			self.usr.setlastread('publictl', 'lastid', str(results.getlastid()))
		elif callable(self.displayer):
			self.displayer("(Nothing new)")

		return results


	def friendstl(self, args):

		ident = ""
		if len(args) > 0:
			ident = args[0]

		since = self.usr.getlastread('friendstl', ident)

		for opt, arg in self.opts:
			if opt in ("-s", "--since"):
				since = arg

		answer = self._callwithauth(
			self.twitter.status_friends_timeline, [ident, since])

		results = data.StatusList(answer)

		if callable(self.displayer):
			if since:
				self.displayer("Since %s:" % since)
			for result in results:
				self.displayer(unicode(result))

		if len(results) > 0:
			self.usr.setlastread('friendstl', ident, results.gettime())
		elif callable(self.displayer):
			self.displayer("(Nothing new)")

		return results


	def usertl(self, args):

		since = ""
		count = 20
		for opt, arg in self.opts:
			if opt in ("-s", "--since"):
				since = arg
			elif opt in ("-c", "--count"):
				count = int(arg)
		
		ident = ""
		if len(args) > 0:
			ident = args[0]

		answer = self._callwithauth(
			self.twitter.status_user_timeline, [ident, count, since])
		results = data.StatusList(answer)

		if callable(self.displayer):
			for result in results:
				self.displayer(unicode(result))

		return results


	def show(self, args):

		ident = ""
		if len(args) > 0:
			ident = args[0]

		answer = ""
		try:
			answer = self.twitter.status_show(ident)
		except twitter.TwitterException, e:
			raise CommandException(str(e))

		if answer:
			result = data.Status()
			result.loads(answer)
			
			if callable(self.displayer):
				self.displayer(unicode(result))

			return result


	def replies(self, args):

		page = 1
		if len(args) > 0:
			try:
				page = int(args[0])
			except ValueError:
				pass

		answer = self._callwithauth(
			self.twitter.status_replies, [page])

		if answer:
			results = data.StatusList(answer)

			if callable(self.displayer):
				for result in results:
					 self.displayer(unicode(result))
			return results


	def delete(self, args):

		sid = -1
		if len(args) > 0:
			try:
				sid = int(args[0])
			except ValueError:
				pass

		if sid < 0:
			raise CommandException("Invalid ID specified")

		answer = self._callwithauth(
			self.twitter.status_destroy, [sid])
		
		if answer:
			result = data.Status()
			result.loads(answer)

			if callable(self.displayer):
				self.displayer(u'Deleted: ' + unicode(result))

			return result
		else:
			if callable(self.displayer):
				self.displayer(u'No messages deleted. Are you authorised?')


	def savepass(self, args):

		username = self.usr.getusername()
		if username == "":
			raise CommandException("Username not specified. Use -u")

		pwd = self._passwdprompt(username)

		self.usr.savepass(username, pwd)
		self.twitter.setauth(username, pwd)

		return True


	def _passwdprompt(self, usr):

		pwd = ""
		for i in range(3):
			pwd = getpass.getpass("Enter %s's Twitter password: " % usr)
			if pwd != "":
				break

		return pwd
		

	def help(self, args):
		if len(args) > 0:
			raise CommandException(
				"Help not yet implemented for specific commands. "
				+ "See the README file.")

		if callable(self.displayer):
			self.displayer("Available commands:")
			keys = self.cmd_map.keys()
			keys.sort()
			for cmd in keys:
				self.displayer("  twyt " + cmd)

