/* 
 * Vega Strike
 * Copyright (C) 2003 Mike Byron
 * 
 * http://vegastrike.sourceforge.net/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifndef __EVENTRESPONDER_H__
#define __EVENTRESPONDER_H__

// See cpp file for detailed descriptions of classes, functions, etc.

#include "guidefs.h"
#include <string>

// The type of a command
typedef std::string EventCommandId;

// Forward reference
class Control;


// The EventResponder class is a virtual base class that allows objects
// to intercept and respond to input and command events. This class is
// used in conjunction with the EventManager.

class EventResponder
{
public:
    // PROCESS COMMAND
    // Process a command event.
    virtual bool processCommand(const EventCommandId& command, Control* control);


    // PROCESS KEYBOARD
    // Process a key pressed down.
    virtual bool processKeyDown(const InputEvent& event);

    // Process a key released.
    virtual bool processKeyUp(const InputEvent& event);


    // PROCESS MOUSE
    // Process a mouse button pressed down.
    virtual bool processMouseDown(const InputEvent& event);
    
    // Process a mouse button released.
    virtual bool processMouseUp(const InputEvent& event);

    // Process a mouse location change.
    virtual bool processMouseMove(const InputEvent& event);

    // Process a mouse location change when at least one mouse button is down.
    virtual bool processMouseDrag(const InputEvent& event);


    // CODE-GENERATED EVENTS.
    // Send a command event into the event chain.
    virtual void sendCommand(const EventCommandId& command, Control* control);

    // Set a specified target for commands.  Commands aren't forwarded into the
    //  event chain, they are sent to this specific target.  This can be used, for
    //  instance, to tie two controls tightly together.
    // Use NULL to clear the target and forward commands into the event chain.
    virtual void setCommandTarget(EventResponder* responder);


    // BEHAVIOR
    // Handle all input events.  Don't forward anything down the event chain.
    virtual void setModal(bool modal);


    // CONSTRUCTION
public:
    EventResponder(void);
    virtual ~EventResponder(void);

    // VARIABLES
protected:
    bool m_modal;                   // true = This window "traps" all events -- events don't go to other windows.
    EventResponder* m_commandTarget;// Forward events to this particular responder, not the event chain.
};

#endif   // __EVENTRESPONDER_H__
