/** @file
 *
 * Seamless mode:
 * Linux guest.
 */

/*
 * Copyright (C) 2006-2007 innotek GmbH
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * in version 2 as it comes in the "COPYING" file of the VirtualBox OSE
 * distribution. VirtualBox OSE is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY of any kind.
 */

#ifndef __Additions_linux_seamless_h
# define __Additions_linux_seamless_h

#include <map>
#include <vector>

#include <X11/Xlib.h>
#include <X11/extensions/shape.h>

class VBoxSeamlessLinux
{
public:
    VBoxSeamlessLinux(void) : mDisplay(0), mcRects(0), mExitHostEventThread(false),
                              mX11SeamlessThread(0), mExitX11SeamlessThread(false) {}

    // Perform the actual class initialisation.
    int init(void);

    ~VBoxSeamlessLinux(void);
private:
    // We don't want a copy constructor
    VBoxSeamlessLinux(const VBoxSeamlessLinux&);

    // Enums
    enum { SEAMLESS_WAIT_TIME = 1000 /* ms */ };

    // Class internal structures
    /** Structure containing information about a guest window's position and visible area */
    struct vboxGuestWinInfo {
    public:
        vboxGuestWinInfo(int x, int y, int cRects, XRectangle *pRects)
                : x(x), y(y), cRects(cRects), pRects(pRects) {}

        /** Co-ordinates in the guest screen. */
        int x, y;
        /** Number of rectangles used to represent the visible area. */
        int cRects;
        /** Rectangles representing the visible area.  These must be allocated by XMalloc
            and will be freed automatically if non-null when the class is destroyed. */
        XRectangle *pRects;
    };

    // Private member variables
    /** Our connection to the X11 display we are running on. */
    Display *mDisplay;
    /** A list of all desktop windows we are aware of - i.e. the ones we are watching for
        changes to its child windows. */
    std::vector<Window> mDesktopWindows;
    /** Associative array of guest window information, indexed by window ID. */
    std::map<Window, vboxGuestWinInfo> mWinInfo;
    /** Total number of rectangles needed for the visible area of all guest windows.  Used for
        allocating the array of rectangles passed to the host. */
    int mcRects;
    /** The thread which waits for seamless status change events from the host. */
    RTTHREAD mHostEventThread;
    /** This is set to true if the event thread should shut down. */
    bool mExitHostEventThread;
    /** The X11 event loop runs in this thread and reads information about visible window
        changes to send to the host. */
    RTTHREAD mX11SeamlessThread;
    /** This is set to true if the X11 event thread should shut down. */
    bool mExitX11SeamlessThread;

    // Private methods

    // Manage the tree of windows
    bool addDesktopWindow(Window hWin);
    bool addWindowToTree(Window hWin, int x, int y);
    void freeWindowTree(void);
    bool updateRectsInTree(Window hWin);
    void updateHostSeamlessInfo(void);
    void removeWindowFromTree(Window hWin);

    // Start and stop the seamless thread
    void startX11SeamlessThread(void);
    void stopX11SeamlessThread(void);

    // Private static methods for use as thread functions
    static int hostEventThread(RTTHREAD, void *pvUser);
    static int X11SeamlessThread(RTTHREAD, void *pvUser);

    // Methods to handle X11 events
    void doConfigureEvent(XConfigureEvent *event);
    void doMapEvent(XMapEvent *event);
    void doUnmapEvent(XUnmapEvent *event);
    void doShapeEvent(XShapeEvent *event);
};

#endif /* __Additions_linux_seamless_h not defined */
