//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        bspbuild.C
//
// Purpose:     building BSP Tree of Qv nodes
//
// Created:     24 Jan 96  Georg Meszaros
//
// Changed:  
//
//</file>


#if defined(PMAX) || defined (HPUX9)
enum Part { goofyPart };        // cfront confused about QvCone::Part and QvCylinder::Part
enum Binding { goofyBinding };  // cfront confused about QvMaterialBinding/QvNormalBinding::Binding
#endif



#include <vrml/QvElement.h>
#include <vrml/QvNodes.h>
#include <vrml/QvExtensions.h>
#include <vrml/QvUnknownNode.h>

#include "scene3d.h"
#include "camera.h"
#include "vrmlscene.h"
#include "vecutil.h"

#include <ge3d/ge3d.h>
#include <ge3d/vectors.h>

#include <hyperg/utils/types.h>
#include <hyperg/utils/verbose.h>

#include <math.h>
#include <iostream.h>

#include "attributes.h"
#include "lightlist.h"
#include "bsptree.h"
#include "faceattr.h"
#include "polygon.h"
#include "clipping.h"

 
QvMaterial* draw_curmtl = 0;
int draw_hilit = 0;       // which material component to use (QvMaterial::hilit_*)
int draw_hilanchors = 0;  // how to highlight anchors    (- " -)
int draw_hilnonanch = 0;  // how to draw non-anchors    (- " -)

static materialsGE3D* material_ptr = nil;
static int backface_culling = 1; // turned on     

// one global lightlist, refered to by light_array entries
static LightList lightlist; 
// one global temporary light_array, needs to be copied to faceattr each faceset
static AnyLight* light_array[NUM_LIGHTS] = { 0 };
static int next_light = 0;


// built from wrldraw.C, run through Qv-Structure, create indexed
// facesets (class FaceAttr, Face) and build the BSP-Tree through
// inserting new faces into the root of the tree.


/***** buildBSPTree *****/
// wrapper around bsp_root_->buildBSP to set state flags

void VRMLScene::buildBSPTree ()
{
  static int anchorshilit [Scene3D::num_colors] =
  { // mapping of Scene anchor highlighting (Scene3D::l_*) to Material highlighting
    0, QvMaterial::hilit_bright /*brightness*/, QvMaterial::hilit_none /*BBox*/,
    QvMaterial::hilit_colshade /* color */, QvMaterial::hilit_none /*edges*/
  };
  static int nonanchhilit [Scene3D::num_colors] =
  { 0, QvMaterial::hilit_dark /*brightness*/, QvMaterial::hilit_none /*BBox*/,
    QvMaterial::hilit_greyshade /* color */, QvMaterial::hilit_none /*edges*/
  };
  // BBox hightlighting anachronism; color edge hightlighting: TODO (also in ge3d)

  if (scene_->linksActive ())
  {
    draw_hilanchors = anchorshilit [scene_->linksColor ()];
    draw_hilnonanch = nonanchhilit [scene_->linksColor ()];
  }
  else
    draw_hilanchors = draw_hilnonanch = QvMaterial::hilit_none;

  draw_curmtl = 0;  // no material active
  draw_hilit = draw_hilnonanch;  // now outside of any anchor

  // cerr << "using material " << draw_hilit << " and " << draw_hilanchors << " for anchors " << endl;


// **************** build BSP Tree *************

  ge3dPushIdentity ();  // top node need not be a separator


  // initialize the light array to nil, delete old lights
  for (int i=0; i < NUM_LIGHTS; i++)
  {
    delete light_array[i];
    light_array[i] = nil;
  }  

  // allocate memory for the BSP tree root
  bsp_root_ = new BSPTree();

  // build the BSP Tree through parsing the Qv structure

  root_->buildBSP(bsp_root_);  // ass.: root_ != 0

  ge3d_pop_matrix ();

  // cout << "###buildBSP: build BSP Tree finished\n";


} // buildBSP





/***** groups *****/

void QvGroup::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvGroup->buildBSP(bsp_root_);\n";

  int n = getNumChildren ();
  //cout << n << " children\n";

  for (int i = 0; i < n; i++)
    getChild (i)->buildBSP(bsp_root);
}


#define BUILDBSP(className)                      \
void className::buildBSP(BSPTree*) { }


BUILDBSP(QvLOD)


void QvSeparator::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvSeparator->buildBSP(bsp_root_);\n";

  int numlights = next_light;
  //cerr << "QvSeparator::draw: push ()" << endl;
  ge3d_push_matrix ();

  int n = getNumChildren ();
  for (int i = 0; i < n; i++)
    getChild (i)->buildBSP(bsp_root);

  //cerr << "QvSeparator::draw: pop ()" << endl;
  ge3d_pop_matrix ();

  // delete all after separator new created lights
  for (int j=numlights; j < NUM_LIGHTS; j++) 
  {
    light_array[j]=nil;
  }
  next_light = numlights;
}


void QvSwitch::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvSwitch->buildBSP(bsp_root_);\n";

  int which = (int) whichChild.value;

  if (which == QV_SWITCH_NONE)  // do nothing
    return;
  if (which == QV_SWITCH_ALL)  // do all children
    QvGroup::buildBSP(bsp_root);
  else if (which < getNumChildren ())  // do one child
    getChild (which)->buildBSP(bsp_root);
}


void QvTransformSeparator::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvTransformSeparator->buildBSP(bsp_root_);\n";

  ge3d_push_matrix ();

  int n = getNumChildren ();
  for (int i = 0; i < n; i++)
    getChild (i)->buildBSP(bsp_root);

  ge3d_pop_matrix ();
}




/***** coordinates *****/

// already handled in build
BUILDBSP(QvCoordinate3)
BUILDBSP(QvNormal)
BUILDBSP(QvTextureCoordinate2)



/***** properties *****/

BUILDBSP(QvFontStyle)


void QvMaterial::buildBSP(BSPTree*)
{
//cout << "*: QvMaterial->buildBSP(bsp_root_);\n";

//cerr << "QvMaterial::draw" << endl;
  // material binding (front/back) is not part of VRML (applys to both sides)

  // ge3dMaterial (mat_front_and_back, materials_ + draw_hilit);
  material_ptr = new materialsGE3D;
  *material_ptr = *(materials_ + draw_hilit);

  // sets only basecolor in wireframe/hidden line or without lighting

// const materialsGE3D* mat = materials_ + draw_hilit;
// cerr << "num base: " << mat->num_base;
// if (mat->num_base)
// cerr << ", base RGB: "<< mat->rgb_base->R << ", " << mat->rgb_base->G << ", " << mat->rgb_base->B;
// cerr << endl;
// cerr << "num ambient: " << mat->num_ambient;
// if (mat->num_ambient)
// cerr << ", ambient RGB: "<< mat->rgb_ambient->R << ", " << mat->rgb_ambient->G << ", " << mat->rgb_ambient->B;
// cerr << endl;

// ???????????????????
   draw_curmtl = this;
}


BUILDBSP(QvMaterialBinding)
BUILDBSP(QvNormalBinding)


void QvShapeHints::buildBSP(BSPTree*)
{
//cout << "*: QvShapeHints->buildBSP(bsp_root_);\n";

  if (scene_->twosidedpolys () == Scene3D::twosided_auto)
    // ge3dHint (hint_backfaceculling, backfaceculling_);
    backface_culling = backfaceculling_;    
}


void QvTexture2::buildBSP(BSPTree*)
{
//cout << "*: QvTexture2->buildBSP(bsp_root_);\n";

}


BUILDBSP(QvTexture2Transform)



/***** lights *****/


void QvDirectionalLight::buildBSP(BSPTree*)
{
//cout << "*: QvDirectionalLight->buildBSP(bsp_root_);\n";


  if (curdrawmode_ < ge3d_flat_shading)
    return;

  if (on.value && next_light < NUM_LIGHTS)
  { 
    PosDirLight* light;
    vector3D dir;
    ge3dTransformVectorMcWc(&direction_, &dir); 

    light = new PosDirLight(color_, dir, 0, 0);
    lightlist.add(light);   
    light_array[next_light++] = light;
  }
}


void QvPointLight::buildBSP(BSPTree*)
{
//cout << "*: QvPointLight>buildBSP(bsp_root_);\n";

  if (curdrawmode_ < ge3d_flat_shading)
    return;

  if (on.value && next_light < NUM_LIGHTS)
  { 
    PosDirLight* light;
    point3D pos;
    ge3dTransformMcWc(position_, &pos);

    light = new PosDirLight(color_, pos, 1, 0);    
    lightlist.add(light);  
    light_array[next_light++] = light;
   
  }
}


void QvSpotLight::buildBSP(BSPTree*)
{
//cout << "*: QvSpotLight->buildBSP(bsp_root_);\n";

  if (curdrawmode_ < ge3d_flat_shading)
    return;

  if (on.value && next_light < NUM_LIGHTS)
  { 
    SpotLight* light;
    point3D pos;
    vector3D dir;
    ge3dTransformMcWc(position_, &pos);
    ge3dTransformVectorMcWc(direction_, &dir); 
   
    light = new SpotLight(color_, pos, dir, dropOffRate.value, cutangle_);
    lightlist.add(light);
    light_array[next_light++] = light;
  }
}



/***** cameras *****/

// TODO: assumed that one camera is defined only, at start of VRML definition 

void QvOrthographicCamera::buildBSP(BSPTree*)
{
  ge3dPushIdentity ();
  ge3dRotate (rotaxis_, - rotangle_);
  ge3d_translate (-pos_->x, -pos_->y, -pos_->z);
  ge3d_get_and_pop_matrix (mat_);

  ge3dMultMatrix ((const float (*)[4]) mat_);  // set up transformation (as in draw)
}


void QvPerspectiveCamera::buildBSP(BSPTree*)
{
  ge3dPushIdentity ();
  ge3dRotate (rotaxis_, - rotangle_);
  ge3d_translate (-pos_->x, -pos_->y, -pos_->z);
  ge3d_get_and_pop_matrix (mat_);

  ge3dMultMatrix ((const float (*)[4]) mat_);  // set up transformation (as in draw)
}

/***** transformations *****/

// TODO: may calculate overall transformations right after loading

void QvTransform::buildBSP(BSPTree*)
{
//cout << "*: QvTransform->buildBSP(bsp_root_);\n";

  // cast: gorasche, 19950709 for WIN32
  ge3dMultMatrix ((const float (*)[4]) mat_);
}

void QvRotation::buildBSP(BSPTree*)
{
//cout << "*: QvRotation->buildBSP(bsp_root_);\n";

  ge3dMultMatrix ((const float (*)[4]) mat_);
}

void QvMatrixTransform::buildBSP(BSPTree*)
{
//cout << "*: QvMatrixTransform->buildBSP(bsp_root_);\n";

  ge3dMultMatrix (mat_);
}

void QvTranslation::buildBSP(BSPTree*)
{
//cout << "*: QvTranslation->buildBSP(bsp_root_);\n";

  ge3dTranslate (trans_);
}

void QvScale::buildBSP(BSPTree*)
{
//cout << "*: QvScale->buildBSP(bsp_root_);\n";

  ge3dScale (scale_);
}



/***** shapes *****/


void QvAsciiText::buildBSP(BSPTree*)  // TODO
{
//cout << "*: QvAscii->buildBSP(bsp_root_);\n";

}


void QvCube::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvCube->buildBSP(bsp_root_);\n";

  QvMFVec3f* vertex_list_wc = new QvMFVec3f;
  int index = 0;

  point3D p;
  point3D q;
  point3D min, max; // for bounding box

  emptyBoundingbox(min, max);

  float dx = omax_.x - omin_.x;
  float dy = omax_.y - omin_.y;
  float dz = omax_.z - omin_.z;

  init3D(p, omin_.x, omin_.y, omin_.z);

  vertex_list_wc->allocValues(8);
   
  for (int i=1; i <= 2; i++)
  {
    ge3d_transform_mc_wc(p.x, p.y, p.z, 
                       &vertex_list_wc->values[index * 3],
                       &vertex_list_wc->values[index * 3 + 1],
                       &vertex_list_wc->values[index * 3 + 2]);

    q = *((point3D*) ((vertex_list_wc->values) + index * 3));
    extendBoundingbox(q, q, min, max);

    index++;

    p.x += dx;
    ge3d_transform_mc_wc(p.x, p.y, p.z, 
                       &vertex_list_wc->values[index * 3],
                       &vertex_list_wc->values[index * 3 + 1],
                       &vertex_list_wc->values[index * 3 + 2]);
 
    q = *((point3D*) ((vertex_list_wc->values) + index * 3));
    extendBoundingbox(q, q, min, max);
 
    index++;

    p.y += dy;
    ge3d_transform_mc_wc(p.x, p.y, p.z, 
                       &vertex_list_wc->values[index * 3],
                       &vertex_list_wc->values[index * 3 + 1],
                       &vertex_list_wc->values[index * 3 + 2]);
 
    q = *((point3D*) ((vertex_list_wc->values) + index * 3));
    extendBoundingbox(q, q, min, max);
 
    index++;

    p.x -= dx;
    ge3d_transform_mc_wc(p.x, p.y, p.z, 
                       &vertex_list_wc->values[index * 3],
                       &vertex_list_wc->values[index * 3 + 1],
                       &vertex_list_wc->values[index * 3 + 2]);
     
    q = *((point3D*) ((vertex_list_wc->values) + index * 3));
    extendBoundingbox(q, q, min, max);
 
    index++;

    p.y -= dy;
    p.z += dz;
  }
  // vertex_list_wc ready

  
  QvMFVec3f* normal_list_wc = new QvMFVec3f;
  normal_list_wc->allocValues(0);

  // create FaceAttr
  FaceAttr* faceattr = new FaceAttr(vertex_list_wc->num, vertex_list_wc,
                                    0 /*normal_num*/, normal_list_wc);
  
  faceattr->setAttributes(material_ptr,    // one material structure
                          next_light,      // number of active lights
                          light_array,     // needs to be copied !
                          backface_culling);
   
  faceattr->setBoundBox(min, max);

  // create faces
  
  int* face_1 = new int[5]; // = {0, 3, 2, 1, -1};
  int* face_2 = new int[5]; // = {1, 2, 6, 5, -1};
  int* face_3 = new int[5]; // = {6, 7, 4, 5, -1};
  int* face_4 = new int[5]; // = {7, 3, 0, 4, -1};
  int* face_5 = new int[5]; // = {0, 1, 5, 4, -1};
  int* face_6 = new int[5]; // = {3, 7, 6, 2, -1};

  face_1[0] = 0; face_1[1] = 3; face_1[2] = 2; face_1[3] = 1; face_1[4] = -1;
  face_2[0] = 1; face_2[1] = 2; face_2[2] = 6; face_2[3] = 5; face_2[4] = -1;
  face_3[0] = 6; face_3[1] = 7; face_3[2] = 4; face_3[3] = 5; face_3[4] = -1;
  face_4[0] = 7; face_4[1] = 3; face_4[2] = 0; face_4[3] = 4; face_4[4] = -1;
  face_5[0] = 0; face_5[1] = 1; face_5[2] = 5; face_5[3] = 4; face_5[4] = -1;
  face_6[0] = 3; face_6[1] = 7; face_6[2] = 6; face_6[3] = 2; face_6[4] = -1;
  
  
  Face* face;

  face = new Face(4, face_1, 0, nil /*normal vector*/, faceattr);
  bsp_root->insert(face);
  face = new Face(4, face_2, 0, nil /*normal vector*/, faceattr);
  bsp_root->insert(face);
  face = new Face(4, face_3, 0, nil /*normal vector*/, faceattr);
  bsp_root->insert(face);
  face = new Face(4, face_4, 0, nil /*normal vector*/, faceattr);
  bsp_root->insert(face);
  face = new Face(4, face_5, 0, nil /*normal vector*/, faceattr);
  bsp_root->insert(face);
  face = new Face(4, face_6, 0, nil /*normal vector*/, faceattr);
  bsp_root->insert(face);


}


void buildCylinder(BSPTree* bsp_root, 
                   float baseRadius,
                   float topRadius,
                   float bottom,
                   float height)
{
 
  int slices = 12;
  // is stacks > 1 useful?
  int stacks = 1;


  double da, r, dr, dy, ny;
  float y;
//  boolean normal_state;  // unused
  int i, j;

  point3D min, max, current;
  emptyBoundingbox(min, max);

  da = 2.0*M_PI / slices;
  dr = (topRadius-baseRadius) / stacks;
  dy = height / stacks;

  FaceAttr* faceattr;
  FaceAttr* base = 0;
  FaceAttr* top = 0;
  int index = 0;
  int index3 = 0;


  // base disk  
  if (baseRadius)
  {
    index = 0;
    index3 = 0;

    QvMFVec3f* base_vertex_list_wc = new QvMFVec3f;
    QvMFVec3f* normal_list_wc = new QvMFVec3f;
    normal_list_wc->allocValues(0);
     
    base_vertex_list_wc->allocValues(slices + 1);
    ge3d_transform_mc_wc(0, bottom, 0,
                          &base_vertex_list_wc->values[index3],
                          &base_vertex_list_wc->values[index3 + 1],
                          &base_vertex_list_wc->values[index3 + 2]);
  
    current = *((point3D*) ((base_vertex_list_wc->values) + index3)); 
    extendBoundingbox(current, current, min, max);

    index++;
    index3+=3;

    y = bottom;
    r = baseRadius;
    
    for (i = 0; i < slices ;i++) 
    {
      float x1 = cos(- i * da);
      float z1 = sin(- i * da);
        
      ge3d_transform_mc_wc(x1 * r, y, z1 * r, 
                          &base_vertex_list_wc->values[index3],
                          &base_vertex_list_wc->values[index3 + 1],
                          &base_vertex_list_wc->values[index3 + 2]);
      
      current = *((point3D*) ((base_vertex_list_wc->values) + index3)); 
      extendBoundingbox(current, current, min, max);

      index++;
      index3+=3;        
    }

    base = new FaceAttr(base_vertex_list_wc->num, base_vertex_list_wc, 0, normal_list_wc);
  
    base->setAttributes(material_ptr,    // one material structure
                            next_light,      // number of active lights
                            light_array,     // needs to be copied !
                            backface_culling);

    base->setBoundBox(min, max);
  }

  
  // top disk
  
  if (topRadius)
  {
    index = 0;
    index3 = 0;

    QvMFVec3f* top_vertex_list_wc = new QvMFVec3f;
    QvMFVec3f* normal_list_wc = new QvMFVec3f;
    normal_list_wc->allocValues(0);

    top_vertex_list_wc->allocValues(slices + 1);
    ge3d_transform_mc_wc(0, bottom + height, 0,
                          &top_vertex_list_wc->values[index3],
                          &top_vertex_list_wc->values[index3 + 1],
                          &top_vertex_list_wc->values[index3 + 2]);
  
    current = *((point3D*) ((top_vertex_list_wc->values) + index3)); 
    extendBoundingbox(current, current, min, max);

    index++;
    index3+=3;

    y = bottom + height;
    r = topRadius;

    for (i = 0; i < slices ;i++) 
    {
      float x1 = cos(- i * da);
      float z1 = sin(- i * da);
         
      ge3d_transform_mc_wc(x1 * r, y, z1 * r, 
                          &top_vertex_list_wc->values[index3],
                          &top_vertex_list_wc->values[index3 + 1],
                          &top_vertex_list_wc->values[index3 + 2]);
      
      current = *((point3D*) ((top_vertex_list_wc->values) + index3)); 
      extendBoundingbox(current, current, min, max);

      index++;
      index3+=3;        
    }

    top = new FaceAttr(top_vertex_list_wc->num, top_vertex_list_wc, 0, normal_list_wc);
  
    top->setAttributes(material_ptr,    // one material structure
                            next_light,      // number of active lights
                            light_array,     // needs to be copied !
                            backface_culling);

    top->setBoundBox(min, max);
  }

  QvMFVec3f* vertex_list_wc = new QvMFVec3f;
  QvMFVec3f* normal_list_wc = new QvMFVec3f;
  index = 0;
  index3 = 0;
 
  vertex_list_wc->allocValues(slices * (stacks + 1));
  normal_list_wc->allocValues(slices * (stacks + 1));      

  for (i = 0; i < slices ;i++) 
  {
    float x1 = cos(- i * da);
    float z1 = sin(- i * da);
    y = bottom;
    r = baseRadius;
    ny = (baseRadius - topRadius) / (height * baseRadius);
    vector3D new_normal;
    double norm;	

    for (j = 0; j <= stacks; j++) 
    {	   
      ge3d_transform_mc_wc(x1 * r, y, z1 * r, 
                          &vertex_list_wc->values[index3],
                          &vertex_list_wc->values[index3 + 1],
                          &vertex_list_wc->values[index3 + 2]);
      
      current = *((point3D*) ((vertex_list_wc->values) + index3)); 
      extendBoundingbox(current, current, min, max);

      // needs to be normalized
      init3D(new_normal, x1, ny, z1);
      norm = norm3D(new_normal);
      if (norm) scl3D(new_normal, 1/norm);
      ge3d_transformvector_mc_wc(new_normal.x, new_normal.y, new_normal.z, 
                          &normal_list_wc->values[index3],
                          &normal_list_wc->values[index3 + 1],
                          &normal_list_wc->values[index3 + 2]);

      index++;
      index3+=3;      

      y += dy;
      r += dr;
    }
  }
  

  // vertex_list_wc finished
  // cerr << "vertexlist  built: " << vertex_list_wc->num << "index: " << index << "\n";
  

  // create FaceAttr
  faceattr = new FaceAttr(vertex_list_wc->num, vertex_list_wc,
                          normal_list_wc->num, normal_list_wc);
  
  faceattr->setAttributes(material_ptr,    // one material structure
                          next_light,      // number of active lights
                          light_array,     // needs to be copied !
                          backface_culling);

  faceattr->setBoundBox(min, max);
    
  // create indexed Faces
  
  Face* face;
  int* triangle;
   
  // create base disk
  if (baseRadius)
  {
    j=1;
    for (i = 1; i < slices; i++)
    {
      triangle = new int[4];
      triangle[0] = 0;
      triangle[3] = -1;  
      triangle[2] = j;
      triangle[1] = ++j;
      face = new Face(3, triangle, 0, nil /*normal vector*/, base);
      bsp_root->insert(face);
    } 
    triangle = new int[4]; 
    triangle[0] = 0;
    triangle[3] = -1;  
    triangle[1] = 1;
    triangle[2] = slices;

    face = new Face(3, triangle, 0, nil /*normal vector*/, base);
    bsp_root->insert(face); 
  }


  // create top disk
  if (topRadius)
  {
    j = 1;
    for (i = 1; i < slices; i++)
    {
      triangle = new int[4];
      triangle[0] = 0;
      triangle[3] = -1;  
      triangle[1] = j;
      triangle[2] = ++j;
      
      face = new Face(3, triangle, 0, nil /*normal vector*/, top);
      bsp_root->insert(face); 
    } 
    triangle = new int[4]; 
    triangle[0] = 0;
    triangle[3] = -1;  
    triangle[2] = 1;
    triangle[1] = slices;

    face = new Face(3, triangle, 0, nil /*normal vector*/, top);
    bsp_root->insert(face); 
  }

  
  // create cylinder, cone apart of disks

  int* four_sided;
  j = 0;
  for (int k = 1; k <= stacks; k++)
  {
    for (i = 1; i < slices; i++)
    {
      four_sided = new int[5];
      four_sided[4] = -1;
      four_sided[0] = j;
      four_sided[3] = j + 1;
      four_sided[1] = j += stacks + 1;
      four_sided[2] = j + 1;           

      face = new Face(4, four_sided, 4, nil, faceattr);
      bsp_root->insert(face);  
    }
    four_sided = new int[5];
    four_sided[4] = -1;
    four_sided[0] = j;
    four_sided[3] = j + 1;
    four_sided[1] = j -= (slices - 1) * (stacks + 1);
    four_sided[2] = j + 1;
    face = new Face(4, four_sided, 4, nil, faceattr);
    bsp_root->insert(face); 
  }


}



void QvCone::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvCone->buildBSP(bsp_root_);\n";
 
  float h = height.value;
  ::buildCylinder(bsp_root, bottomRadius.value, 0 /* topradius */, - h / 2.0, h);
}


void QvCylinder::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvCylinder->buildBSP(bsp_root_);\n";

  float h = height.value;
  float r = radius.value;
  ::buildCylinder(bsp_root, r, r, - h / 2.0, h);
}



void QvSphere::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvSphere->buildBSP(bsp_root_);\n";
  
  // good values:
  // slices = 12
  // stacks = 6
 
  int slices = 12;
  int stacks = 6;

  float rho, drho, theta, dtheta;
  float x, y, z;
  int i, j;

  point3D min, max, current;
  emptyBoundingbox(min, max);

  QvMFVec3f* vertex_list_wc = new QvMFVec3f;
  QvMFVec3f* normal_list_wc = new QvMFVec3f; 
  vertex_list_wc->allocValues((stacks-1)*slices+2);
  normal_list_wc->allocValues((stacks-1)*slices+2);

  int index = 0;
  int index3 = 0;

  
  // top vertex 
  ge3d_transform_mc_wc(0.0, radius.value, 0.0, 
                          &vertex_list_wc->values[index3],
                          &vertex_list_wc->values[index3 + 1],
                          &vertex_list_wc->values[index3 + 2]);
  
  current = *((point3D*) ((vertex_list_wc->values) + index3)); 
  extendBoundingbox(current, current, min, max);

  ge3d_transformvector_mc_wc(0.0, radius.value, 0.0, 
                          &normal_list_wc->values[index3],
                          &normal_list_wc->values[index3 + 1],
                          &normal_list_wc->values[index3 + 2]);

  index++;      // vertices, normals
  index3+=3;  

  // intermediate vertices
  
  drho = M_PI / stacks;
  dtheta = 2.0 * M_PI / slices;

  for (i = 1; i < stacks; i++) 
  {
    rho = i * drho; 
    for (j = 0; j <  slices; j++) 
    {
      theta = - j * dtheta;
      x = cos(theta) * sin(rho);
      y = cos(rho);
      z = sin(theta) * sin(rho);

      ge3d_transform_mc_wc(x*radius.value, y*radius.value, z*radius.value, 
                          &vertex_list_wc->values[index3],
                          &vertex_list_wc->values[index3 + 1],
                          &vertex_list_wc->values[index3 + 2]);

      current = *((point3D*) ((vertex_list_wc->values) + index3)); 
      extendBoundingbox(current, current, min, max);
   
      ge3d_transformvector_mc_wc(x*radius.value, y*radius.value, z*radius.value, 
                          &normal_list_wc->values[index3],
                          &normal_list_wc->values[index3 + 1],
                          &normal_list_wc->values[index3 + 2]);
    
      index++;
      index3+=3;
    }
  }

  //cerr << "vertexlist middle built: " << vertex_list_wc->num << "index: " << index << "\n";
  
  // bottom vertex
  ge3d_transform_mc_wc(0.0, -radius.value, 0.0, 
                          &vertex_list_wc->values[index3],
                          &vertex_list_wc->values[index3 + 1],
                          &vertex_list_wc->values[index3 + 2]);

  current = *((point3D*) ((vertex_list_wc->values) + index3)); 
  extendBoundingbox(current, current, min, max);
 
  ge3d_transformvector_mc_wc(0.0, -radius.value, 0.0, 
                          &normal_list_wc->values[index3],
                          &normal_list_wc->values[index3 + 1],
                          &normal_list_wc->values[index3 + 2]);

     
  //cerr << "vertexlist bottom built: " << vertex_list_wc->num << "index: " << index << "\n";


  // create FaceAttr
  FaceAttr* faceattr = new FaceAttr(vertex_list_wc->num, vertex_list_wc,
                                    normal_list_wc->num, normal_list_wc);
  
  faceattr->setAttributes(material_ptr,    // one material structure
                          next_light,      // number of active lights
                          light_array,     // needs to be copied !
                          backface_culling);
    
  faceattr->setBoundBox(min, max);


  //faceattr->print();


  // create indexed Faces
  
  Face* face;
  int* triangle; // triangle is the index list for top and bottom


  // draw +Z end as a triangle fan

  
  for (j = 1; j < slices; j++)
  {
      triangle = new int[4];
      triangle[0] = 0;  
      triangle[3] = -1;
      triangle[1] = j;
      triangle[2] = j + 1;
      face = new Face(3, triangle, 3, nil /*normal vector*/,faceattr);
      //cerr << "BSPBuild::Sphere - insert face mit id: " << face->id() << "*****\n";
      //face->print();
      bsp_root->insert(face);   
      //cerr << "after insert in sphere\n";
  } 
  triangle = new int[4];
  triangle[0] = 0;  
  triangle[3] = -1;
  triangle[1] = slices;
  triangle[2] = 1;
  face = new Face(3, triangle, 3, nil /*normal vector*/, faceattr);
  //cerr << "BSPBuild::Sphere - insert face mit id: " << face->id() << "*****\n";
  //face->print();
  bsp_root->insert(face);
  //cerr << "after insert in sphere\n";

  // +Z end triangle fan finished

  //cerr << "top finished\n";

  // draw intermediate stacks as quad strips 

  
  int* four_sided; // is index list for in-between polygons
  int first;
  for (i = 1; i <= stacks - 2; i++)
  {
    for (j = 1; j < slices; j++)
    { 
      four_sided = new int[5];
      first = (i - 1) * slices + j;
      four_sided[4] = -1; 
      four_sided[0] = first;
      four_sided[1] = first + slices;
      four_sided[2] = first + slices + 1;
      four_sided[3] = first + 1;
      face = new Face(4, four_sided, 4, nil /*normal vector*/, faceattr);
      //cerr << "BSPBuild::Sphere - insert face mit id: " << face->id() << "*****\n";
      //face->print();
      bsp_root->insert(face);
      //cerr << "after insert in sphere\n";    
    }
    four_sided = new int[5];
    four_sided[4] = -1;
    four_sided[0] = i * slices;
    four_sided[1] = (i + 1) * slices;
    four_sided[2] = i * slices + 1;
    four_sided[3] = (i - 1) * slices + 1;
    face = new Face(4, four_sided, 4, nil /*normal vector*/, faceattr);
    //cerr << "BSPBuild::Sphere - insert face mit id: " << face->id() << "*****\n";
    //face->print();
    bsp_root->insert(face);
    //cerr << "after insert in sphere\n";
  }

 
  // intermediate quad strips finished
  //cerr << "middle finished\n";

  // draw -Z end as a triangle fan //

  // the last of the original sphere vertices
  int last = (stacks-1)*slices+1;
  
  for (j = last - slices; j < last - 1; j++)
  {
      triangle = new int[4];
      triangle[0] = last;
      triangle[3] = -1;
      triangle[1] = j + 1;
      triangle[2] = j;
      face = new Face(3, triangle, 3, nil /*normal vector*/, faceattr);
      //cerr << "BSPBuild::Sphere - insert face mit id: " << face->id() << "*****\n";
      //face->print();
      bsp_root->insert(face);  
      //cerr << "after insert in sphere\n"; 
  }
  triangle = new int[4];
  triangle[0] = last;
  triangle[3] = -1;
  triangle[1] = last - slices;
  triangle[2] = last - 1;
  face = new Face(3, triangle, 3, nil /*normal vector*/, faceattr);
  //cerr << "BSPBuild::Sphere - insert face mit id: " << face->id() << "*****\n"; 
  //face->print();
  bsp_root->insert(face); 
  //cerr << "after insert in sphere\n";

  // -Z end triangle fan finished

  //cout << "after inserting all Faces:\n";
  //faceattr->print();

  //cerr << "QvSphere finished\n";
}



void QvIndexedFaceSet::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvIndexedFaceSet->buildBSP(bsp_root_);\n";

// TODO: appropriate material on anchor highlighting

   unsigned vertex_num = 0;
   unsigned face_num = 0;
  
    int count = 0;
   const int* old_vertindices = vertindices_;
   const int* current_old_vertindices = vertindices_;
   
   // is allocated here and used in the created faceset
   QvMFVec3f* vertexlist_wc = new QvMFVec3f;
   vertexlist_wc->allocValues(numvertices_);   

   // numvertinds_ is the exact number of indices (with -1 values), last one optional
  
   int* new_vertindices = new int[numvertinds_];
   int new_vertex_pos = -1;
   int new_index_pos = -1;
   point3D v;  
   int found;
   point3D min, max;  // bounding box
      
 
   emptyBoundingbox(min, max);
  
   while ((*current_old_vertindices != -1) && (count < numvertinds_))
   {
     while ((*current_old_vertindices != -1) && (count < numvertinds_))
     {
       // inside a face: 
       new_index_pos++;
       // check if this index has already accured
       int i = 0;
       found = 0;
       while ((i < new_index_pos) && (!found))
       { 
         if (old_vertindices[i] == *current_old_vertindices)
	 { 
           new_vertindices[new_index_pos] = new_vertindices[i]; 
           found = 1;
         }
         i++;
       } 

       if (!found)
       {
         // this index is used the first time, new vertex to vertexlist
         new_vertex_pos++;       
         v = vertexlist_[*current_old_vertindices];      
         ge3d_transform_mc_wc(v.x, v.y,v.z, 
                          &vertexlist_wc->values[new_vertex_pos * 3],
                          &vertexlist_wc->values[new_vertex_pos * 3 + 1],
                          &vertexlist_wc->values[new_vertex_pos * 3 + 2]);

         new_vertindices[new_index_pos] = new_vertex_pos;
         
         v = *((point3D*) ((vertexlist_wc->values) + new_vertex_pos * 3));
         extendBoundingbox(v, v, min, max);

       } // !found
      
       current_old_vertindices++; 
       count++;
     
     } // one face

     // index == -1
     current_old_vertindices++;
     new_vertindices[++new_index_pos] = -1;
 
     count++;
     face_num++;
   } // all indices through

   vertex_num = new_vertex_pos + 1;   
   vertexlist_wc->allocValues(vertex_num); 


   //cout << "vertex_num:" << vertex_num << "\n";
   //for (int i=0; i < numvertinds_; i++) cout << new_vertindices[i] << " ";
   //cout << "\n";
     
 
   //cout << "##### before new faceattr #####" << this->vertexlist_wc->values  <<"\n";
   //for (int i=0; i < vertexlist_wc->num; i++) 
   //  Clipping::print3D(*( (point3D*) (this->vertexlist_wc->values+(3*i))));
  
  
   QvMFVec3f* normallist_wc = new QvMFVec3f;
   normallist_wc->allocValues(0);

   FaceAttr* faceattr = new FaceAttr(vertex_num, vertexlist_wc,
                                     0 /*normal_num*/, normallist_wc);
  
   
   faceattr->setAttributes(material_ptr,    // one material structure
                           next_light,      // number of active lights
                           light_array,     // needs to be copied !
                           backface_culling);

   faceattr->setBoundBox(min, max);

/*
   faceattr->print();
   cout << "*:vertex_list_->values  after new faceattr" << "\n";
   faceattr->printVertices();
*/

   if (faceattr->id() == -1) cout << "+++++++++++++++++++++++++++++++++++++++++++++++\n";


   if ( faceattr->id() == -1 )
   {
     cout << "addr of light_: " << faceattr->light() << "\n";
     cout << int (*(faceattr->light())) << "\n";
     faceattr->print(); 
   }

   
   // create all faces
   count = 0;
   int index_count = 0;
   int* int_temp;
   int* vert_marker = new_vertindices;

   int tmp=0; 
   Face* face;

   while (count < numvertinds_)
   { 
     index_count = 0;
     int_temp = vert_marker;
     do { 
       //cout << *int_temp << ",";
       int_temp++; 
       count++; 
       index_count++;
     } while (( *int_temp != -1) && (count < numvertinds_));
     count++;     
    
    tmp++;

    int* index_list = new int[index_count + 1];
    for (int i=0; i < index_count + 1; i++)
        index_list[i] = vert_marker[i];

    face = new Face(index_count , index_list, 0, nil /*normal vector*/, faceattr);

    bsp_root->insert(face);

    //deleting face is done in BSPTree::insert, if necessary (splitting)
      
     // set new startpointers for next face
     vert_marker += index_count + 1;

   } //all faces done

   //cout << "final count:" << count << "\n";   

   // turn off texturing
   ge3dDoTexturing (0);

   delete [] new_vertindices;

//cout << "*: QvIndexedFaceSet->buildBSP(bsp_root_) END;\n";

} // QvIndexedFaceSet



void QvIndexedLineSet::buildBSP(BSPTree*)
{
cout << "*: QvIndexedLineSet->buildBSP(bsp_root_);\n";

//cerr << "QvIndexedLineSet::draw" << endl;

  // assert: int and long of QvMFLong are of same size - see <vrml/QvMFLong>
  // assert: arrays of point3D's are compatible to arrays of float triples

  // ge3dLineSet (vertexlist_, numvertinds_, vertindices_);
  // currently materials ignored (TODO) - (normals and textures not relevant)

} // QvIndexedLineSet


void QvPointSet::buildBSP(BSPTree*)
{
//cout << "*: QvPointSet->buildBSP(bsp_root_);\n";

//  ge3dPointSet (points_, num_);  // TODO: materials (normals, textures not relevant)

} // QvPointSet



/***** WWW *****/
BUILDBSP(QvWWWAnchor)
BUILDBSP(QvWWWInline)


//not implemented yet
/*

// TODO anchors,  inline, etc.


void QvWWWAnchor::draw ()
{
  int oldhilit = draw_hilit;  // care for nested anchors

  if (draw_hilit != draw_hilanchors)  // activate anchor highlighting
  {
    draw_hilit = draw_hilanchors;
    //cerr << "anchor: changing draw_hilit to " << draw_hilit << ", saved " << oldhilit << "; " << draw_curmtl << endl;
    if (draw_curmtl)
      draw_curmtl->draw ();
  }

  // spec: WWWAnchor must behave like a separator (although it is not derived from it)
  int numlights = next_light;
  ge3d_push_matrix ();

  // draw children of group
  int n = getNumChildren ();
  for (int i = 0; i < n; i++)
    getChild (i)->draw ();

  ge3d_pop_matrix ();
  for (int i=numlights; i < NUM_LIGHTS; i++) light_array[i] = nil;


//  // test: draw a bounding box around anchor objects
//  if (scene_->linksActive ())
//  { // color change in effect for further primitives
//    ge3dLineColor (&scene_->col_anchoredge);
//    ge3dWireCube (&omin_, &omax_);
//  }


  if (draw_hilit != oldhilit)  // restore non-anchor material
  {
    //cerr << "anchor: resetting draw_hilit to " << oldhilit << "; " << draw_curmtl << endl;
    draw_hilit = oldhilit;  // draw_hilnonanch unless nested anchors
    if (draw_curmtl)
      draw_curmtl->draw ();
  } // TODO: keep track of draw_curmtl in Separators (restore; including Anchors)

} // QvWWWAnchor


void QvWWWInline::draw ()
{
  // fetch first time needed
  if (state_ == s_virgin)
  {
    DEBUGNL ("QvWWWInline first encountered. Sending request");
    state_ = s_requested;
    // the request of the inline URL is VRML specific, but was put into Scene3D to allow
    // separate treatment for different viewer types (Harmony, standalone, etc.)
    scene_->requestInlineURL (this, name.value.getString (), parentURL_.getString ());
  }

  // draw bounding box until scene fetched
  if (hasextent_ && state_ != s_completed)
  {
    ge3dWireCube (&omin_, &omax_);
  }

  QvGroup::draw ();  // draw children (if already fetched)
} // QvWWWInline
*/




/***** misc *****/


BUILDBSP(QvInfo)

void QvUnknownNode::buildBSP(BSPTree* bsp_root)
{
//cout << "*: QvUnKnownNode->buildBSP(bsp_root_);\n";

  QvGroup::buildBSP(bsp_root);
}



/***** extensions *****/


BUILDBSP(QvLabel)


void QvLightModel::buildBSP(BSPTree*)
{
//cout << "*: QvLightModel->buildBSP(bsp_root_);\n";

//  if (scene_->dolighting () == Scene3D::lighting_auto)
//    ge3dHint (hint_lighting, dolighting_);
}
