//<copyright>
//
// Copyright (c) 1992,93,94,95,96,97
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        hg3dvw.C
//
// Purpose:     implementation of class Hg3dViewer
//
// Created:      9 Jul 92   Keith Andrews, IICM (template)
//
// Changed:     28 Apr 97   Michael Pichler
//
// $Id: hg3dvw.C,v 1.25 1997/05/27 12:46:12 mpichler Exp $
//
// Description
// -----------
//
// Implementation of the abstract 3D scene viewer Hg3dViewer
// (independent of look-and-feel).
// Responsible for loading a 3D scene and managing the source anchors.
//
//
//</file>


#include "hg3dvw.h"

#include "scenewin.h"
#include "camera.h"
#include "geomobj.h"
#include "srcanch.h"
#include "material.h"
#include "stranslate.h"
#include "urlserver.h"  /* see below */

#include <vrml/QvSFString.h>
#include <vrml/QvWWWAnchor.h>
#include <vrml/QvWWWInline.h>

#include <InterViews/window.h>

//#include <Dispatch/rpcstream.h>
#include <hyperg/Dispatch/dispatcher.h>

#include "instrumented.h"

#include <hyperg/utils/str.h>
#include <hyperg/hyperg/object.h>
#include <hyperg/hyperg/message.h>
#include <hyperg/utils/verbose.h>

#include <hyperg/widgets/msgbox.h>

#include <string.h>
#include <ctype.h>
#include <iostream.h>
#include <unistd.h>
#include <stdlib.h>

#ifdef PMAX
// close prototype only contained in sysent.h, conflicts with other headers
extern "C" { int close (int); }
#endif


typedef long objid_type;


// SceneWindow::setdefaults sets this for VRweb,
// but harscened does not use this - linker error.
// hack (error recognized after VRweb 1.5 release test)
int URLServer::senduseragent_ = 1;



implementIOCallback(Hg3dViewer)


//
// class HyperGScene
//
// Manages the 3D scene and communication
// from the viewer to the viewer manager:
// * activation of links
// * definition of source and destination anchor
//


class HyperGScene: public SceneWindow
{
  public:
    HyperGScene (Session*, HgViewerManager*, Hg3dViewer*);

    // Scene3D
    void clear ();
    void deleteSourceAnchor (long id);
    int followLink (const GeometricObject*, const SourceAnchor*);
    int followLink (const QvWWWAnchor*);
    void setSourceAnchor ();
    void setDestinationAnchor ();
    void setDefDestAnchor ();

    void requestURLnode (QvWWWInline* nodeI, QvTexture2* nodeT, const char* url, const char* docurl)
    { hg3dviewer_->requestURLnode (nodeI, nodeT, url, docurl); }  // via ImageServer::requestURLdata

    const char* mostRecentURL () const
    { return hg3dviewer_->lastURL ();
    }

    void requestTextures ()  { hg3dviewer_->requestTextures (); }
    // the HyperGScene does not load textures demanded in the file,
    // but instead loads textures provided via "texture links"

    // Scene3D (see also viewer/vwstuff.h)
    void back ();
    void forward ();
    void history ();
    void hold ();
    void docinfo (int);
    void anchorInfo (const RString& anchorobj);

    // SceneWindow
    void beginFeedback ();
    void endFeedback ();
    void endFeedback (long, long);

    void setEditMode (int mode);

  private:
    HgViewerManager* manager_;
    Hg3dViewer* hg3dviewer_;
    int warnededit_;
};


HyperGScene::HyperGScene (Session* session, HgViewerManager* manager, Hg3dViewer* hg3dviewer)
: SceneWindow (session, STranslate::HARMONY3DVIEWER)
{
  manager_ = manager;
  hg3dviewer_ = hg3dviewer;
  warnededit_ = 0;
}


void HyperGScene::clear ()
{
  DEBUGNL ("HyperGScene::clear: clearing pending requests and scene data");
  hg3dviewer_->clearAllRequests ();
  SceneWindow::clear ();
}


void HyperGScene::deleteSourceAnchor (long id)
{
  if (!id)
    return;

  ObjectID ID (id);

  RString link = "ObjectID=" + ID.IDString () + "\n"
               + "DocumentID=" + hg3dviewer_->objIdStr () + "\n";
  DEBUGNL ("hg3d: delete link <" << link << ">.");

  manager_->deleteLink (link, hg3dviewer_);
}


int HyperGScene::followLink (const GeometricObject*, const SourceAnchor* anchor)
{
  if (!anchor)
    return 0;

  long id = anchor->id ();

  ObjectID ID (id);

  RString link ("ObjectID=" + ID.IDString ());
  DEBUGNL ("scene viewer: follow link to <" << link << ">.");
  manager_->followLink (link, hg3dviewer_);

  return 0;  // I am not responsible for redraw
}


int HyperGScene::followLink (const QvWWWAnchor* anchor)
{
  if (!anchor)
    return 0;

  // Hyper-G anchor
  if (anchor->hganchorid_)
  {
    // ObjectID ID (anchor->hganchorid_);
    // RString link ("ObjectID=" + ID.IDString ());

    // see VRMLScene::defineHyperGanchor
    RString link = anchor->parentURL_.getString ();  // parentURL_ holds complete anchor object
    link += "WWWType=text\n";  // text viewer will redirect the document appropriately

    DEBUGNL ("scene viewer: follow link to <" << link << ">.");
    manager_->followLink (link, hg3dviewer_);

    return 0;
  }

  // anchor URL (WWW)
  RString absurl;
  anchorURL (anchor, absurl);  // add point map if necessary

  RString msg;
  hg3dviewer_->parseURL (absurl, anchor->parentURL_.getString (), msg);

  msg += "WWWType=text\n";  // text viewer will redirect the document appropriately

  DEBUGNL ("scene viewer: follow link to <" << msg << ">.");
  manager_->followLink (msg, hg3dviewer_);

  return 0;  // I am not responsible for redraw
}


void HyperGScene::beginFeedback ()
{
  SceneWindow::beginFeedback ();
  Dispatcher::instance ().startTimer (0, (long int) feedbacktime (), feedbackhandler ());
  // no need for stopTimer
}


void HyperGScene::endFeedback ()
{ // end of feedback via IO-callback
}

void HyperGScene::endFeedback (long, long)
{
  SceneWindow::endFeedback ();
}


// editing a Hyperwave scene only edits a local copy

void HyperGScene::setEditMode (int mode)
{
  if (getEditMode () == EditMode::ed_viewer&& !warnededit_)  // warn when turning on editing
  {
    // this viewer is "hold"
    MessageBox::message (
      slanguage, appwin (),
      "You will edit a local copy of the scene, which you must\n"
      "insert back into the Hyperwave server manually.",
      "Warning",
      MessageBox::Ok
    );
    warnededit_ = 1;
  }

  SceneWindow::setEditMode (mode);
}


// setSourceAnchor
// define a Hyper-G source anchor

void HyperGScene::setSourceAnchor ()
{
  // specify the source anchor:

  const GeometricObject* selobj = selectedObj ();
  const QvNode* selnode = selectedNode ();

  if (selobj)  // SDF
  {
    // ObjectId=<ID of current scene>\n
    // Position=Object <ObjectNumber>[/<GroupName>]\n
    // (could also use <ObjectName> for Position instead)
    char onum [16];
    sprintf (onum, "%d", selobj->getobj_num ());

    RString srcanch (hg3dviewer_->docIdStr () + "Position=Object " + onum);
    const char* selgroup = selectedGroup ();
    if (selgroup)
      srcanch = srcanch + RString ("/") + selgroup;
    srcanch += "\n";

    DEBUGNL ("scene viewer: defining source anchor <" << srcanch << ">.");
    manager_->defineSourceAnchor (srcanch, hg3dviewer_);
  }

  if (selnode)  // VRML
  {
    // ObjectId=<ID of current scene>\n
    // Position=Path <selectedPosition>\n
    // (could also use name of selected node [if set and unique] for Position instead)
    // selectedPosition does not include anchor nodes inserted from link DB

    RString selpathpos;
    selectedPosition (selpathpos);

    if (!selpathpos.length () || *selpathpos.string () != ' ')
      return;  // bad selection for anchor definition (begins with ' ' if all ok.)

    RString srcanch (hg3dviewer_->docIdStr () + "Position=Path" + selpathpos + "\n");

    DEBUGNL ("scene viewer: defining source anchor <" << srcanch << ">.");
    manager_->defineSourceAnchor (srcanch, hg3dviewer_);
  }

} // setSourceAnchor


void HyperGScene::setDestinationAnchor ()
{
  // specify the destination anchor:
  // "ObjectId=<ID of current scene>\nPosition=<destination position>\n"
  Camera* cam = getCamera ();

  if (cam)
  {
    point3D pos, look;
    cam->getposition (pos);
    cam->getlookat (look);
    char ps [256];
    sprintf (ps, "(%f, %f, %f) (%f, %f, %f)\n", pos.x, pos.y, pos.z, look.x, look.y, look.z);

    RString dstanch (hg3dviewer_->docIdStr () + "Position=" + ps);
    DEBUGNL ("hg3d: defining destination anchor <" << dstanch << ">.");
    manager_->defineDestAnchor (dstanch, hg3dviewer_);
  }
  // cannot define a destination anchor without camera
}


void HyperGScene::setDefDestAnchor ()
{
  // specify the default destination anchor:
  // "ObjectId=<ID of current scene>\n"
  RString dstanch (hg3dviewer_->docIdStr ());
  DEBUGNL ("hg3d: defining default destination anchor <" << dstanch << ">.");
  manager_->defineDestAnchor (dstanch, hg3dviewer_);
}


void HyperGScene::back ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(n_back); //--- by iegger
#endif

  manager_->back (hg3dviewer_);
}

void HyperGScene::forward ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(n_forward); //--- by iegger
#endif

  manager_->forward (hg3dviewer_);
}

void HyperGScene::history ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(n_history); //--- by iegger
#endif

  manager_->history (hg3dviewer_);
}

void HyperGScene::hold ()
{
#ifdef INSTRUMENTED
  Instrumentation::instance()->write_to_log(n_hold); //--- by iegger
#endif

  DEBUGNL ("scene: hold document '" << hg3dviewer_->docIdStr () << "'.");
  manager_->hold (hg3dviewer_->docIdStr (), hg3dviewer_);
}


void HyperGScene::docinfo (int item)
{
  const RString& docobj = hg3dviewer_->docObj ();

  switch (item)
  {
    case info_references:
#ifdef INSTRUMENTED
      Instrumentation::instance()->write_to_log(n_ref_links); //--- by iegger
#endif
      manager_->showReferences (docobj, hg3dviewer_);
    break;
    case info_annotations:
#ifdef INSTRUMENTED
      Instrumentation::instance()->write_to_log(n_annotations); //--- by iegger
#endif
      manager_->showAnnotations (docobj, hg3dviewer_);
    break;
    // annotated now missing
    case info_parents:
#ifdef INSTRUMENTED
      Instrumentation::instance()->write_to_log(n_parents); //--- by iegger
#endif
      manager_->showParents (docobj, hg3dviewer_);
    break;
    case info_attributes:
#ifdef INSTRUMENTED
      Instrumentation::instance()->write_to_log(n_attributes); //--- by iegger
#endif
      manager_->showAttributes (docobj, hg3dviewer_);
    break;
    case info_textures:
#ifdef INSTRUMENTED
      Instrumentation::instance()->write_to_log(n_showtextures); //--- by iegger
#endif
      manager_->showTextures (docobj, hg3dviewer_);
    break;
    default:
      cerr << "HyperGScene::docinfo: index " << item << " out of range (internal error)." << endl;
  }
} // docinfo


void HyperGScene::anchorInfo (const RString& anchorobj)
{
  manager_->showAttributes (anchorobj, hg3dviewer_);
}



//
// class Hg3dViewer
//
// Used to display 3d documents, stored in sdf format.
//


Hg3dViewer::Hg3dViewer (HgViewerManager* manager, Session* session) 
: HgViewer (manager),
  // DcConnect (),  // RpcService (0),
  read_callback_ (this, &Hg3dViewer::readInput, nil, nil),
  imgserver_ (manager, this)
{
  manager_ = manager;
  terminated_ = 0;
  pipe_port_ = -1;
  to_close_ = -1;

  // open communication
  // if (!_service)  // RpcScervice
  // { HgMessage::fatalError ("could not open port for piping document.", 1);  // exit
  // }
  // no equivalent in DcConnect

  // options tell the Session Manager about the Viewer's features
  options_ = "ActiveConnect=on\n";  // both protocol directions
  // "Header=on\n", "Edit=on\n" ... some time in future

  scene_ = new HyperGScene (session, manager, this);

  Window* appwin = scene_->appwin ();
  appwin->map ();  // map application window
}


Hg3dViewer::~Hg3dViewer ()
{
  if (!terminated_)
    terminate ();
}


int Hg3dViewer::port () const           // tell document port (to viewer manager)
{
  return DcConnect::dcport ();  // port on which DcConnect will connect to DC

  // return RpcService::_port;
  // RpcService::_port is the port for initiating the connection
  // pipe_port_ is the port for the data
}


// getHostName
// make host string of serverID
// e.g. 0x811b9908 becomes 129.27.153.8

static RString getHostName (const RString& serverid)
{
  RString hostname;
  static const char* hexdigit = "0123456789abcdef";

  const char* sid = serverid.string ();
  if (strncmp (sid, "0x", 2))
  { cerr << "getHostName: server ID " << serverid << " does not start with 0x" << endl;
    return hostname;
  }
  sid += 2;
  while (sid [0] && sid [1])
  {
    if (hostname.length ())
      hostname += ".";
    // ass.: serverid consists of valid hex digits chars
    int num = 16 * (strchr (hexdigit, tolower (sid [0])) - hexdigit)
                 + (strchr (hexdigit, tolower (sid [1])) - hexdigit);
    char strbuf [16];
    sprintf (strbuf, "%d", num);
    hostname += strbuf;
    sid += 2;
  }
  return hostname;

} // getHostName


void Hg3dViewer::load (                 // load document
  const char* doc,                      //   document
  const char* anchors,                  //   source anchors
  const char* // info                      //   document info flags
)
{
  DEBUGNL ("hg3d: load document <" << doc << ">.");

  if (anchors)
  { DEBUGNL ("  3D source anchors <" << anchors << ">.");
  }
  else
  { DEBUGNL ("  without 3D source anchors.");
  }

  if (!doc || !*doc)
  { HgMessage::message ("note: no document specified for load.");
    return;
  }

  // maintain document id string (needed for terminate and anchor specification)
  Object dobj = Object (doc);
  docobj_ = dobj;
  objidstr_ = dobj.ID ().IDString ();  // id string of current object (document)
  docidstr_ = "ObjectID=" + objidstr_ + "\n";

  // remote URL: prot://host:port<path>, path has a leading '/'
  // HG-local URL: http://host/name, host of GOid, name empty if unset
  RString prot = dobj.protocol ();
  if (prot.length ())  // remote object
  {
    lasturl_ = docurl_ = dobj.protocol () + ":/""/" + dobj.host () + ":" + dobj.port () + dobj.path ();
  }
  else  // Hyper-G object
  {
    RString serverid = dobj.field ("GOid=").gSubstrIndex (0, 9);
    // cerr << "document:\n" << doc << "serverID: " << serverid << endl;
    lasturl_ = docurl_ = "http:/""/" + getHostName (serverid) + "/" + dobj.name ();
    // Harmony will connect to server via Hyper-G protocol (if local)
  }

  // cerr << "hg3d: document URL: " << docurl_ << endl;
  DEBUGNL ("hg3d: document URL: " << docurl_);
  scene_->currentURL (docurl_);

  DEBUGNL ("hg3d: loading scene with object ID <" << docidstr_ << ">.");

  updateTitle ();

  // if "Function=reload" the current document should be reused, readInput
  // will not be called, and we are in same situation as when reading completed

  DEBUGNL ("hg3d: pipe port for load: " << pipe_port_);

  srcanchstr_ = anchors;  // store the anchors

  // TODO: use document info flags to enable/disable menu entries
//   Object rinfo (info);
//   rinfo.tolower ();
//   prevVersion_ = rinfo.field ("previousversion=") == "on";
//   nextVersion_ = rinfo.field ("nextversion=") == "on";
//   annotation_ = rinfo.field ("isannotation=") == "on";
//   annotated_ = rinfo.field ("isannotated=") == "on";

  if (docobj_.field ("Function=") == "reload")
  {
    DEBUGNL ("hg3d: reloading for source anchor redefinition");

    readingCompleted ();  // already have document
  }
  else
  {
    RString dchost = dobj.field ("DCHost=");
    RString dcportstr = dobj.field ("DCPort=");
    int dcport = atoi (dcportstr.string ());

    DEBUGNL ("hg3d: connecting to DC on " << dchost << ":" << dcportstr << " (" << dcport << ")");

    imgserver_.setDCHostPort (dchost, dcport);
    connect (dchost, dcport);
  }

} // load


// void Hg3dViewer::flags (int bits)  // TODO
// {
//   historyBack_ = flags & HistoryBack;
//   historyForward_ = flags & HistoryForward;
// }


void Hg3dViewer::createReader (int fd)
{
  DEBUGNL ("Hg3dViewer::createReader (" << fd << ")");

  // close old port - not here, because data may still arrive
  if (pipe_port_ != -1)
    to_close_ = pipe_port_;
  else
    to_close_ = -1;

  // open new port
  pipe_port_ = fd;
  Dispatcher::instance ().link (pipe_port_, Dispatcher::ReadMask, &read_callback_);

} // createReader



void Hg3dViewer::storeanchor (Object& anobj)     // store an anchor object
{
  // extract fields from anchor object

  ObjectID ID;

  if (!anobj.ID (ID))
  { HgMessage::message ("warning: source anchor without object ID.");
    return;
  }
  objid_type objid = ID.id ();  // Object ID

  // State currently unused ("seen" will be evaluated in future)

  RString posRstr;  // anobj.field ("Position=");

  if (!anobj.position (posRstr))
  { HgMessage::message ("warning: source anchor without field 'Position='.");
    return;
  }

  const char* posstr = posRstr;

  // valid position specifications, where NN is an object number (decimal):
  //
  // a) "true" source anchors
  //   SDF:                             VRML:
  //   Object NN                        Path N N N ...
  //   Object NN/groupname              nodename
  //   objectname
  //   objectname/groupname
  //
  // b) texture anchors (LinkType "texture")
  //   materialname (SDF)               filename field of Texture2 node (VRML)
  //
  // c) anchors to be ignored (used for annotations)
  //   Position=invisible (any format)

  if (!strcmp (posstr, "invisible"))  // 19950523
  {
    DEBUGNL ("hg3d: invisible source anchor (" << posstr << ") encountered (ignored).");
    return;
  }

  if (anobj.field ("LinkType=") == "texture")  // texture anchor
  {
    RString dest = anobj.field ("Dest=");
    if (!dest.length ())
    { HgMessage::message ("warning: texture anchor without field 'Dest='.");
      return;
    }
    DEBUGNL ("harscened: texture anchor to destination " << dest);

    Material* mat = 0;
    QvTexture2* tex = 0;
    scene_->findTextureMaterial (posstr, mat, tex);

    if (mat || tex)
      imgserver_.appendRequest (objIdStr (), anobj, dest, mat, tex);
    else
      cerr << "harscened. warning: invalid texture anchor position (material name) '" << posstr << "'" << endl;

    return;
  }

  // TODO: handle inline anchors (VRML)

  // else: ordinary source anchor

  RString field;
  getLangField ("Title", anobj, field);
  const char* title = field.string ();  // see HgIv3dViewer::updateTitle
  if (title && strlen (title) > 2 && title [2] == ':')
    title += 3;
  scene_->defineHyperGanchor (objid, anobj, posstr, title);

} // storeanchor


// getPos: tell current position
// it will used on "back" in history as destination anchor
// with a field "Function=NoMark" to turn off highlighting of destination anchor

void Hg3dViewer::getPos (RString& position)
{
  Camera* cam = scene_->getCamera ();

  if (cam)
  {
    point3D pos, look;
    cam->getposition (pos);
    cam->getlookat (look);
    char ps [256];
    sprintf (ps, "(%f, %f, %f) (%f, %f, %f)\n", pos.x, pos.y, pos.z, look.x, look.y, look.z);

    position = "Position=";
    position += ps;
    DEBUGNL ("hg3d: reporting current position <" << position << ">.");
  }
} // getPos


int Hg3dViewer::readImage (Material* mat, QvTexture2* tex, const char* filename)
{
  if (mat)
    scene_->loadTextureFile (mat, filename);  // does redraw if necessary
  if (tex)
  { if (scene_->readTextureFile (tex, filename))
      scene_->redraw ();
  }
  return 0;  // have to return 1 when reading should be stopped
}


int Hg3dViewer::readInlineURL (QvWWWInline* nodeI, QvTexture2* nodeT, const char* filename, const char* absurl)
{
  int redraw = 0;

  // compressed data will be handled by SceneWindow
  if (nodeI)
  {
    lasturl_ = absurl;
    scene_->currentURL (absurl);
    // cerr << "new parent URL set to " << absurl << endl;

    if (scene_->readInlineVRMLFile (nodeI, filename))
      redraw = 1;
  }
  if (nodeT && scene_->readTextureFile (nodeT, filename))
    redraw = 1;

  if (redraw)
    scene_->redraw ();

  return 0;  // 0 to continue, 1 to stop
}


void Hg3dViewer::requestImage (const char* texname)
{
  char buf [256];
  sprintf (buf, "%s %.200s", STranslate::str (STranslate::ProgressREQUESTTEXTURE), texname);
  scene_->statusMessage (buf);
}


void Hg3dViewer::requestURL (const char* url, int texture)
{
  char buf [256];
  sprintf (buf, "%s %.200s",
    STranslate::str (texture ? STranslate::ProgressREQUESTTEXTURE : STranslate::ProgressREQUESTINLINEVRML),
    url);
  scene_->statusMessage (buf);
}


void Hg3dViewer::requestFinished ()
{
  scene_->showTitle ();  // switch back to title
}


void Hg3dViewer::cancelRequest (const char* refno_str)
{
  DEBUGNL ("Hg3dViewer::cancelRequest (" << refno_str << ")");
  imgserver_.cancelRequest (refno_str);
}


void Hg3dViewer::terminate ()           // terminate viewer
{
  if (terminated_)
  { HgMessage::message ("possibly strange: terminate was called twice.");
    return;
  }

  DEBUGNL ("hg3d: terminating.");

  if (pipe_port_ != -1)
  { ::close (pipe_port_);
    pipe_port_ = -1;
  }
  if (to_close_ != -1)
  { ::close (to_close_);
    to_close_ = -1;
  }

  // RpcService::stopListening ();  // done in ~DcConnect

  terminated_ = 1;
  delete scene_;
  scene_ = 0;

  manager_->viewerTerminated (docidstr_, this);
  DEBUGNL ("hg3d: termination reported to viewer manager.");
}



// storeSourceAnchors
// stores source anchors, possible after load () and readInput () arrived

void Hg3dViewer::storeSourceAnchors ()
{
  DEBUGNL ("Hg3dViewer::storeSourceAnchors");

  // requests of imageserver already cleared in HyperGScene::clear

  // clear "default" anchors of scene (possibly contained in SDF file)
  scene_->clearAnchors ();
  // anchors in VRML scenes now kept in (must do so if got scene from WWW)

  const char* anchors = srcanchstr_;

  if (anchors && *anchors)  // store new source anchors
  {
    char* anchor_str = new char [strlen (anchors) + 1];
    strcpy (anchor_str, anchors);  // working copy for parsing

    char* aptr;  char* eptr;

    // all anchors are terminated with empty lines ("\n\n")

    for (aptr = anchor_str;  (eptr = strstr (aptr, "\n\n")) != nil;  aptr = eptr+1)
    {
      *++eptr = '\0';  // eptr discards second \n
      // aptr now contains an anchor definition

      Object anobj (aptr);  // current anchor
      storeanchor (anobj);

    } // for all anchors

    delete anchor_str;

  } // if anchors

  GeometricObject* selobj = scene_->selectedObj ();
  if (selobj)
  { // anchors on selected object possibly (likely) changed on reload
    selobj->checkAnchorSelection ();
    scene_->selectionChanged ();
  }

} // storeSourceAnchors



// readInput
// read data
// currently the whole sdf-file is read at once --
// later it will be possible to read only pieces at once
// TODO: write input piece by piece onto temporary file, progress feedback

int Hg3dViewer::readInput (int fd)
{
//     Dispatcher::instance ().unlink (pipe_port_);
//     ::close (pipe_port_);
  if (fd == to_close_)  // fd to close
  { ::close (to_close_);
    to_close_ = -1;
    return -1;  // to remove the handler from the dispatcher
  }

  DEBUGNL ("hg3d: readInput from file descriptor " << fd);

  FILE* file = fdopen (fd, "r");

  if (!file)
  {
    HgMessage::fatalError ("could not read from document port.", 3);  // exit
  }

  int retval = scene_->readSceneFILE (file);  // clear old data and read new scene
  fclose (file);  // 19950831

  if (retval)
  { // if readscene detects an error it should be reported to the viewer manager here
    HgMessage::error ("error in scene description file.");
  }
  else
  { DEBUGNL ("hg3d: read input from fd " << fd << " - OK.");
  }

  // Dispatcher::instance ().unlink (fd);  // should not be necessary
  readingCompleted ();
  pipe_port_ = -1;
  return -1;  // to remove the handler from the dispatcher

} // readInput
