//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>
//
// Note: this triangulation code is a modified version of:
// Fast Polygon Triangulation Based on Seidel's Algorithm,
// Atul Narkhede and Dinesh Manocha (both Univ. of North Carolina)
// Graphics Gems V, Academic Press, 1995 (ch. VII.5, p. 394 - 397).
//
// The original source (public domain) is available at
// ftp://princeton.edu/pub/Graphics/GraphicsGems/GemsV
//
// Complementary algorithms taken from:
// Computational Geometry in C, Joseph O'Rourke,
// Cambridge University Press, 1993

//<file>
//
// Name:        triangulate.C
//
// Purpose:     triangulation (convexify) of IndexedFaceSet
//
// Created:      4 Apr 96   Richard Persche
//
// Changed:      3 Jun 96   Richard Persche
//
// Changed:     17 Jun 96   Michael Pichler
//
// $Id: triangulate.C,v 1.6 1997/02/25 17:03:58 mpichler Exp $
//
//</file>



#include "vrmlscene.h"
#include "vecutil.h"
#include "arrays.h"
#include "monotone.h"

#include <vrml/QvIndexedFaceSet.h>
#include <ge3d/vectors.h>
#include <iostream.h>
#include <math.h>

#include <hyperg/utils/types.h>
#include <hyperg/utils/verbose.h>

#define MAXVERTEX 1000


typedef double tuple [2]; // to store points
typedef int triple [3]; // to store triangles

// area2 : double area of a triangle. area2 > 0 if CCW
static double area2 (const double* a, const double* b, const double* c)
{
  return
    a[0] * b[1] - a[1] * b[0] +
    a[1] * c[0] - a[0] * c[1] +
    b[0] * c[1] - b[1] * c[0];
}

// area of a polygon. area > 0 if CCW
static double area (int n, double polygon [] [2])
{
  double sum = 0.0;

  for (int i = 2; i < n; i++)
    sum += area2( polygon[1], polygon[i], polygon[i+1] );

  return sum;
}

// checks if a polygon (CCW) is convex
static boolean isConvex (int n, double polygon [] [2])
{
  int i = 1;

  if (n > 3)
    while (i <= n)
    {
      if (area2 (polygon [i], polygon [(i+1)%n], polygon [(i+2)%n]) < 0.0)
	return false;
      i++;
    }

  return true;
}

// checks if vertex c lies on the linesegment ab
static boolean isBetween( double* a,
                      double* b,
		      double* c )
{
  if (area2 (a, b, c) == 0.0)  // collinear
  {
    if (a[0] == b[0])  // vertical
    {
      if ( b[1] > a[1] )
      { if ( (c[1] >= a[1]) && (c[1] <= b[1]) )
          return true;
      }
      else
      { if ( (c[1] <= a[1]) && (c[1] >= b[1]) )
          return true;
      }
    }
    else               // not vertical
    {
      if ( b[0] > a[0] )
      { if ( (c[0] >= a[0]) && (c[0] <= b[0]) )
          return true;
      }
      else
      { if ( (c[0] <= a[0]) && (c[0] >= b[0]) )
          return true;
      }
    }
  }

  return false;
}

// checks if the segments (i,i+1) and (j,j+1) intersect
static boolean intersection (int i, int j, double polygon [] [2])
{
  boolean intersect = false;
  intersect = ( (area2( polygon[i], polygon[i+1], polygon[j]) > 0.0 &&
		 area2( polygon[i], polygon[i+1], polygon[j+1]) < 0.0 )
	      ||(area2( polygon[i], polygon[i+1], polygon[j]) < 0.0 &&
		 area2( polygon[i], polygon[i+1], polygon[j+1]) > 0.0 ));
  intersect = intersect &&
              ( (area2( polygon[j], polygon[j+1], polygon[i]) > 0.0 &&
		 area2( polygon[j], polygon[j+1], polygon[i+1]) < 0.0 )
	      ||(area2( polygon[j], polygon[j+1], polygon[i]) < 0.0 &&
		 area2( polygon[j], polygon[j+1], polygon[i+1]) > 0.0 ));
  intersect = intersect || isBetween ( polygon[i], polygon[i+1], polygon[j] ) ||
                           isBetween ( polygon[i], polygon[i+1], polygon[j+1] ) ||
                           isBetween ( polygon[j], polygon[j+1], polygon[i] ) ||
                           isBetween ( polygon[j], polygon[j+1], polygon[i+1] );
  return intersect;
}

// checks if a polygon is not self intersecting  O(n^2)
// TODO: faster algorithm
static boolean isSimple (int n, double polygon [] [2])
{
  int i = 0;
  int j = 2;

  while (i < n)
  {
    while (j <= i + n - 2)
    {
      if (intersection (i, (j % n), polygon))
        return false;

      j++;
    }
    i++;
    j = i+2;
  }

  return true;
}


// changes vertexorder of a polygon
static void flipPolygon( int n,
			 double polygon [] [2],
			 unsigned int * indexArray )
{
  double swapCoordinate;
  int swapIndex;
  int i;

  for ( i = 1; i <= (n/2); i++ )
  {
      swapCoordinate = polygon [i][0];
      polygon [i][0] = polygon[n-i][0];
      polygon [n-i][0] = swapCoordinate;

      swapCoordinate = polygon [i][1];
      polygon [i][1] = polygon[n-i][1];
      polygon [n-i][1] = swapCoordinate;

      swapIndex = indexArray[i];
      indexArray[i] = indexArray[n-i];
      indexArray[n-i] = swapIndex;
  }
}

static void computeNormalizedNormal( point3D a,
				     point3D b,
				     point3D c,
				     vector3D& normal )
{
  computeNormal (a, b, c, normal);
  float norm2 = dot3D (normal, normal);
  if (norm2 > 0.0)  // normalize
  {
    norm2 = 1 / sqrt (norm2);
    scl3D (normal, norm2);
  }
}



// convexifyFaceSet


void VRMLScene::convexifyFaceSet (QvIndexedFaceSet* faceSet)
{
  unsigned int i = 0;                 // vertex counter
  unsigned int c = 0;                 // convex vertex counter
  unsigned int n = 0;                 // # vertex per polygon
  unsigned int faceNum = 0;           // face counter
  unsigned int convexFaceNum = 0;     // convex face counter
  unsigned int direction = 0;         // direction of the facenormal
  double normalMaxCoordinate = 0;     // magnitude of the facenormal
  boolean flipedPolygon;                 // vertexorder was changed
  tuple* vertexArray = new tuple [ MAXVERTEX  ];  // 2D vertex array
  unsigned int* vertexIndexArray = new unsigned int [ MAXVERTEX ];
  triple* triangles =  new triple [ MAXVERTEX ];     // result of the triangulation
  faceSet->convexFaceNormals_.allocValues ( faceSet->numvertinds_ -2 ); // face normals
  vector3D* cfn = ( vector3D* )faceSet->convexFaceNormals_.values;
  vector3D convexNormal;

  if (!doconvexify_ || !faceSet)
    return;

  DEBUGNL ("convexify for faceSet " << (void*) faceSet << " ...");


  while ( i < faceSet->coordIndex.num )
  {
      // determine the largest coordinat of the face-normal
      direction = 0;
      normalMaxCoordinate = fabs(faceSet->facenormals_ [faceNum].x);
      if ( fabs(faceSet->facenormals_ [faceNum].y) > normalMaxCoordinate )
      {
        direction = 1;
        normalMaxCoordinate = fabs(faceSet->facenormals_ [faceNum].y);
      }
      if ( fabs(faceSet->facenormals_ [faceNum].z) > normalMaxCoordinate )
        direction = 2;

      n = 1;  // no. of vertices
      while ( faceSet->coordIndex.values[i] != -1 ) // projection into 2D plane
      {
          const point3D& curvert = faceSet->vertexlist_ [faceSet->coordIndex.values[i]];
	  switch ( direction )
          {
	    case 0:  // yz plane
	      vertexArray [n][0] = - curvert.z;
	      vertexArray [n][1] = curvert.y;
            break;
	    case 1: // xz plane
	      vertexArray [n][0] = curvert.z;
	      vertexArray [n][1] = curvert.x;
            break;
	    case 2: // xy plane
	      vertexArray [n][0] = curvert.x;
	      vertexArray [n][1] = curvert.y;
            break;
          }
	  vertexIndexArray [n] = faceSet->coordIndex.values[i];
	  if ( n == 1 )
            n++;
	  else if ( vertexArray[n][0] != vertexArray[n-1][0] ||
		    vertexArray[n][1] != vertexArray[n-1][1] )
	    n++; // ignore duplicates
	  i++;
      }
      n--;
      vertexArray [0][0] = vertexArray [n][0];
      vertexArray [0][1] = vertexArray [n][1];

      // change vertexorder if vertexorder is not CCW
      flipedPolygon = false;
      if (area (n, vertexArray) < 0)
      {
	  flipPolygon (n, vertexArray, vertexIndexArray);
	  flipedPolygon = true;
      }

      if ( !isConvex( n, vertexArray ) && isSimple( n, vertexArray ) )  // need and able to triangulate
      {
          triangulate_polygon( n, vertexArray, triangles );
	  // compute face normal from the first triangle
	  if ( flipedPolygon )
	    computeNormalizedNormal(
	      faceSet->vertexlist_[ vertexIndexArray[ triangles[n-3][2] ] ],
	      faceSet->vertexlist_[ vertexIndexArray[ triangles[n-3][1] ] ],
	      faceSet->vertexlist_[ vertexIndexArray[ triangles[n-3][0] ] ],
	      convexNormal );
	  else
	    computeNormalizedNormal(
	      faceSet->vertexlist_[ vertexIndexArray[ triangles[0][0] ] ],
	      faceSet->vertexlist_[ vertexIndexArray[ triangles[0][1] ] ],
	      faceSet->vertexlist_[ vertexIndexArray[ triangles[0][2] ] ],
	      convexNormal );

	  faceSet->convexCoordIndex.allocValues( faceSet->convexCoordIndex.num + (n-2) * 4 );


	  for ( int t = 0; t < n-2; t++ )   // n-2 new triangles
          {
	      for ( int tx = 0; tx < 3; tx ++ )  // each triangle has 3 vertices
              {
		  if ( flipedPolygon )
		    faceSet->convexCoordIndex.values[c] = vertexIndexArray[ triangles[n-3-t][2-tx] ];
		  else
		    faceSet->convexCoordIndex.values[c] = vertexIndexArray[ triangles[t][tx] ];
		  c++;
              }
	      faceSet->convexCoordIndex.values[c] = QV_END_FACE_INDEX;
	      *cfn = convexNormal;
	      cfn++;
              convexFaceNum++;
	      c++;
          }
      }
      else // no triangulation
      {
	  faceSet->convexCoordIndex.allocValues( faceSet->convexCoordIndex.num + n + 1 );
	  for ( int t = 0; t < n; t++ )   // just copy the vertexIndices
	    {
	      if ( flipedPolygon )
		faceSet->convexCoordIndex.values[c] =
		  vertexIndexArray[ n-t ];
	      else
		faceSet->convexCoordIndex.values[c] =
		  vertexIndexArray[ t+1 ];
	      c++;
	    }
	  faceSet->convexCoordIndex.values[c] = QV_END_FACE_INDEX;
	  *cfn = faceSet->facenormals_ [faceNum];
	  cfn++;
          convexFaceNum++;
	  c++;
      }
      i++;
      faceNum++;
  }

  faceSet->convexCoordIndex.allocValues (faceSet->convexCoordIndex.num - 1);// shrink to actual number
  faceSet->convexFaceNormals_.allocValues (convexFaceNum);  // shrink to actual number
  faceSet->numconvexinds_ = faceSet->convexCoordIndex.num;

  DEBUGNL (convexFaceNum << " triangles made out of " << faceNum << " faces");

  delete [] vertexArray;
  delete [] vertexIndexArray;
  delete [] triangles;

  DEBUGNL ("... finished convexify");

} // convexifyFaceSet
