/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef linux

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <pwd.h>
#include <dirent.h>
#include <sys/param.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <errno.h>

#include "wflisteners.h"
#include "defs.h"


#define _PATH_PROCNET_TCP "/proc/net/tcp"
#define _PATH_PROCNET_UDP "/proc/net/udp"

enum {
  TCP_ESTABLISHED = 1,
  TCP_SYN_SENT,
  TCP_SYN_RECV,
  TCP_FIN_WAIT1,
  TCP_FIN_WAIT2,
  TCP_TIME_WAIT,
  TCP_CLOSE,
  TCP_CLOSE_WAIT,
  TCP_LAST_ACK,
  TCP_LISTEN,
  TCP_CLOSING
};

#define PROGNAME_WIDTH  20

#define PRG_SOCKET_PFX  "socket:["
#define PRG_SOCKET_PFXl  (strlen(PRG_SOCKET_PFX))

#ifndef LINE_MAX
#define LINE_MAX  4096
#endif

#define PATH_PROC  "/proc"
#define PATH_FD_SUFF  "fd"
#define PATH_FD_SUFFl  strlen(PATH_FD_SUFF)
#define PATH_PROC_X_FD  PATH_PROC "/%s/" PATH_FD_SUFF
#define PATH_CMDLINE  "cmdline"
#define PATH_CMDLINEl  strlen(PATH_CMDLINE)


#define PRG_HASH_SIZE  211
#define PRG_HASHIT(x)  ((x) % PRG_HASH_SIZE)

static struct prg_node {
  struct prg_node *next;
  int inode;
  int pid;
  char name[PROGNAME_WIDTH];
} *prg_hash[PRG_HASH_SIZE];

static char prg_cache_loaded = 0;


static void
prg_cache_add(int inode, int pid, char *name) {
  unsigned hi = PRG_HASHIT(inode);
  struct prg_node **pnp, *pn;
  
  prg_cache_loaded = 2;
  for (pnp = prg_hash + hi; (pn = *pnp); pnp = &pn->next) {
    if (pn->inode == inode) {
      /* Some warning should be appropriate here
	 as we got multiple processes for one i-node */
      return;
    }
  }
  if ((*pnp = (struct prg_node*)malloc(sizeof(**pnp))) == NULL)
    return;

  pn = *pnp;
  pn->next = NULL;
  pn->inode = inode;
  pn->pid = pid;
  if (strlen(name) > sizeof(pn->name) - 1)
    name[sizeof(pn->name) - 1] = '\0';
  strcpy(pn->name, name);
}

static const char*
prg_cache_get(int inode, int *pid) {
  unsigned int hi = PRG_HASHIT(inode);
  struct prg_node *pn;
  
  for (pn = prg_hash[hi]; pn != NULL; pn = pn->next)
    if (pn->inode == inode) {
      *pid = pn->pid;
      return pn->name;
    }

  *pid = 0;
  return "-";
}

static void
prg_cache_load(void) {
  char line[LINE_MAX], *serr, eacces = 0;
  int procfdlen, fd, cmdllen, lnamelen;
  char lname[30], cmdlbuf[512], finbuf[PROGNAME_WIDTH];
  long inode;
  const char *cs, *cmdlp;
  DIR *dirproc, *dirfd = NULL;
  struct dirent *direproc, *direfd;
  
  if (prg_cache_loaded)
    return;
  prg_cache_loaded = 1;
  
  cmdlbuf[sizeof(cmdlbuf)-1] = '\0';
  dirproc = opendir(PATH_PROC);
  if (dirproc == NULL)
    goto fail;

  while (errno = 0, direproc = readdir(dirproc)) {
#ifdef DIRENT_HAVE_D_TYPE_WORKS
    if (direproc->d_type != DT_DIR)
      continue;
#endif
    for (cs = direproc->d_name; *cs; cs++)
      if (!isdigit(*cs)) 
	break;
    if (*cs) 
      continue;
    procfdlen = snprintf(line, sizeof(line), PATH_PROC_X_FD, direproc->d_name);
    if (procfdlen <= 0 || procfdlen >= (int)sizeof(line) - 5) 
      continue;
    
    errno = 0;
    dirfd = opendir(line);
    if (dirfd == NULL) {
      if (errno == EACCES) 
	eacces = 1;
      continue;
    }
    line[procfdlen] = '/';
    cmdlp = NULL;
    while ((direfd = readdir(dirfd))) {
#ifdef DIRENT_HAVE_D_TYPE_WORKS
      if (direfd->d_type != DT_LNK) 
	continue;
#endif
      if (procfdlen + 1 + strlen(direfd->d_name) + 1 > sizeof(line)) 
	continue;
      memcpy(line + procfdlen - PATH_FD_SUFFl, PATH_FD_SUFF "/",
	     PATH_FD_SUFFl + 1);
      strcpy(line + procfdlen + 1, direfd->d_name);
      lnamelen = readlink(line, lname, sizeof(lname));
      if (lnamelen < (int)strlen(PRG_SOCKET_PFX + 2)) 
	continue;
      if (memcmp(lname, PRG_SOCKET_PFX, PRG_SOCKET_PFXl) ||
	  lname[lnamelen - 1] != ']') 
	continue;
      lname[lnamelen - 1] = '\0';
      inode = strtol(lname + PRG_SOCKET_PFXl, &serr, 0);
      if (!serr || *serr || inode < 0 || inode >= INT_MAX) 
	continue;
      
      if (!cmdlp) {
	if (procfdlen - PATH_FD_SUFFl + PATH_CMDLINEl >= sizeof(line) - 5) 
	  continue;
	strcpy(line + procfdlen - PATH_FD_SUFFl, PATH_CMDLINE);
	fd = open(line, O_RDONLY);
	if (fd < 0) 
	  continue;
	cmdllen = read(fd, cmdlbuf, sizeof(cmdlbuf) - 1);
	if (close(fd)) 
	  continue;
	if (cmdllen == -1) 
	  continue;
	if (cmdllen < (int)sizeof(cmdlbuf) - 1) 
	  cmdlbuf[cmdllen] = '\0';
	if ((cmdlp = strrchr(cmdlbuf, '/'))) 
	  cmdlp++;
	else 
	  cmdlp = cmdlbuf;
      }
      
      strncpy(finbuf, cmdlp, sizeof(finbuf));
      {
	int pid = atoi(direproc->d_name);
	prg_cache_add(inode, pid, finbuf);
      }
    }
    closedir(dirfd); 
    dirfd = NULL;
  }
  if (dirproc) 
    closedir(dirproc);
  if (dirfd) 
    closedir(dirfd);
  if (!eacces) 
    return;
  if (prg_cache_loaded == 1) {
  fail:
    fprintf(stderr, _("(No info could be read for process: geteuid()=%d but you should be root.)\n"),
	    geteuid());
  }
  /*
  else
    fprintf(stderr, _("(Not all processes could be identified, non-owned process info\n"
		      " will not be shown, you would have to be root to see it all.)\n"));
  */ /* this should not be printed */
}

static wf_listener*
listener_get_tcp(const char* line) {
  unsigned long rxq, txq, time_len, retr, inode;
  int num, local_port, rem_port, d, state, uid, timer_run, timeout;
  char rem_addr[128], local_addr[128], more[512];
#if HAVE_AFINET6
  struct sockaddr_in6 localaddr, remaddr;
  char addr6p[16][3], addr6[128];
  extern struct aftype inet6_aftype;
#else
  struct sockaddr_in localaddr, remaddr;
#endif

  more[0] = '\0';
  num = sscanf(line,
	       "%d: %64[0-9A-Fa-f]:%X %64[0-9A-Fa-f]:%X %X %lX:%lX %X:%lX %lX %d %d %ld %512s\n",
	       &d, local_addr, &local_port, rem_addr, &rem_port, &state,
	       &txq, &rxq, &timer_run, &time_len, &retr, &uid, &timeout,
	       &inode, more);

  if (strlen(local_addr) > 8) {
#if HAVE_AFINET6
    /* Demangle what the kernel gives us */
    sscanf(local_addr,
	   "%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s",
	   addr6p[0], addr6p[1], addr6p[2], addr6p[3],
	   addr6p[4], addr6p[5], addr6p[6], addr6p[7],
	   addr6p[8], addr6p[9], addr6p[10], addr6p[11],
	   addr6p[12], addr6p[13], addr6p[14], addr6p[15]);
    snprintf(addr6, sizeof(addr6), "%s%s:%s%s:%s%s:%s%s:%s%s:%s%s:%s%s:%s%s",
	     addr6p[3], addr6p[2], addr6p[1], addr6p[0],
	     addr6p[7], addr6p[6], addr6p[5], addr6p[4],
	     addr6p[11], addr6p[10], addr6p[9], addr6p[8],
	     addr6p[15], addr6p[14], addr6p[13], addr6p[12]);
    inet6_aftype.input(1, addr6, (struct sockaddr *)&localaddr);
    sscanf(rem_addr,
	   "%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s%2s",
	   addr6p[0], addr6p[1], addr6p[2], addr6p[3],
	   addr6p[4], addr6p[5], addr6p[6], addr6p[7],
	   addr6p[8], addr6p[9], addr6p[10], addr6p[11],
	   addr6p[12], addr6p[13], addr6p[14], addr6p[15]);
    snprintf(addr6, sizeof(addr6), "%s%s:%s%s:%s%s:%s%s:%s%s:%s%s:%s%s:%s%s",
	     addr6p[3], addr6p[2], addr6p[1], addr6p[0],
	     addr6p[7], addr6p[6], addr6p[5], addr6p[4],
	     addr6p[11], addr6p[10], addr6p[9], addr6p[8],
	     addr6p[15], addr6p[14], addr6p[13], addr6p[12]);
    inet6_aftype.input(1, addr6, (struct sockaddr *)&remaddr);
    localaddr.sin6_family = AF_INET6;
    remaddr.sin6_family = AF_INET6;
#endif
  }
  else {
    sscanf(local_addr, "%X",
	   &(&localaddr)->sin_addr.s_addr);
    sscanf(rem_addr, "%X",
	   &(&remaddr)->sin_addr.s_addr);
    ((struct sockaddr *)&localaddr)->sa_family = AF_INET;
    ((struct sockaddr *)&remaddr)->sa_family = AF_INET;
  }

  if (num < 11) {
    fprintf(stderr, _("warning, got bogus tcp line.\n"));
    return NULL;
  }

  if (((struct sockaddr *)&localaddr)->sa_family != AF_INET) {
    fprintf(stderr, _("warning, unsupported address family %d !\n"),
	    ((struct sockaddr *)&localaddr)->sa_family);
    return NULL;
  }

  if (state != TCP_LISTEN) {
    //    fprintf(stderr, _("warning, state != LISTEN: %i.\n"), state);
    return NULL;
  }

  if (rem_port)
    return NULL;

  int pid;
  string prog;

  prog = prg_cache_get(inode, &pid);

  wf_listener* listener = new wf_listener(IPPROTO_TCP, local_port,
					  localaddr.sin_addr.s_addr, prog,
					  uid, pid);
  return listener;
}

static wf_listener*
listener_get_udp(const char* line) {
  unsigned long rxq, txq, time_len, retr, inode;
  char local_addr[64], rem_addr[64], more[512];
  int num, local_port, rem_port, d, state, timer_run, uid, timeout;
#if HAVE_AFINET6
  struct sockaddr_in6 localaddr, remaddr;
  char addr6p[8][5];
  char addr6[128];
  extern struct aftype inet6_aftype;
#else
  struct sockaddr_in localaddr, remaddr;
#endif

  more[0] = '\0';
  num = sscanf(line,
	       "%d: %64[0-9A-Fa-f]:%X %64[0-9A-Fa-f]:%X %X %lX:%lX %X:%lX %lX %d %d %ld %512s\n",
	       &d, local_addr, &local_port, rem_addr, &rem_port, &state,
	       &txq, &rxq, &timer_run, &time_len, &retr, &uid, &timeout,
	       &inode, more);

  if (strlen(local_addr) > 8) {
#if HAVE_AFINET6
    sscanf(local_addr, "%4s%4s%4s%4s%4s%4s%4s%4s",
	   addr6p[0], addr6p[1], addr6p[2], addr6p[3],
	   addr6p[4], addr6p[5], addr6p[6], addr6p[7]);
    snprintf(addr6, sizeof(addr6), "%s:%s:%s:%s:%s:%s:%s:%s",
	     addr6p[0], addr6p[1], addr6p[2], addr6p[3],
	     addr6p[4], addr6p[5], addr6p[6], addr6p[7]);
    inet6_aftype.input(1, addr6, (struct sockaddr *) &localaddr);
    sscanf(rem_addr, "%4s%4s%4s%4s%4s%4s%4s%4s",
	   addr6p[0], addr6p[1], addr6p[2], addr6p[3],
	   addr6p[4], addr6p[5], addr6p[6], addr6p[7]);
    snprintf(addr6, sizeof(addr6), "%s:%s:%s:%s:%s:%s:%s:%s",
	     addr6p[0], addr6p[1], addr6p[2], addr6p[3],
	     addr6p[4], addr6p[5], addr6p[6], addr6p[7]);
    inet6_aftype.input(1, addr6, (struct sockaddr *) &remaddr);
    localaddr.sin6_family = AF_INET6;
    remaddr.sin6_family = AF_INET6;
#endif
  }
  else {
    sscanf(local_addr, "%X",
	   &(&localaddr)->sin_addr.s_addr);
    sscanf(rem_addr, "%X",
	   &(&remaddr)->sin_addr.s_addr);
    ((struct sockaddr *)&localaddr)->sa_family = AF_INET;
    ((struct sockaddr *)&remaddr)->sa_family = AF_INET;
  }

  if (num < 10) {
    fprintf(stderr, _("warning, got bogus udp line.\n"));
    return NULL;
  }
  if (((struct sockaddr *)&localaddr)->sa_family != AF_INET) {
    fprintf(stderr, _("warning, unsupported address family %d !\n"),
	    ((struct sockaddr *)&localaddr)->sa_family);
    return NULL;
  }

  if (state != TCP_CLOSE) {
    //    fprintf(stderr, _("warning, state != TCP_CLOSE: %i.\n"), state);
    return NULL;
  }

#if HAVE_AFINET6
#define notnull(A) (((A.sin6_family == AF_INET6) && \
	 ((A.sin6_addr.s6_addr32[0]) ||            \
	  (A.sin6_addr.s6_addr32[1]) ||            \
	  (A.sin6_addr.s6_addr32[2]) ||            \
	  (A.sin6_addr.s6_addr32[3]))) ||          \
	((A.sin6_family == AF_INET) &&             \
	 ((struct sockaddr_in *) &A)->sin_addr.s_addr))
#else
#define notnull(A) (A.sin_addr.s_addr)
#endif

  if (notnull(remaddr))
    return NULL;

  int pid;
  string prog;

  prog = prg_cache_get(inode, &pid);

  wf_listener* listener = new wf_listener(IPPROTO_UDP, local_port,
					  localaddr.sin_addr.s_addr, prog,
					  uid, pid);
  return listener;
}

bool
wf_listeners::probe_local_linux(int proto) {
  char* filename;
  wf_listener* (*parse_line)(const char*);
  FILE* procfile;
  char buffer[8192];

  if (proto == IPPROTO_TCP) {
    filename = _PATH_PROCNET_TCP;
    parse_line = listener_get_tcp;
  }
  else if (proto == IPPROTO_UDP) {
    filename = _PATH_PROCNET_UDP;
    parse_line = listener_get_udp;
  }
  else {
    /* print protocol name if available, number if not RV@@1 */
    fprintf(stderr, _("Error: unknown protocol %i.\n"), proto);
    return false;
  }

  procfile = fopen(filename, "r");
  if (procfile == NULL) {
    if (errno != ENOENT)
      perror(filename);
    else
      fprintf(stderr, _("Error: no support for `%s' on this system.\n"),
	      filename);
    return false;
  }

  prg_cache_load();

  fgets(buffer, sizeof(buffer), procfile); /* ignore first line */
  do {
    if (fgets(buffer, sizeof(buffer), procfile)) {
      wf_listener* listener = parse_line(buffer);
      if (listener != NULL)
	elems.push_back(listener);
    }
  } while (!feof(procfile));

  fclose(procfile);
  return true;
}

#endif /* linux */
