/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <sstream>

#include <stdlib.h>
#if defined(__FreeBSD__)
#include <sys/param.h> /* for htons() */
#else
#include <netinet/in.h> /* for htons() */
#endif
#include <netdb.h> /* for getservbyport */

#include "wfport.h"
#include "wfindent.h"
#include "defs.h"


wf_port::wf_port() :
  _port(0)
{}

wf_port::wf_port(uint16_t port) {
  set(port);
}

wf_port::wf_port(const string& servname, const string& proto) {
  set(servname, proto);
}


bool
wf_port::set(uint16_t port) {
  if (wf_port_check(port) == false)
    return false;
  _port = port;
  return true;
}

bool
wf_port::set(const string& servname, const string& proto) {
  struct servent* serv;

  serv = getservbyname(servname.c_str(), proto.c_str());
  if (serv == NULL)
    return false;

  _port = ntohs(serv->s_port);
  return true;
}

bool
wf_port::set(const string& servname) {
  int num;
  char *p;
  num = strtol(servname.c_str(), &p, 10);
  if (*p == '\0')
    return set(num);
  /* here, it is not a number */
  return set(servname, "tcp") || set(servname, "udp");
}

uint16_t
wf_port::get() const {
  return _port;
}

string
wf_port::tostr(const string& proto) const {
  string str;
  struct servent *serv;

  serv = getservbyport(htons(_port), proto.c_str());
  if (serv == NULL) {
    ostringstream outs;
    outs << _port;
    str = outs.str();
  }
  else
    str = serv->s_name;
  return str;
}

ostream&
wf_port::print(ostream& os) const {
  /* be serious and check if there is an ambiguity between tcp and udp */
  string tcp = tostr("tcp");
  string udp = tostr("udp");
  if (tcp == udp) /* resolvable and same for both, or unresolvable for both */
    os << tcp;
  else {
    if (wf_port_check(_port, "tcp")) /* resolvable for tcp */
      os << tcp;
    else if (wf_port_check(_port, "udp")) /* resolvable for udp */
      os << udp;
    else
      os << _port;
  }
  return os;
}

ostream&
wf_port::output_xml(ostream& os, unsigned int indent_level) const {
  return os << wf_indent(indent_level) << "<port>" << _port << "</port>"
	    << endl;
}


ostream&
operator<<(ostream& os, const wf_port& port) {
  return port.print(os);
}

bool
wf_port_check(uint16_t port) {
  /* return (port > 0 && port < 65536);  is useless because of data type */
  return true;
}

bool
wf_port_check(uint16_t port, const string& proto) {
  return (getservbyport(htons(port), proto.c_str()) != NULL);
}

bool
wf_port_check(const string& servname, const string& proto) {
  return (getservbyname(servname.c_str(), proto.c_str()) != NULL);
}

string
wf_port_toservice(uint16_t port, const string& proto) {
  string str;
  struct servent* servent;

  servent = getservbyport(htons(port), proto.c_str());
  if (servent != NULL)
    str = servent->s_name;
  return str;
}
