/*
 *  libghal
 *
 *  Copyright (c) 2007 Brian Tarricone <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <exo-hal/exo-hal.h>

#include "ghal-volume.h"
#include "ghal-context.h"
#include "ghal-private.h"

struct _GHalVolumePrivate
{
    LibHalVolume *hal_volume;
};

enum
{
    SIG_MOUNTED = 0,
    SIG_UNMOUNTED,
    N_SIGS,
};

static void ghal_volume_class_init(GHalVolumeClass *klass);

static void ghal_volume_init(GHalVolume *instance);
static void ghal_volume_finalize(GObject *obj);

static void ghal_volume_property_changed(GHalDevice *device,
                                         const gchar *property,
                                         GhalPropertyChangeType change_type);

guint ghv_sigs[N_SIGS] = { 0, };


G_DEFINE_TYPE(GHalVolume, ghal_volume, GHAL_TYPE_DEVICE)


static void
ghal_volume_class_init(GHalVolumeClass *klass)
{
    GObjectClass *object_class = (GObjectClass *)klass;
     
    g_type_class_add_private(klass, sizeof(GHalVolumePrivate));
     
    object_class->finalize = ghal_volume_finalize;
    
    ghv_sigs[SIG_MOUNTED] = g_signal_new("mounted",
                                         GHAL_TYPE_VOLUME,
                                         G_SIGNAL_RUN_LAST,
                                         G_STRUCT_OFFSET(GHalVolumeClass,
                                                         mounted),
                                         NULL, NULL,
                                         g_cclosure_marshal_VOID__VOID,
                                         G_TYPE_NONE, 0);
    ghv_sigs[SIG_UNMOUNTED] = g_signal_new("unmounted",
                                           GHAL_TYPE_VOLUME,
                                           G_SIGNAL_RUN_LAST,
                                           G_STRUCT_OFFSET(GHalVolumeClass,
                                                           unmounted),
                                           NULL, NULL,
                                           g_cclosure_marshal_VOID__VOID,
                                           G_TYPE_NONE, 0);
}

static void
ghal_volume_init(GHalVolume *volume)
{
    volume->priv = G_TYPE_INSTANCE_GET_PRIVATE(volume, GHAL_TYPE_VOLUME,
                                               GHalVolumePrivate);
    
    g_signal_connect(G_OBJECT(volume), "property-changed",
                     G_CALLBACK(ghal_volume_property_changed), NULL);
}

static void
ghal_volume_finalize(GObject *obj)
{
    GHalVolume *volume = GHAL_VOLUME(obj);
    
    if(volume->priv->hal_volume)
        libhal_volume_free(volume->priv->hal_volume);
    
    G_OBJECT_CLASS(ghal_volume_parent_class)->finalize(obj);
}



static gboolean
ghal_volume_ensure_hal_volume(GHalVolume *volume)
{
    g_return_val_if_fail(GHAL_IS_VOLUME(volume), FALSE);
    
    if(G_UNLIKELY(!volume->priv->hal_volume)) {
        GHalContext *context = NULL;
        const gchar *udi = NULL;
        LibHalContext *hal_ctx = NULL;
        
        g_return_val_if_fail((context = _ghal_device_peek_context(GHAL_DEVICE(volume)))
                             && (udi = ghal_device_peek_udi(GHAL_DEVICE(volume)))
                             && (hal_ctx = _ghal_context_peek_libhal_context(context)),
                             FALSE);
        
        volume->priv->hal_volume = libhal_volume_from_udi(hal_ctx, udi);
    }
    
    return volume->priv->hal_volume ? TRUE : FALSE;
}

static void
ghal_volume_property_changed(GHalDevice *device,
                             const gchar *property,
                             GhalPropertyChangeType change_type)
{
    GHalVolume *volume = GHAL_VOLUME(device);
    
    /* ditch the old LibHalVolume, as it's no longer valid */
    if(G_LIKELY(volume->priv->hal_volume)) {
        libhal_volume_free(volume->priv->hal_volume);
        volume->priv->hal_volume = NULL;
    }
    
    if(!g_ascii_strcasecmp(property, "volume.is_mounted")) {
        gboolean is_mounted = ghal_device_get_property_bool(device, property,
                                                            NULL);
        
        if(is_mounted)
            g_signal_emit(G_OBJECT(device), ghv_sigs[SIG_MOUNTED], 0);
        else
            g_signal_emit(G_OBJECT(device), ghv_sigs[SIG_UNMOUNTED], 0);
    }
}



LibHalVolume *
_ghal_volume_peek_libhal_volume(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), NULL);
    return volume->priv->hal_volume;
}



guint64
ghal_volume_get_size(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), 0);
    
    return libhal_volume_get_size(volume->priv->hal_volume);
}

const gchar *
ghal_volume_get_fstype(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), NULL);
    
    return libhal_volume_get_fstype(volume->priv->hal_volume);
}

const gchar *
ghal_volume_get_fsversion(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), NULL);
    
    return libhal_volume_get_fsversion(volume->priv->hal_volume);
}

LibHalVolumeUsage
ghal_volume_get_fsusage(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume),
                         LIBHAL_VOLUME_USAGE_UNKNOWN);
    
    return libhal_volume_get_fsusage(volume->priv->hal_volume);
}

gboolean
ghal_volume_is_mounted(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), FALSE);
    
    return libhal_volume_is_mounted(volume->priv->hal_volume);
}

gboolean
ghal_volume_is_mounted_read_only(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), FALSE);
    
    return libhal_volume_is_mounted_read_only(volume->priv->hal_volume);
}

const char *
ghal_volume_get_label(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), NULL);
    
    return libhal_volume_get_label(volume->priv->hal_volume);
}

const char *
ghal_volume_get_mount_point(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), NULL);
    
    return libhal_volume_get_mount_point(volume->priv->hal_volume);
}

const char *
ghal_volume_get_uuid(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), NULL);
    
    return libhal_volume_get_uuid(volume->priv->hal_volume);
}

gboolean
ghal_volume_should_ignore(GHalVolume *volume)
{
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume), TRUE);
    
    return libhal_volume_should_ignore(volume->priv->hal_volume);
}

GHalDevice *
ghal_volume_get_storage_device(GHalVolume *volume)
{
    GHalDevice *device = NULL;
    GHalContext *context = NULL;
    const char *udi;
    
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume)
                         && (context = _ghal_device_peek_context(GHAL_DEVICE(volume))),
                         NULL);
    
    udi = libhal_volume_get_storage_device_udi(volume->priv->hal_volume);
    if(udi)
        device = ghal_context_get_device_for_udi(context, udi);
    
    return device;
}

gchar *
ghal_volume_get_display_name(GHalVolume *volume)
{
    gchar *name;
    GHalContext *context = NULL;
    LibHalContext *hal_ctx = NULL;
    GHalDevice *storage_drive = NULL;
    LibHalDrive *storage_hal_drive = NULL;
    
    exo_hal_init();
    
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume)
                         && (context = _ghal_device_peek_context(GHAL_DEVICE(volume)))
                         && (hal_ctx = _ghal_context_peek_libhal_context(context))
                         && (storage_drive = ghal_volume_get_storage_device(volume))
                         && (storage_hal_drive = _ghal_drive_peek_libhal_drive(GHAL_DRIVE(storage_drive))),
                         NULL);
    
    name = exo_hal_volume_compute_display_name(hal_ctx,
                                               volume->priv->hal_volume,
                                               storage_hal_drive);
    
    g_object_unref(G_OBJECT(storage_drive));
    
    return name;
}

GList *
ghal_volume_get_icon_list(GHalVolume *volume)
{
    GList *icons;
    GHalContext *context = NULL;
    LibHalContext *hal_ctx = NULL;
    GHalDevice *storage_drive = NULL;
    LibHalDrive *storage_hal_drive = NULL;
    
    exo_hal_init();
    
    g_return_val_if_fail(ghal_volume_ensure_hal_volume(volume)
                         && (context = _ghal_device_peek_context(GHAL_DEVICE(volume)))
                         && (hal_ctx = _ghal_context_peek_libhal_context(context))
                         && (storage_drive = ghal_volume_get_storage_device(volume))
                         && (storage_hal_drive = _ghal_drive_peek_libhal_drive(GHAL_DRIVE(storage_drive))),
                         NULL);
    
    icons = exo_hal_volume_compute_icon_list(hal_ctx,
                                             volume->priv->hal_volume,
                                             storage_hal_drive);
    
    g_object_unref(G_OBJECT(storage_drive));
    
    return icons;
}
