/* 
 *  Xfmedia Remote Plugin - Xfmedia Remote Control Plugin for Xfce4 Panel
 *  Copyright (C) 2005  Pasi Orovuo <pasi.ov@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <glib.h>
#include <glib-object.h>

#include <libxfcegui4/libxfcegui4.h>
#include <libxfce4util/libxfce4util.h>

#ifdef HAVE_OLD_PANEL
#include <panel/plugins.h>
#include <panel/xfce.h>
#include <stdlib.h>
#else
#include <libxfce4panel/xfce-panel-plugin.h>
#endif

#include "xfmedia-remote.h"

#define BORDER_WIDTH            8
#define MAX_XFMEDIA_INSTANCES   4
#define LIST_CACHE_TIME         ( 1000 * 2 )
/* Cache instance information for 5 seconds */
#define INSTANCE_CACHE_TIME     ( 1000 * 5 )


#if LIBXFCEGUI4_CHECK_VERSION( 4, 3, 4 ) && XFCE_DISABLE_DEPRECATED
#ifdef XFCE_FILE_CHOOSER
#undef XFCE_FILE_CHOOSER
#endif
#define XFCE_FILE_CHOOSER GTK_FILE_CHOOSER

#define xfce_file_chooser_new gtk_file_chooser_dialog_new
#define xfce_file_chooser_get_filename gtk_file_chooser_get_filename
#define xfce_file_chooser_set_select_multiple gtk_file_chooser_set_select_multiple
#define xfce_file_chooser_get_filenames gtk_file_chooser_get_filenames
#endif

typedef struct _TrackAddList TrackAddList;
typedef struct _TrackPlayInfo TrackPlayInfo;
typedef struct _PlaybackActionInfo PlaybackActionInfo;
typedef struct _XfmediaRemoteInstance XfmediaRemoteInstance;
typedef struct _XfmediaPanelPlugin XfmediaPanelPlugin;

struct _TrackAddList {
    guint           evsrc;
    gint            instance_no;
    GSList          *list;
    GSList          *current;
};

struct _TrackPlayInfo {
    XfmediaPanelPlugin  *plugin;
    guint               index;
    guint               instance;
};

struct _PlaybackActionInfo {
    XfmediaRemoteInstance   *instance;
    gboolean (*remote_cb)( XfmediaRemote *remote );
};

struct _XfmediaRemoteInstance {
    gint            no;
    guint32         atime;
    gchar           *now_playing;
    gint            now_playing_index;
    XfmediaRemote   *remote;
};

struct _XfmediaPanelPlugin {
    GtkWidget       *evbox;
    GtkWidget       *image;
    GdkPixbuf       *pixbuf;

    GtkTooltips     *tooltip;
    GtkWidget       *menu;

    guint32         atime;
    XfmediaRemoteInstance   **remote_list;
    guint           max_instances;
    
    gboolean        show_playlist;

    /* Currently open dialog */
    GtkWidget       *dialog;

    TrackAddList    *add_list;
};

static gboolean
plugin_playback_action_cb( PlaybackActionInfo *info )
{
    info->instance->atime = 0;
    
    if ( !info->remote_cb( info->instance->remote ) ) {
        g_warning( "xfmedia-remote: plugin_track_action_cb failed!" );
    }

    g_free( info );

    return ( FALSE );
}

static gboolean
plugin_play_track_cb( TrackPlayInfo *track )
{
    XfmediaPanelPlugin      *plugin = track->plugin;
    XfmediaRemoteInstance   *instance = (plugin->remote_list)[track->instance];

    if ( instance ) {
        if ( !xfmedia_remote_song_play( instance->remote, track->index ) ) {
            g_warning( "xfmedia_remote_song_play( %d, %d ) failed!", instance->no, track->index );
        }

        instance->atime = 0;
    }

    g_free( track );

    return ( FALSE );
}

static void
plugin_menu_prev_activate_cb( GtkWidget *widget, gpointer user_data )
{
    gint                    inst_no;
    XfmediaPanelPlugin      *plugin = user_data;
    PlaybackActionInfo      *info;

    inst_no = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                                  "xfmedia-instance" ) );
    info            = g_new0( PlaybackActionInfo, 1 );
    info->instance  = (plugin->remote_list)[inst_no];
    info->remote_cb = xfmedia_remote_song_previous;

    g_idle_add( (GSourceFunc) plugin_playback_action_cb, info );
}

static void
plugin_menu_play_activate_cb( GtkWidget *widget, gpointer user_data )
{
    TrackPlayInfo           *info;
    GtkWidget               *parent = gtk_widget_get_parent( widget );

    info = g_new0( TrackPlayInfo, 1 );
    info->plugin        = user_data;
    info->index         = -1;
    info->instance      = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( parent ),
                                                              "xfmedia-instance" ) );

    g_idle_add( (GSourceFunc) plugin_play_track_cb, info );
}

static void
plugin_menu_pause_activate_cb( GtkWidget *widget, gpointer user_data )
{
    gint                    inst_no;
    XfmediaPanelPlugin      *plugin = user_data;
    PlaybackActionInfo      *info;

    inst_no = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                                  "xfmedia-instance" ) );
    info            = g_new0( PlaybackActionInfo, 1 );
    info->instance  = (plugin->remote_list)[inst_no];
    info->remote_cb = xfmedia_remote_song_pause;

    g_idle_add( (GSourceFunc) plugin_playback_action_cb, info );
}

static void
plugin_menu_stop_activate_cb( GtkWidget *widget, gpointer user_data )
{
    gint                    inst_no;
    XfmediaPanelPlugin      *plugin = user_data;
    PlaybackActionInfo      *info;

    inst_no = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                                  "xfmedia-instance" ) );
    info            = g_new0( PlaybackActionInfo, 1 );
    info->instance  = (plugin->remote_list)[inst_no];
    info->remote_cb = xfmedia_remote_song_stop;

    g_idle_add( (GSourceFunc) plugin_playback_action_cb, info );
}

static void
plugin_menu_next_activate_cb( GtkWidget *widget, gpointer user_data )
{
    gint                    inst_no;
    XfmediaPanelPlugin      *plugin = user_data;
    PlaybackActionInfo      *info;

    inst_no = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                                  "xfmedia-instance" ) );
    info            = g_new0( PlaybackActionInfo, 1 );
    info->instance  = (plugin->remote_list)[inst_no];
    info->remote_cb = xfmedia_remote_song_next;

    g_idle_add( (GSourceFunc) plugin_playback_action_cb, info );
}

#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
static void _init_dialog( GtkWidget *dialog, XfmediaPanelPlugin *plugin )
{
    GdkPixbuf               *pb;
    
    plugin->dialog = dialog;
    g_signal_connect( G_OBJECT( dialog ), "destroy",
                      G_CALLBACK( gtk_widget_destroyed ), &(plugin->dialog) );

    if ( ( pb = xfce_themed_icon_load( "xfmedia", 48 ) ) ) {
        gtk_window_set_icon( GTK_WINDOW( dialog ), pb );
        g_object_unref( G_OBJECT( pb ) );
    }
}

static void
plugin_menu_playlist_load_activate_cb( GtkWidget *widget, gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    GtkWidget               *chooser;

    chooser = xfce_file_chooser_new( _( "Choose playlist to load" ),
                                     NULL,
                                     XFCE_FILE_CHOOSER_ACTION_OPEN,
                                     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                     GTK_STOCK_OK, GTK_RESPONSE_OK,
                                     NULL );
    _init_dialog( chooser, plugin );

    if ( gtk_dialog_run( GTK_DIALOG( chooser ) ) == GTK_RESPONSE_OK ) {
        gchar       *fn;

        fn = xfce_file_chooser_get_filename( XFCE_FILE_CHOOSER( chooser ) );
        if ( fn ) {
            gint                    i;
            XfmediaRemoteInstance   *instance;
            
            i = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                                    "xfmedia-instance" ) );
            instance = (plugin->remote_list)[i];

            if ( !xfmedia_remote_playlist_load( instance->remote, fn ) ) {
                g_warning( "xfmedia_remote_playlist_load( %d ) failed!", instance->no );
            }
            g_free( fn );
        }
    }
    gtk_widget_destroy( chooser );
}

static void
plugin_menu_playlist_save_activate_cb( GtkWidget *widget, gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    GtkWidget               *chooser;

    chooser = xfce_file_chooser_new( _( "Save playlist" ),
                                     NULL,
                                     XFCE_FILE_CHOOSER_ACTION_SAVE,
                                     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                     GTK_STOCK_OK, GTK_RESPONSE_OK,
                                     NULL );

    _init_dialog( chooser, plugin );

    if ( gtk_dialog_run( GTK_DIALOG( chooser ) ) == GTK_RESPONSE_OK ) {
        gint                    i;
        XfmediaRemoteInstance   *instance;
        gchar                   *fn;

        fn = xfce_file_chooser_get_filename( XFCE_FILE_CHOOSER( chooser ) );
        if ( fn ) {
            i = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_toplevel( widget ) ),
                                                    "xfmedia-instance" ) );
            instance = (plugin->remote_list)[i];

            if ( !xfmedia_remote_playlist_save( instance->remote, fn ) ) {
                xfce_err( _( "Failed to save playlist" ) );
            }
            g_free( fn );
        }
    }
    gtk_widget_destroy( chooser );
}

static void
plugin_menu_playlist_clear_activate_cb( GtkWidget *widget, gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    gint                    i;
    XfmediaRemoteInstance   *instance;

    i = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                            "xfmedia-instance" ) );

    instance = (plugin->remote_list)[i];

    if ( !xfmedia_remote_playlist_clear( instance->remote ) ) {
        g_warning( "xfmedia_remote_playlist_clear( %d ) failed!", instance->no );
    }
}

static gboolean
plugin_add_track_cb( gpointer data )
{
    XfmediaPanelPlugin  *plugin = data;
    guint               no;
    
    g_assert( plugin->add_list );

    no = plugin->add_list->instance_no;
    if ( (plugin->remote_list)[no] ) {
        XfmediaRemoteInstance   *instance = (plugin->remote_list)[no];

        if ( !xfmedia_remote_playlist_add_file( instance->remote,
                    (gchar *) plugin->add_list->current->data, -1 ) ) {
            g_warning( "xfmedia_remote_playlist_add_file( %d ) failed!", instance->no );
        }

        g_free( plugin->add_list->current->data );
        plugin->add_list->current = plugin->add_list->current->next;
    }
    else {
        while ( plugin->add_list->current ) {
            g_free( plugin->add_list->current->data );
            plugin->add_list->current = plugin->add_list->current->next;
        }
    }

    if ( !plugin->add_list->current ) {
        g_slist_free( plugin->add_list->list );
        g_free( plugin->add_list );
        plugin->add_list = NULL;

        return ( FALSE );
    }
    return ( TRUE );
}
        
static void
plugin_menu_playlist_track_add_activate_cb( GtkWidget *widget, gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    GtkWidget               *chooser;

    if ( plugin->add_list ) {
        xfce_info( _( "Please wait until the previous add operation completes." ) );
        return;
    }

    chooser = xfce_file_chooser_new( _( "Add track(s) to playlist" ),
                                     NULL,
                                     XFCE_FILE_CHOOSER_ACTION_OPEN,
                                     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                     GTK_STOCK_OK, GTK_RESPONSE_OK,
                                     NULL );

    _init_dialog( chooser, plugin );

    xfce_file_chooser_set_select_multiple( XFCE_FILE_CHOOSER( chooser ), TRUE );

    if ( gtk_dialog_run( GTK_DIALOG( chooser ) ) == GTK_RESPONSE_OK ) {
        GSList      *list;

        list = xfce_file_chooser_get_filenames( XFCE_FILE_CHOOSER( chooser ) );
        if ( list ) {
            gint                    i;
            TrackAddList            *add_list;
            
            i = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                                    "xfmedia-instance" ) );
            
            plugin->add_list = add_list = g_new0( TrackAddList, 1 );
            add_list->instance_no   = i;
            add_list->list          = list;
            add_list->current       = list;
            add_list->evsrc         = g_idle_add( plugin_add_track_cb, plugin );
        }
    }
    gtk_widget_destroy( chooser );
}

static void
plugin_menu_playlist_track_play_activate_cb( GtkWidget *widget,
                                             gpointer user_data )
{
    TrackPlayInfo           *info;

    info = g_new( TrackPlayInfo, 1 );
    info->plugin    = user_data;
    info->index     = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( widget->parent ),
                                                          "track-index" ) );
    info->instance  = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( widget->parent ),
                                                          "xfmedia-instance" ) );

    g_idle_add( (GSourceFunc) plugin_play_track_cb, info );
}

static void
plugin_menu_playlist_track_remove_activate_cb( GtkWidget *widget,
                                               gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    gint                    i, track;
    XfmediaRemoteInstance   *instance;

    i = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( widget->parent ),
                                            "xfmedia-instance" ) );
    track = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( widget->parent ),
                                                "track-index" ) );
    instance = (plugin->remote_list)[i];

    if ( !xfmedia_remote_playlist_remove_file( instance->remote, track ) ) {
        g_warning( "xfmedia_remote_playlist_remove_file( %d, %d ) failed!", instance->no, track );
    }

}

static void
plugin_menu_playlist_show_cb( GtkWidget *playlist_menu, gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    gchar                   **playlist = NULL;
    guint                   ntracks, i;
    guint                   instance_no;
    XfmediaRemoteInstance   *instance;

    if ( GPOINTER_TO_INT( g_object_get_data( G_OBJECT( playlist_menu ), "playlist" ) ) ) {
        /* Playlist already filled */
        DBG( "Playlist menu already built" );
        return;
    }

    instance_no = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( playlist_menu ),
                                                      "xfmedia-instance" ) );

    g_assert( (plugin->remote_list)[instance_no] != NULL );
    
    instance = (plugin->remote_list)[instance_no];

    if ( !xfmedia_remote_playlist_get( instance->remote, &playlist, &ntracks ) ) {
        DBG( "Failed to get playlist from Xfmedia instance %d", instance->no );
        return;
    }

    if ( ntracks ) {
        GtkWidget       *track_menu;
        GtkWidget       *item;
        GtkWidget       *image;
        gchar           *trackitem;

        /* Separator */
        item = gtk_separator_menu_item_new();
        gtk_widget_show( item );
        gtk_menu_shell_append( GTK_MENU_SHELL( playlist_menu ), item );

        for ( i = 0; i < ntracks; i++ ) {
            /* Song name to playlist menu */
            trackitem = g_strdup_printf( "%.2d: %s", i + 1, playlist[i] );

            item = gtk_menu_item_new_with_label( trackitem );

            if ( i == instance->now_playing_index ) {
                gchar   *markup = g_markup_printf_escaped( "<b>%s</b>", trackitem );
             
                gtk_label_set_markup( GTK_LABEL( GTK_BIN( item )->child ), markup );

                g_free( markup );
            }

            gtk_widget_show( item );
            gtk_menu_shell_append( GTK_MENU_SHELL( playlist_menu ), item );

            g_free( trackitem );

            track_menu = gtk_menu_new();
            g_object_set_data( G_OBJECT( track_menu ),
                               "xfmedia-instance",
                               GINT_TO_POINTER( instance->no ) );
            g_object_set_data( G_OBJECT( track_menu ),
                               "track-index",
                               GINT_TO_POINTER( i ) );
            gtk_menu_item_set_submenu( GTK_MENU_ITEM( item ), track_menu );

            /* Play */
            item = gtk_image_menu_item_new_with_mnemonic( _( "_Play" ) );
            g_signal_connect( G_OBJECT( item ), "activate",
                    G_CALLBACK( plugin_menu_playlist_track_play_activate_cb ),
                    plugin );
            gtk_widget_show( item );
            gtk_menu_shell_append( GTK_MENU_SHELL( track_menu ), item );

            image = gtk_image_new_from_stock( GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_MENU );
            gtk_widget_show( image );
            gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

            /* Remove */
            item = gtk_image_menu_item_new_with_mnemonic( _( "_Remove" ) );
            g_signal_connect( G_OBJECT( item ),
                              "activate",
                              G_CALLBACK( plugin_menu_playlist_track_remove_activate_cb ),
                              plugin );
            gtk_widget_show( item );
            gtk_menu_shell_append( GTK_MENU_SHELL( track_menu ), item );

            image = gtk_image_new_from_stock( GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU );
            gtk_widget_show( image );
            gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );
        }
    }

    if ( playlist ) {
        g_strfreev( playlist );
    }

    g_object_set_data( G_OBJECT( playlist_menu ), "playlist", GINT_TO_POINTER( 1 ) );
}
#endif /* !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API ) */

static void
plugin_menu_quit_activate_cb( GtkWidget *widget,
                              gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    gint                    i;
    XfmediaRemoteInstance   *instance;

    i = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( gtk_widget_get_parent( widget ) ),
                                            "xfmedia-instance" ) );

    instance = (plugin->remote_list)[i];
    instance->atime = 0;

    if ( !xfmedia_remote_quit( instance->remote ) ) {
        g_warning( "xfmedia_remote_quit( %d ) failed!", instance->no );
    }
}

static void
plugin_menu_start_xfmedia_activate_cb( GtkWidget *widget,
                                       gpointer user_data )
{
    GError      *error = NULL;
    
    if ( !xfce_exec( "xfmedia", FALSE, TRUE, &error ) ) {
        xfce_err( _( "Failed to execute xfmedia: %s" ), error->message );
    }
}

static GtkWidget *
plugin_instance_menu_build( XfmediaPanelPlugin *plugin,
                            XfmediaRemoteInstance *instance )
{
    GtkWidget       *instance_menu;
#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
    GtkWidget       *playlist_menu;
#endif /* !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API ) */
    GtkWidget       *item;
    GtkWidget       *image;

    instance_menu = gtk_menu_new();
    g_object_set_data( G_OBJECT( instance_menu ),
                       "xfmedia-instance",
                       GINT_TO_POINTER( instance->no ) );
    
    /* Prev */
    item = gtk_image_menu_item_new_with_mnemonic( _( "P_rev" ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_prev_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_MEDIA_PREVIOUS, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Play */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Play" ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_play_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Pause */
    item = gtk_image_menu_item_new_with_mnemonic( _( "P_ause" ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_pause_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Stop */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Stop" ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_stop_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_MEDIA_STOP, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Next */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Next" ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_next_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_MEDIA_NEXT, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Separator */
    item = gtk_separator_menu_item_new();
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );
    
#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
    /* Playlist */
    item = gtk_menu_item_new_with_mnemonic( _( "P_laylist" ) );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );

    if ( plugin->show_playlist ) {
        /* Playlist menu */
        playlist_menu = gtk_menu_new();
        
        g_object_set_data( G_OBJECT( playlist_menu ),
                           "xfmedia-instance",
                           GINT_TO_POINTER( instance->no ) );
        g_object_set_data( G_OBJECT( playlist_menu ),
                           "playlist",
                           GINT_TO_POINTER( 0 ) );
        
        g_signal_connect( G_OBJECT( playlist_menu ), "show",
                          G_CALLBACK( plugin_menu_playlist_show_cb ), plugin );

        gtk_menu_item_set_submenu( GTK_MENU_ITEM( item ), playlist_menu );
    }
    else {
        gtk_label_set_text( GTK_LABEL( GTK_BIN( item )->child ), _( "Playlist" ) );
        gtk_widget_set_sensitive( item, FALSE );

        playlist_menu = instance_menu;
    }

    /* Load Playlist */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Load..." ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_playlist_load_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( playlist_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_OPEN, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Save Playlist */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Save..." ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_playlist_save_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( playlist_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_SAVE, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Clear Playlist */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Clear" ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_playlist_clear_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( playlist_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_CLEAR, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Add Track */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Add track(s)..." ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_playlist_track_add_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( playlist_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_ADD, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    /* Separator */
    item = gtk_separator_menu_item_new();
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );
#endif /* !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API ) */
    
    /* Quit */
    item = gtk_image_menu_item_new_with_mnemonic( _( "_Quit" ) );
    g_signal_connect( G_OBJECT( item ), "activate",
                      G_CALLBACK( plugin_menu_quit_activate_cb ), plugin );
    gtk_widget_show( item );
    gtk_menu_shell_append( GTK_MENU_SHELL( instance_menu ), item );

    image = gtk_image_new_from_stock( GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU );
    gtk_widget_show( image );
    gtk_image_menu_item_set_image( GTK_IMAGE_MENU_ITEM( item ), image );

    return ( instance_menu );
}

static void
plugin_popup_menu( XfmediaPanelPlugin *plugin, guint32 time, gint button )
{
    GtkWidget       *item, *instance_menu;
    int             i, count;
    GList           *list = NULL, *li = NULL;

    if ( plugin->menu ) {
        /* TODO:
         * Cache the menu somehow?
         * FIXME:
         * This causes a delay in menu popup if we have a really
         * big playlist (+1000 entries)
         */
        gtk_widget_destroy( plugin->menu );
        plugin->menu = NULL;
    }

    for ( i = 0, count = 0; i < plugin->max_instances; i++ ) {
        if ( (plugin->remote_list)[i] != NULL ) {
                        
            count++;

            instance_menu = plugin_instance_menu_build( plugin, (plugin->remote_list)[i] );
            
            if ( instance_menu ) {
                list = g_list_prepend( list, instance_menu );
            }
        }
    }
    list = g_list_reverse( list );

    if ( count > 1 ) {
        gchar       *str;
        
        plugin->menu = gtk_menu_new();
        
        for ( i = 0, li = list; li != NULL; li = li->next, i++ ) {
            instance_menu = li->data;
            
            str = g_strdup_printf( "Xfmedia #%d", i + 1 );
            
            item = gtk_menu_item_new_with_label( str );
            gtk_widget_show( item );
            gtk_menu_shell_append( GTK_MENU_SHELL( plugin->menu ), item );

            gtk_menu_item_set_submenu( GTK_MENU_ITEM( item ), instance_menu );

            g_free( str );
        }
    }
    else if ( count == 1 ) {
        plugin->menu = list->data;
    }
    else {
        plugin->menu = gtk_menu_new();

        /* Start Xfmedia */
        item = gtk_menu_item_new_with_mnemonic( _( "Start _Xfmedia" ) );
        g_signal_connect( G_OBJECT( item ), "activate",
                          G_CALLBACK( plugin_menu_start_xfmedia_activate_cb ), plugin );
        gtk_widget_show( item );
        gtk_menu_shell_prepend( GTK_MENU_SHELL( plugin->menu ), item );
    }

    if ( list ) {
        g_list_free( list );
    }

    /* Separator */
    item = gtk_separator_menu_item_new();
    gtk_widget_show( item );
    gtk_menu_shell_prepend( GTK_MENU_SHELL( plugin->menu ), item );

    /* Header */
    item = gtk_menu_item_new_with_label( _( "Xfmedia Remote" ) );
    gtk_widget_show( item );
    gtk_widget_set_sensitive( item, FALSE );
    gtk_menu_shell_prepend( GTK_MENU_SHELL( plugin->menu ), item );

    gtk_menu_popup( GTK_MENU( plugin->menu ), NULL, NULL, NULL, NULL, button, time );
    g_signal_connect( G_OBJECT( plugin->menu ), "destroy",
                      G_CALLBACK( gtk_widget_destroyed ), &(plugin->menu) );
}

static gboolean
plugin_button_press_cb( GtkWidget *widget, GdkEventButton *button, gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;

    if ( button->type == GDK_BUTTON_PRESS && button->button == 1 ) {
        if ( plugin->dialog ) {
            gtk_window_present( GTK_WINDOW( plugin->dialog ) );
        }
        else {
            plugin_popup_menu( plugin, button->time, button->button );
        }
        return ( TRUE );
    }

    return ( FALSE );
}

static gboolean
plugin_mouse_enter_cb( GtkWidget *widget, GdkEventCrossing *event,
                               gpointer user_data )
{
    XfmediaPanelPlugin      *plugin = user_data;
    guint                   i;
    GString                 *string = NULL;

    if ( event->time - plugin->atime < LIST_CACHE_TIME ) {
        return ( FALSE );
    }
    plugin->atime = event->time;

    for ( i = 0; i < plugin->max_instances; i++ ) {
        XfmediaRemoteInstance   *inst = (plugin->remote_list)[i];

retry:
        
        if ( !inst ) {
            XfmediaRemote   *remote = xfmedia_remote_new( i );

            DBG( "No instance %d", i );

            if ( remote ) {
                inst = g_new0( XfmediaRemoteInstance, 1 );

                inst->remote    = remote;
                inst->no        = i;
            }
        }

        if ( inst ) {
            /* Cache update delay sorta solved in
             * (prev|play|stop|next) handlers...
             */
            if ( event->time - inst->atime > INSTANCE_CACHE_TIME ) {
                gchar       *title;
                gint        index;
                guint       length;

                DBG( "Instance %d expired", i );

                if ( inst->now_playing ) {
                    g_free( inst->now_playing );
                    inst->now_playing = NULL;
                }

                if ( xfmedia_remote_now_playing( inst->remote, &index, &title, &length ) ) {
                    DBG( "Instance %d playing", i );

                    inst->atime = event->time;

                    if ( !(plugin->remote_list)[i] ) {
                        (plugin->remote_list)[i] = inst;
                    }
                    /*
                    else {
                        g_assert( (plugin->remote_list)[i] == inst );
                    }
                    */

                    if ( index >= 0 ) {
                        inst->now_playing = g_strdup_printf( "#%u: %s (%.2u:%.2u)",
                                                             index + 1,
                                                             title,
                                                             length / 60,
                                                             length % 60 );

                        g_free( title );
                    }
                    inst->now_playing_index = index;
                }
                else {
                    DBG( "now_playing error, removing %d", i );

                    xfmedia_remote_free( inst->remote );

                    g_free( inst );
                    
                    if ( (plugin->remote_list)[i] ) {
                        /* inst has been free'd, but we're just comparing the pointer */
                        /*
                        g_assert( (plugin->remote_list)[i] == inst );
                        */
                        (plugin->remote_list)[i] = NULL;
                        inst = NULL;
                        goto retry;
                    }
                }
            }
        }
    }

    for ( i = 0; i < plugin->max_instances; i++ ) {
        if ( (plugin->remote_list)[i] ) {
            gchar       *s = (plugin->remote_list)[i]->now_playing ?
                             (plugin->remote_list)[i]->now_playing :
                             _( "Stopped" );
            
            if ( !string ) {
                string = g_string_new( s );
            }
            else {
                g_string_append( string, "\n" );
                g_string_append( string, s );
            }
        }
    }

    if ( string ) {
        gtk_tooltips_set_tip( plugin->tooltip, plugin->evbox, string->str, NULL );
        g_string_free( string, TRUE );
    }
    else {
        gtk_tooltips_set_tip( plugin->tooltip, plugin->evbox,
                              _( "No Xfmedia instances running" ), NULL );
    }

    return ( FALSE );
}

static void
xfmedia_plugin_set_max_instances( XfmediaPanelPlugin *plugin,
                                  guint max_instances )
{
    int                     i;
    XfmediaRemoteInstance   **list;
    int                     min, max;

    if ( !max_instances ) {
        max_instances = MAX_XFMEDIA_INSTANCES;
    }
    
    if ( plugin->remote_list &&
         plugin->max_instances == max_instances ) {
        return;
    }
    
    list = g_new0( XfmediaRemoteInstance *, max_instances );

    min = MIN( plugin->max_instances, max_instances );
    max = MAX( plugin->max_instances, max_instances );

    for ( i = 0; i < max; i++ ) {
        if ( i < min ) {
            list[i] = (plugin->remote_list)[i];
        }
        else if ( i >= min ) {
            if ( plugin->max_instances != max ) {
                /* Pointless to continue looping */
                break;
            }

            if ( (plugin->remote_list)[i] ) {
                XfmediaRemoteInstance   *inst = (plugin->remote_list)[i];

                if ( inst->now_playing ) {
                    g_free( inst->now_playing );
                }
                xfmedia_remote_free( inst->remote );
                g_free( inst );
            }
        }
    }

    g_free( plugin->remote_list );

    plugin->remote_list     = list;
    plugin->max_instances   = max_instances;
}

static XfmediaPanelPlugin *
xfmedia_plugin_create( GtkContainer *container )
{
    XfmediaPanelPlugin      *plugin;

    plugin                  = g_new0( XfmediaPanelPlugin, 1 );

    plugin->evbox           = gtk_event_box_new();
    gtk_widget_show( plugin->evbox );
    gtk_container_add( container, plugin->evbox );

    plugin->image           = xfce_scaled_image_new();
    gtk_widget_show( plugin->image );
    gtk_container_add( GTK_CONTAINER( plugin->evbox ), plugin->image );

    plugin->tooltip         = gtk_tooltips_new();

    plugin->max_instances   = MAX_XFMEDIA_INSTANCES;
    plugin->remote_list     = g_new0( XfmediaRemoteInstance *, MAX_XFMEDIA_INSTANCES );
    plugin->show_playlist   = TRUE;

    g_signal_connect( G_OBJECT( plugin->evbox ), "button-press-event",
                      G_CALLBACK( plugin_button_press_cb ), plugin );
    g_signal_connect( G_OBJECT( plugin->evbox ), "enter-notify-event",
                      G_CALLBACK( plugin_mouse_enter_cb ), plugin );
    
    return ( plugin );
}

void xfmedia_plugin_destroy( XfmediaPanelPlugin *plugin )
{
    int         i;

    if ( plugin->add_list ) {
        g_source_remove( plugin->add_list->evsrc );

        while ( plugin->add_list->current ) {
            g_free( plugin->add_list->current->data );
            plugin->add_list->current = plugin->add_list->current->next;
        }
        g_slist_free( plugin->add_list->list );
        g_free( plugin->add_list );
    }

    if ( plugin->pixbuf ) {
        g_object_unref( plugin->pixbuf );
    }

    for ( i = 0; i < plugin->max_instances; i++ ) {
        XfmediaRemoteInstance   *instance = (plugin->remote_list)[i];

        if ( instance ) {
            if ( instance->now_playing ) {
                g_free( instance->now_playing );
            }
            xfmedia_remote_free( instance->remote );

            g_free( instance );
        }
    }

    gtk_object_sink( GTK_OBJECT( plugin->tooltip ) );

    xfmedia_remote_disconnect();

    g_free( plugin );
}

static void
xfmedia_plugin_set_size( XfmediaPanelPlugin *plugin, gint size )
{
    gint        sz = size - BORDER_WIDTH;
    
    if ( plugin->pixbuf ) {
        g_object_unref( G_OBJECT( plugin->pixbuf ) );
    }

    plugin->pixbuf = xfce_themed_icon_load( "xfmedia", sz );
    xfce_scaled_image_set_from_pixbuf( XFCE_SCALED_IMAGE( plugin->image ),
                                       plugin->pixbuf );
    
    gtk_widget_set_size_request( plugin->evbox, size, size );
}

#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
static void
xfmedia_plugin_menu_show_playlist_toggled_cb( GtkWidget *widget,
                                              XfmediaPanelPlugin *plugin )
{
    plugin->show_playlist =
        gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
}

static void
xfmedia_plugin_create_options( XfmediaPanelPlugin *plugin, GtkWidget *dlgvbox )
{
    GtkWidget           *frame, *frame_bin;
    GtkWidget           *checkbox;
    GtkWidget           *vbox;
    
#if LIBXFCEGUI4_CHECK_VERSION( 4, 3, 4 )
    frame = xfce_create_framebox( _( "Popup menu" ), &frame_bin );
#else
    frame = xfce_framebox_new( _( "Popup menu" ), TRUE );
    frame_bin = XFCE_FRAMEBOX( frame )->hbox;
#endif
    gtk_widget_set_size_request( frame, 300, 120 );
    gtk_frame_set_shadow_type( GTK_FRAME( frame ), GTK_SHADOW_NONE );
    gtk_widget_show( frame );
    gtk_box_pack_start( GTK_BOX( dlgvbox ), frame, FALSE, TRUE, 0 );

    vbox = gtk_vbox_new( FALSE, BORDER_WIDTH );
    gtk_widget_show( vbox );
    gtk_container_add( GTK_CONTAINER( frame_bin ), vbox );
    
    checkbox = gtk_check_button_new_with_mnemonic( _( "Show _playlist" ) );
    g_signal_connect( G_OBJECT( checkbox ), "toggled",
                      G_CALLBACK( xfmedia_plugin_menu_show_playlist_toggled_cb ),
                      plugin );
    gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( checkbox ),
                                  plugin->show_playlist );
    gtk_widget_show( checkbox );
    gtk_box_pack_start( GTK_BOX( vbox ), checkbox, FALSE, FALSE, 0 );
}
#endif /* !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API ) */

#ifdef HAVE_OLD_PANEL

#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
static void
xfmedia_plugin_configure( Control *control, GtkContainer *container,
                          GtkWidget *done )
{
    GtkWidget       *vbox;

    vbox = gtk_vbox_new( FALSE, BORDER_WIDTH );
    gtk_widget_show( vbox );
    gtk_container_add( container, vbox );

    xfmedia_plugin_create_options( control->data, vbox );
}
#endif /* !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API ) */

static void
xfmedia_plugin_read_config( Control *control, xmlNodePtr node )
{
    XfmediaPanelPlugin      *plugin = control->data;
    xmlChar                 *value;

    value = xmlGetProp( node, (const xmlChar *) "show_playlist" );
    if ( value ) {
        plugin->show_playlist = ( *value == '1' ? TRUE : FALSE );
        xmlFree( value );
    }

    value = xmlGetProp( node, (const xmlChar *) "max_instances" );
    if ( value ) {
        guint       max_instances;
        
        max_instances = atoi( value );
        xmlFree( value );

        xfmedia_plugin_set_max_instances( plugin, max_instances );
    }
}

static void
xfmedia_plugin_write_config( Control *control, xmlNodePtr node )
{
    XfmediaPanelPlugin      *plugin = control->data;
    gchar                   buf[64];

    xmlSetProp( node, (const xmlChar *) "show_playlist",
                plugin->show_playlist ? "1" : "0" );
    g_snprintf( buf, sizeof( buf ) / sizeof( gchar ), "%u", plugin->max_instances );
    xmlSetProp( node, (const xmlChar *) "max_instances", buf );
}

static void
xfmedia_plugin_size_changed( Control *ctrl, int size )
{
    XfmediaPanelPlugin  *plugin = ctrl->data;

    xfmedia_plugin_set_size( plugin, icon_size[size] + border_width );
}

static void
xfmedia_plugin_attach_callback( Control *control, const gchar *signal,
                                GCallback callback, gpointer data )
{
    XfmediaPanelPlugin  *plugin = (XfmediaPanelPlugin *) control->data;

    g_signal_connect( G_OBJECT( plugin->evbox ), signal, callback, data );
}

static void
xfmedia_plugin_free( Control *ctrl )
{
    XfmediaPanelPlugin      *plugin = ctrl->data;

    xfmedia_plugin_destroy( plugin );

    ctrl->data = NULL;
}

static gboolean
xfmedia_plugin_new( Control *control )
{
    XfmediaPanelPlugin  *plugin;

    xfce_textdomain( GETTEXT_PACKAGE, LOCALEDIR, "UTF-8" );

    plugin = xfmedia_plugin_create( GTK_CONTAINER( control->base ) );

    control->data = plugin;
    
    return ( TRUE );
}

G_MODULE_EXPORT void
xfce_control_class_init( ControlClass *cc )
{
    xfce_textdomain( GETTEXT_PACKAGE, LOCALEDIR, "UTF-8" );

    cc->name            = "xfmedia-remote-plugin";
    cc->caption         = _( "Xfmedia Remote" );
    cc->create_control  = (CreateControlFunc) xfmedia_plugin_new;
    cc->free            = xfmedia_plugin_free;
    cc->read_config     = xfmedia_plugin_read_config;
    cc->write_config    = xfmedia_plugin_write_config;
#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
    cc->create_options  = xfmedia_plugin_configure;
#endif 
    cc->attach_callback = xfmedia_plugin_attach_callback;
    cc->set_size        = xfmedia_plugin_size_changed;
    cc->set_orientation = NULL;
}

XFCE_PLUGIN_CHECK_INIT

#else /* HAVE_OLD_PANEL */

#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
static void
xfmedia_plugin_dialog_response_cb( GtkWidget *dialog,
                                   gint response,
                                   XfcePanelPlugin *panel_plugin )
{
    gtk_widget_destroy( dialog );
    xfce_panel_plugin_unblock_menu( panel_plugin );
}

static void
xfmedia_plugin_configure( XfcePanelPlugin *panel_plugin,
                          XfmediaPanelPlugin *xfmedia_plugin )
{
    GtkWidget           *dialog;
    GtkWidget           *header;
    GtkWidget           *vbox;

    xfce_panel_plugin_block_menu( panel_plugin );
 
    dialog = gtk_dialog_new_with_buttons( _( "Xfmedia Remote Properties" ),
                                          GTK_WINDOW( gtk_widget_get_toplevel(
                                                  GTK_WIDGET( panel_plugin ) ) ),
                                          GTK_DIALOG_DESTROY_WITH_PARENT |
                                          GTK_DIALOG_NO_SEPARATOR,
                                          GTK_STOCK_CLOSE, GTK_RESPONSE_OK,
                                          NULL );
    xfmedia_plugin->dialog = dialog;
    g_signal_connect( G_OBJECT( dialog ), "destroy",
                      G_CALLBACK( gtk_widget_destroyed ), &(xfmedia_plugin->dialog) );
    g_signal_connect( G_OBJECT( dialog ), "response",
                      G_CALLBACK( xfmedia_plugin_dialog_response_cb ), panel_plugin );
    gtk_container_set_border_width( GTK_CONTAINER( dialog ), 2 );

    header = xfce_create_header( NULL, _( "Xfmedia Remote" ) );
    gtk_widget_set_size_request( GTK_BIN( header )->child, -1, 32 );
    gtk_container_set_border_width( GTK_CONTAINER( header ), BORDER_WIDTH - 2 );
    gtk_widget_show( header );
    gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ), header,
                                 FALSE, TRUE, 0 );
    
    vbox = gtk_vbox_new( FALSE, BORDER_WIDTH );
    gtk_widget_show( vbox );
    gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ), vbox, TRUE, TRUE, 0 );
    
    xfmedia_plugin_create_options( xfmedia_plugin, vbox );
    
    gtk_widget_show( dialog );
}
#endif /* !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API ) */

static void
xfmedia_plugin_read_config( XfcePanelPlugin *panel_plugin,
                            XfmediaPanelPlugin *xfmedia_plugin )
{
    gchar       *fn;
    XfceRc      *rc;
    guint       max_instances;

    fn = xfce_panel_plugin_lookup_rc_file( panel_plugin );
    if ( !fn ) {
        return;
    }

    rc = xfce_rc_simple_open( fn, TRUE );
    g_free( fn );

    if ( !rc ) {
        return;
    }

    xfmedia_plugin->show_playlist = xfce_rc_read_bool_entry( rc, "show_playlist", TRUE );
    max_instances = xfce_rc_read_int_entry( rc, "max_instances", MAX_XFMEDIA_INSTANCES );
    xfmedia_plugin_set_max_instances( xfmedia_plugin, max_instances );

    xfce_rc_close( rc );
}   

static void
xfmedia_plugin_write_config( XfcePanelPlugin *panel_plugin,
                             XfmediaPanelPlugin *xfmedia_plugin )
{
    gchar           *fn;
    XfceRc          *rc;

    fn = xfce_panel_plugin_save_location( panel_plugin, TRUE );
    if ( !fn ) {
        return;
    }

    rc = xfce_rc_simple_open( fn, FALSE );
    g_free( fn );

    if ( !rc ) {
        return;
    }

    xfce_rc_write_bool_entry( rc, "show_playlist", xfmedia_plugin->show_playlist );
    xfce_rc_write_int_entry( rc, "max_instances", xfmedia_plugin->max_instances );

    xfce_rc_close( rc );
}

static gboolean
xfmedia_plugin_size_changed( XfcePanelPlugin *panel_plugin,
                                gint size,
                                XfmediaPanelPlugin *xfmedia_plugin )
{
    xfmedia_plugin_set_size( xfmedia_plugin, size );
    
    return ( TRUE );
}

static void
xfmedia_plugin_free( XfcePanelPlugin *panel_plugin,
                        XfmediaPanelPlugin *xfmedia_plugin )
{
    xfmedia_plugin_destroy( xfmedia_plugin );
}

static void
xfmedia_plugin_new( XfcePanelPlugin *panel_plugin )
{
    XfmediaPanelPlugin      *xfmedia;

    xfce_textdomain( GETTEXT_PACKAGE, LOCALEDIR, "UTF-8" );

    xfmedia = xfmedia_plugin_create( GTK_CONTAINER( panel_plugin ) );
    xfmedia_plugin_read_config( panel_plugin, xfmedia );
    
    xfce_panel_plugin_add_action_widget( panel_plugin, xfmedia->evbox );

    g_signal_connect( G_OBJECT( panel_plugin ), "free-data",
                      G_CALLBACK( xfmedia_plugin_free ), xfmedia );
    
    g_signal_connect( G_OBJECT( panel_plugin ), "size-changed",
                      G_CALLBACK( xfmedia_plugin_size_changed ), xfmedia );
    
    g_signal_connect( G_OBJECT( panel_plugin ), "save",
                      G_CALLBACK( xfmedia_plugin_write_config ), xfmedia );
    
#if !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API )
    xfce_panel_plugin_menu_show_configure( panel_plugin );
    g_signal_connect( G_OBJECT( panel_plugin ), "configure-plugin",
                      G_CALLBACK( xfmedia_plugin_configure ), xfmedia );
#endif /* !defined( XFMEDIA_TOO_OLD ) || defined( DBUS_USE_OLD_API ) */
}

XFCE_PANEL_PLUGIN_REGISTER_EXTERNAL( xfmedia_plugin_new );

#endif /* HAVE_OLD_PANEL */

