/*
  Copyright (C) 2000-2007

  Code contributed by Greg Collecutt, Joseph Hope and the xmds-devel team

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_simulation.h 1577 2007-11-11 20:19:36Z paultcochrane $
*/

/*! @file xmds_simulation.h
  @brief 

  More detailed explanation...
*/

#ifndef XMDS_SIMULATION_H
#define XMDS_SIMULATION_H

#include <xmds_element.h>
#include <xmds_argv.h>
#include <xmds_field.h>
#include <xmds_output.h>
#include <xmds_sequence.h>
#include <xmds_globals.h>

// *****************************************************************************
// *****************************************************************************
//                              xmdsSimulation
// *****************************************************************************
// *****************************************************************************

//! Class to organise a simulation in xmds
class xmdsSimulation : public xmdsElement {

  public :

    //! Structure of parameters used in an xmds simulation
    struct simulationParametersStruct {
      XMLString rawFileName;    //!< Raw simulation file name
      XMLString simulationName; //!< Simulation name
      XMLString propDimName;    //!< Propagation dimension name
      XMLString authorName;     //!< Name of the author the xmds script
      XMLString description;    //!< Description of what the script is supposed to do
      long nThreads;            //!< Number of threads
      bool stochastic;          //!< Is this a stochastic simulation?
      XMLString nPaths;         //!< Number of paths
      bool mpiAvailable;        //!< Is mpi available?
      bool usempi;              //!< Use mpi?
      bool bing;                //!< Play sound when simulation is finished?
      XMLString seed[2];        //!< Random number seed
      long nNoises;             //!< Number of noise terms
      XMLString noiseKind;      //!< The kind of noise to be generated
      XMLString mpiMethod;      //!< The method of dividing paths amongst nodes
      double noiseMean;         //!< The mean of the noise distribution; only used for Poissonian distributions
      bool errorCheck;          //!< Should I error check?
      bool useWisdom;           //!< Should wisdom be used?
      bool usePrefs;            //!< Should the values in the preferences file be used?
      bool binaryOutput;        //!< Is binary output required?
      bool useDouble;           //!< Should double precision be used?
      bool benchmark;           //!< Time the main loop?
      long fftwVersion;         //!< Which version of fftw shall we use?
      bool useOpenMP;           //!< Shall we use OpenMP for parallelisation?
      bool useIntelMKL;         //!< Use Intel MKL libraries for faster random numbers
      double runTimeLimit;      //!< Maximum running time for integrate elements
      XMLString version;        //!< The current xmds version number
      XMLString revision;       //!< The subversion revision of xmds
    };

  //! Structure of arguments on command line to simulation
  struct argvStruct {
    mutable list<string> nameList;             //!< List storing the argument names
    mutable list<string> typeList;             //!< List storing the argument types
    mutable list<string> defaultValueList;     //!< List storing the argument default values
    mutable list<string> shortOptionList;      //!< List storing the short option names
    mutable list<string> typeConversionList;   //!< List storing how to perform the type conversion
  };

  //! Constructor of xmdsSimulation object
  xmdsSimulation(
                 const char *const yourRawFileName,
                 const bool& yourVerboseMode,
                 const bool& mpiAvailable);

  //! Destructor
  ~xmdsSimulation();

  //! Process element within xmds simulation
  void processElement(
                      const Element *const yourElement);

  //! Returns structure holding parameters controlling the simulation
  const simulationParametersStruct* parameters() const;

  //! Returns the structure containing the command line argument information
  argvStruct* argStruct() const;

  //! Returns an xmdsArgv object
  xmdsArgv* argv() const;

  //! Returns an xmdsField object
  xmdsField* field() const;

  //! Returns an xmdsOutput object to manage output to file
  const xmdsOutput* output() const;

  //! Returns an xmdsSequence object
  const xmdsSequence* sequence() const;

  //! Returns the next segment number in the simulation sequence
  unsigned long nextSegmentNumber() const;

  //! Inserts relevant code at relevant point in output C++ file
  void makeCode(
                const unsigned long& inFileSplitPoint) const;

  private :

    simulationParametersStruct    myParameters;            //!< Structure of parameters of simulation
  mutable argvStruct            myArgStruct;             //!< Structure of command line arguments to simulation
  mutable list<string>          myArgvNameList;          //!< List of names of command line arguments
  xmdsArgv*                     myArgv;                  //!< The command line argument "vector" object
  xmdsField*                    myField;                 //!< The xmdsField object
  xmdsOutput*                   myOutput;                //!< The xmdsOutput object
  xmdsSequence*                 mySequence;              //!< The xmdsSequence object
  mutable unsigned long         myCurrentSegmentNumber;  //!< The current segment number

  //! Write the simulation include statements to file
  void writeIncludes(
                     FILE *const outfile) const;

  //! Write the simulation define statements to file
  void writeDefines(
                    FILE *const outfile) const;

  //! Write the simulation global variables to file
  void writeGlobals(
                    FILE *const outfile) const;

  //! Write the simulation prototypes to file
  void writePrototypes(
                       FILE *const outfile) const;

  //! Write the simulation routines to file
  void writeRoutines(
                     FILE *const outfile) const;

  //! Create an xmdsGlobals object
  xmdsGlobals* createxmdsGlobals();

  //! Create an xmdsField object
  xmdsField* createxmdsField();

  //! Create an xmdsArgv object
  xmdsArgv* createxmdsArgv();

  //! Create an xmdsOutput object
  xmdsOutput* createxmdsOutput();

  //! Create an xmdsSequence object
  xmdsSequence* createxmdsSequence();
};


#endif // XMDS_SIMULATION_H
