/* properties.c - properties box for xpenguins_applet
 * Copyright (C) 1999-2001  Robin Hogan
 * Copyright (C) 2002       Sebastian Henschel
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*****************************************************************************
 *  preprocessor
 ****************************************************************************/

#include <config.h>
#include <gnome.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>

#include "xpenguins.h"
#include "applet.h"



/*****************************************************************************
 *  declarations
 ****************************************************************************/

static GtkWidget * create_page_general (XPenguinsApplet *);
static void fill_theme_list (
    XPenguinsApplet *, GtkListStore *, GtkTreeSelection *);
static gboolean new_selection (GtkTreeSelection *, GtkTreeModel *,
    GtkTreePath *, gboolean, gpointer);
static GtkWidget * create_theme_list (XPenguinsApplet *);
static GtkWidget * create_page_themes (XPenguinsApplet *);
static void set_atk_name_desc (GtkWidget *, const gchar *, const gchar *);
static void set_relation (GtkWidget *, GtkLabel *);
static void destroy (GtkWidget *, gpointer);
static void response (GtkDialog *, gint, gpointer);
static void help ();
static void restart_timer (XPenguinsApplet *);
static void set_theme_error_label (
    XPenguinsApplet *, const gchar *, const gchar *);
static void set_theme_number_speed (XPenguinsApplet *);
static void update_overridetheme (XPenguinsApplet *, gboolean);
static void update_number (XPenguinsApplet *, gint);
static void update_speed (XPenguinsApplet *, gint);
static void update_ignorepopups (XPenguinsApplet *, gboolean);
static void update_noblood (XPenguinsApplet *, gboolean);
static void update_squish (XPenguinsApplet *, gboolean);
static void update_noangels (XPenguinsApplet *, gboolean);
static void update_theme_widgets (XPenguinsApplet *);
static gboolean update_theme (XPenguinsApplet *, const gchar *);
static void cb_update (GtkWidget *, gpointer);
static void cb_update_theme (GtkTreeSelection *, gpointer);

/* Properties (gconf keys) */
static const gchar *key_themename = "theme_name";
static const gchar *key_overridetheme = "override_theme";
static const gchar *key_number = "number";
static const gchar *key_speed = "speed";
static const gchar *key_noblood = "no_blood";
static const gchar *key_noangels = "no_angels";
static const gchar *key_ignorepopups = "ignore_popups";
static const gchar *key_squish = "squish";

/* Default constants */
static const gint default_speed = 20;
static const gint default_divident = 1000;


extern void debug (gchar *msg)
{
 GtkWidget *dialog;
 dialog = gtk_message_dialog_new (NULL,
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_WARNING,
                                  GTK_BUTTONS_OK_CANCEL,
                                  msg);
gtk_dialog_run (GTK_DIALOG (dialog));
gtk_widget_destroy (dialog);
}


/*****************************************************************************
 *  extern functions
 ****************************************************************************/

/* Load the current settings from gconf */
extern void
xpa_property_load (XPenguinsApplet *xpa)
{
    gchar *theme_name;

    theme_name = panel_applet_gconf_get_string (
        xpa->applet, key_themename, NULL);
    if (theme_name == NULL) theme_name = g_strdup ("Penguins");

    update_theme (xpa, theme_name);
    update_overridetheme (xpa, panel_applet_gconf_get_bool (
        xpa->applet, key_overridetheme, NULL));
    update_number (xpa, panel_applet_gconf_get_int (
        xpa->applet, key_number, NULL));
    update_speed (xpa, panel_applet_gconf_get_int (
        xpa->applet, key_speed, NULL));
    update_ignorepopups (xpa, panel_applet_gconf_get_bool (
        xpa->applet, key_ignorepopups, NULL));
    update_noblood (xpa, panel_applet_gconf_get_bool (
        xpa->applet, key_noblood, NULL));
    update_squish (xpa, panel_applet_gconf_get_bool (
        xpa->applet, key_squish, NULL));
    update_noangels (xpa, panel_applet_gconf_get_bool (
        xpa->applet, key_noangels, NULL));

    g_free (theme_name);
}



/* Create the property box, from which the properties of xpenguins can
 * be changed */
extern void
xpa_property_show (BonoboUIComponent *uic, XPenguinsApplet *xpa,
                   const gchar *verbname)
{
    GtkWidget *notebook;
    GtkWidget *page;
    GtkWidget *label;

    /* If property box already exists, bring it to the top and return */
    if (xpa->prop_window)
    {
        gtk_window_present (GTK_WINDOW (xpa->prop_window));
        return;
    }

    /* Make a new property window */
    xpa->prop_window = gtk_dialog_new ();
    gtk_window_set_title (
        GTK_WINDOW (xpa->prop_window), _("XPenguins Properties"));

    /* Add the buttons */
    gtk_dialog_add_button (
        GTK_DIALOG (xpa->prop_window), GTK_STOCK_HELP, GTK_RESPONSE_HELP);
    gtk_dialog_add_button (
        GTK_DIALOG (xpa->prop_window), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE);
    gtk_dialog_set_default_response (
        GTK_DIALOG (xpa->prop_window), GTK_RESPONSE_CLOSE);

    /* Property box contains two pages: "General" and "Themes" */
    notebook = gtk_notebook_new ();

    /* First the "General" page, which is a vertical container */
    page = create_page_general (xpa);
    /* Post the "General" page in the property box */
    label = gtk_label_new_with_mnemonic (_("_General"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), page, label);

    /* Now the "Themes" page, again a vertical container */
    page = create_page_themes (xpa);
    /* Post the "Themes" page in the property box */
    label = gtk_label_new_with_mnemonic (_("_Themes"));
    gtk_notebook_append_page (GTK_NOTEBOOK(notebook), page, label);

    gtk_box_pack_start (
        GTK_BOX (GTK_DIALOG (xpa->prop_window)->vbox), notebook, TRUE, TRUE, 0);

    /* Callbacks for the various property-box buttons */
    g_signal_connect (
        G_OBJECT (xpa->prop_window), "response", G_CALLBACK (response), xpa);
    g_signal_connect (
        G_OBJECT (xpa->prop_window), "destroy", G_CALLBACK (destroy), xpa);

    gtk_widget_grab_focus (xpa->overridetheme_check);
    gtk_widget_show_all (xpa->prop_window);
}



/*****************************************************************************
 *  static functions
 ****************************************************************************/

/* Make the general page of the notebook */
static GtkWidget *
create_page_general (XPenguinsApplet *xpa)
{
    GtkWidget *page;
    GtkWidget *range;
    GtkWidget *label;
    GtkWidget *evtbox;
    GtkWidget *table;
    GtkTooltips *tooltip = NULL;

    tooltip = gtk_tooltips_new ();
    page = gtk_vbox_new (FALSE, GNOME_PAD);
    gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);


    /* Override the number and speed of toons in the current theme */
    xpa->overridetheme_check = gtk_check_button_new_with_mnemonic (
        _("_Override number and speed of toons in theme"));
    gtk_tooltips_set_tip (
        tooltip, GTK_WIDGET (xpa->overridetheme_check),
        _("Shall the values for the number of toons and the frame rate "
        "provided by the theme be overriden by the user?"), NULL);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (xpa->overridetheme_check), xpa->overridetheme);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->overridetheme_check);
    g_signal_connect (
        G_OBJECT (xpa->overridetheme_check), "clicked",
        G_CALLBACK (cb_update), xpa);

    /* The override button controls this table, which contains the
     * number and speed override controls */
    xpa->table = table = gtk_table_new (2, 2, FALSE);
    gtk_box_pack_start_defaults (GTK_BOX (page), table);
    gtk_widget_set_sensitive (GTK_WIDGET (xpa->table), xpa->overridetheme);

    /* Override the number of penguins */
    /* First the label */
    evtbox = gtk_event_box_new ();
    gtk_tooltips_set_tip (
        tooltip, evtbox, _("How many toons shall be drawn?"), NULL);
    label = gtk_label_new_with_mnemonic (_("Number of toons:"));
    gtk_container_add (GTK_CONTAINER (evtbox), label);
    gtk_table_attach (
        GTK_TABLE (table), evtbox, 0, 1, 0, 1,
        GTK_FILL | GTK_SHRINK, GTK_FILL | GTK_SHRINK, 5, 5);
    /* Then the slider */
    xpa->number_adj = (GtkAdjustment *) gtk_adjustment_new (
        xpa->number, 1, 110, 1, 1, 10);
    range = gtk_hscale_new (GTK_ADJUSTMENT (xpa->number_adj));
    gtk_scale_set_digits (GTK_SCALE (range), 0);
    gtk_table_attach (
        GTK_TABLE (table), range, 1, 2, 0, 1,
        GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_SHRINK, 5, 5);
    g_signal_connect (
        G_OBJECT (xpa->number_adj), "value_changed", G_CALLBACK (cb_update),
        xpa);
    gtk_range_set_update_policy (GTK_RANGE (range), GTK_UPDATE_DISCONTINUOUS);
    set_relation (range, GTK_LABEL (label));

    /* Override the frame rate */
    /* First the label */
    evtbox = gtk_event_box_new ();
    gtk_tooltips_set_tip (
        tooltip, evtbox, _("How fast shall the toons move?"), NULL);
    label = gtk_label_new_with_mnemonic (_("Frame rate (Hz):"));
    gtk_container_add (GTK_CONTAINER (evtbox), label);
    gtk_table_attach (
        GTK_TABLE (table), evtbox, 0, 1, 1, 2,
        GTK_FILL | GTK_SHRINK , GTK_FILL | GTK_SHRINK, 5, 5);
    /* Then the slider */
    xpa->speed_adj = (GtkAdjustment *) gtk_adjustment_new (
        xpa->speed, 1, 110, 1, 1, 10);
    range = gtk_hscale_new (GTK_ADJUSTMENT (xpa->speed_adj));
    gtk_scale_set_digits (GTK_SCALE (range), 0);
    gtk_table_attach (
        GTK_TABLE (table), range, 1, 2, 1, 2,
        GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_SHRINK, 5, 5);
    g_signal_connect (
        G_OBJECT (xpa->speed_adj), "value_changed", G_CALLBACK (cb_update),
        xpa);
    gtk_range_set_update_policy (GTK_RANGE (range), GTK_UPDATE_DISCONTINUOUS);
    set_relation (range, GTK_LABEL (label));

    /* Some check buttons: first one for ignoring popup windows */
    xpa->ignorepopups_check = gtk_check_button_new_with_mnemonic (
        _("_Ignore popup windows"));
    gtk_tooltips_set_tip (
        tooltip, GTK_WIDGET (xpa->ignorepopups_check),
        _("Shall popup-windows be ignored by the toons?"), NULL);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (xpa->ignorepopups_check), xpa->ignorepopups);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->ignorepopups_check);
    g_signal_connect (
        G_OBJECT (xpa->ignorepopups_check), "clicked", G_CALLBACK (cb_update),
        xpa);

    /* Then one for omitting all the gory images! */
    xpa->noblood_check = gtk_check_button_new_with_mnemonic (
        _("Do not show gory _death sequences"));
    gtk_tooltips_set_tip (
        tooltip, GTK_WIDGET (xpa->noblood_check),
        _("Do not show gory death sequences when toons are squished?"),
        NULL);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (xpa->noblood_check), xpa->noblood);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->noblood_check);
    g_signal_connect (
        G_OBJECT (xpa->noblood_check), "clicked", G_CALLBACK (cb_update), xpa);

    /* Then one for squishing the toons via mouse button */
    xpa->squish_check = gtk_check_button_new_with_mnemonic (
        _("_Squish Toons"));
    gtk_tooltips_set_tip (
        tooltip, GTK_WIDGET (xpa->squish_check),
        _("Enable squishing toons with your mouse buttons? Note that this "
        "disables any existing function of the mouse buttons on the root "
        "window."), NULL);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (xpa->squish_check), xpa->squish);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->squish_check);
    g_signal_connect (
        G_OBJECT (xpa->squish_check), "clicked", G_CALLBACK (cb_update), xpa);

    /* Then one for omitting all the angels! */
    xpa->noangels_check = gtk_check_button_new_with_mnemonic (
        _("Do not show _angels"));
    gtk_tooltips_set_tip (
        tooltip, GTK_WIDGET (xpa->noangels_check),
        _("Do not show angels after dying?"), NULL);
    gtk_toggle_button_set_active (
        GTK_TOGGLE_BUTTON (xpa->noangels_check), xpa->noangels);
    gtk_box_pack_start_defaults (GTK_BOX (page), xpa->noangels_check);
    g_signal_connect (
        G_OBJECT (xpa->noangels_check), "clicked", G_CALLBACK (cb_update), xpa);

    return page;
}



static void
fill_theme_list (
    XPenguinsApplet *xpa, GtkListStore *storage, GtkTreeSelection *select)
{
    GtkCellRenderer *cell;
    GtkTreeViewColumn *column;
    GtkTreeIter iters[xpa->nthemes];
    GSList *active_iters = NULL;
    gboolean active_found = FALSE;
    gchar **active_theme_names;
    gint active_nthemes = 0;
    gint i, j;

    /* Create list of themes */
    active_theme_names = g_strsplit (xpa->theme_name, "/", 0);
    while (active_theme_names[active_nthemes] != NULL) active_nthemes++;

    /* Append all themes to storage */
    for (i = 0; i < xpa->nthemes; ++i)
    {
        gint len = strlen (xpa->theme_list[i]);
        gtk_list_store_append (storage, &iters[i]);
        gtk_list_store_set (storage, &iters[i], 0, xpa->theme_list[i], -1);

        /* Check if one of the active themes is in the list */
        for (j = 0; j < active_nthemes; j++)
        {
            if (g_ascii_strncasecmp (
                xpa->theme_list[i], active_theme_names[j], len) == 0)
            {
                active_iters = g_slist_append (active_iters, &iters[i]);
                active_found = TRUE;
            }
        }
    }

    /* Put Storage into treeview */
    cell = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes (
        _("_Select Theme(s)"), cell, "text", 0, NULL);
    gtk_tree_view_column_set_alignment (column, 0.5);
    gtk_tree_sortable_set_sort_column_id (
        GTK_TREE_SORTABLE (storage), 0, GTK_SORT_ASCENDING);
    gtk_tree_view_column_set_sort_column_id (column, 0);
    gtk_tree_view_column_set_sort_indicator (column, TRUE);
    gtk_tree_view_append_column (xpa->treeview, column);

    /* Select entries corresponding to active theme(s) */
    if (active_found)
    {
        guint i;
        guint active_iters_len = g_slist_length (active_iters);

        for (i = 0; i < active_iters_len; i++)
        {
            GtkTreeIter *iter = (GtkTreeIter *) g_slist_nth_data (
                active_iters, i);
            gtk_tree_selection_select_iter (select, iter);
        }
    }

    g_strfreev (active_theme_names);
}



/* Callback when an entry will be (de-)selected */
static gboolean
new_selection (
    GtkTreeSelection *select, GtkTreeModel *model, GtkTreePath *path,
    gboolean is_selected, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;
    GtkTreeIter iter;

    if (xpa->focussed_theme_name) g_free (xpa->focussed_theme_name);
    if (is_selected)
    {
        gchar **theme_names = g_strsplit (xpa->theme_name, "/", 2);
        xpa->focussed_theme_name = g_strdup (theme_names[0]);
        g_strfreev (theme_names);
    }
    else
    {
        gtk_tree_model_get_iter (model, &iter, path);
        gtk_tree_model_get (model, &iter, 0, &xpa->focussed_theme_name, -1);
    }

    return TRUE;
}



/* Creates the theme list */
static GtkWidget *
create_theme_list (XPenguinsApplet *xpa)
{
    GtkListStore *storage;
    GtkTreeSelection *select;
    GtkWidget *scroll;

    /* Create scroll window */
    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (
        GTK_SCROLLED_WINDOW (scroll), GTK_POLICY_AUTOMATIC,
        GTK_POLICY_AUTOMATIC);

    /* Create storage model */
    storage = gtk_list_store_new (1, G_TYPE_STRING);

    /* Create treeview */
    //if (xpa->treeview) gtk_widget_destroy (GTK_WIDGET (xpa->treeview));
    xpa->treeview = (GtkTreeView *) gtk_tree_view_new ();
    gtk_widget_set_size_request (GTK_WIDGET (xpa->treeview), -1, 100);
    gtk_tree_view_set_model (xpa->treeview, GTK_TREE_MODEL (storage));

    /* Setup selection */
    select = gtk_tree_view_get_selection (xpa->treeview);
    gtk_tree_selection_set_mode (select, GTK_SELECTION_MULTIPLE);
    gtk_tree_selection_set_select_function (select, new_selection, xpa, NULL);

    /* Prepare theme list */
    if (xpa->theme_list)
    {
        xpenguins_free_list (xpa->theme_list);
        xpa->theme_list = NULL;
    }
    xpa->theme_list = xpenguins_list_themes (&(xpa->nthemes));

    /* Append each theme to the storage and activate selected theme(s) */
    if (xpa->theme_list)
    {
        fill_theme_list (xpa, storage, select);
        xpa->treeview_signal_id = g_signal_connect (
            G_OBJECT (select), "changed", G_CALLBACK (cb_update_theme), xpa);
    }
    else /* No themes found! */
    {
        set_theme_error_label (
            xpa, XPENGUINS_SYSTEM_DIRECTORY,
            _("No themes found.\nMake sure you have properly installed the "
            "'xpenguins' package"));
    }

    gtk_container_add (GTK_CONTAINER (scroll), GTK_WIDGET (xpa->treeview));
    return scroll;
}



/* Make the themes page of the notebook */
static GtkWidget *
create_page_themes (XPenguinsApplet *xpa)
{
    GtkWidget *page;
    GtkWidget *scroll;

    page = gtk_vbox_new (FALSE, GNOME_PAD);
    gtk_container_set_border_width (GTK_CONTAINER (page), GNOME_PAD);

    /* Preview themes in a horizontal container in which we put the
     * icon and the text information
     */
    xpa->preview = gtk_hbox_new (FALSE, GNOME_PAD);
    xpa->info = gtk_label_new (NULL);
    xpa->preview_pixmap = NULL;
    set_atk_name_desc (
        xpa->preview_pixmap, _("Preview Pixmap"),
        _("A small pixmap to preview the look of the theme"));
    gtk_label_set_justify (GTK_LABEL (xpa->info), GTK_JUSTIFY_LEFT);
    gtk_box_pack_end_defaults (GTK_BOX (xpa->preview), xpa->info);
    gtk_box_pack_start (GTK_BOX (page), xpa->preview, FALSE, FALSE, 0);

    /* The last widget is a scrolled window containing a list of
     * the available themes
     */
    scroll = create_theme_list (xpa);

    gtk_box_pack_start_defaults (GTK_BOX (page), scroll);

    /* Update widgets to show current theme */
    update_theme_widgets (xpa);

    return page;
}



/* Sets the name and description of a widget for ATK */
static void
set_atk_name_desc (GtkWidget *widget, const gchar* name, const gchar *desc)
{
    AtkObject *obj;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (name != NULL);
    g_return_if_fail (desc != NULL);

    obj = gtk_widget_get_accessible (widget);
    if (!GTK_IS_ACCESSIBLE (obj)) return;

    atk_object_set_name (obj, name);
    atk_object_set_description (obj, desc);
}



/* Set the relation between a widget and a label for ATK */
static void
set_relation (GtkWidget *widget, GtkLabel *label)
{
    AtkObject *atk_widget, *atk_label;
    AtkRelationSet *set;
    AtkRelation *relation;

    g_return_if_fail (widget != NULL);
    g_return_if_fail (label != NULL);

    /* Set the ATK_RELATION_LABEL_FOR relation */
    gtk_label_set_mnemonic_widget (label, widget);

    atk_widget = gtk_widget_get_accessible (widget);
    if (!GTK_IS_ACCESSIBLE (atk_widget)) return;

    atk_label = gtk_widget_get_accessible (GTK_WIDGET (label));
    relation = atk_relation_new (&atk_label, 1, ATK_RELATION_LABELLED_BY);
    set = atk_object_ref_relation_set (atk_widget);
    atk_relation_set_add (set, relation);

    g_object_unref (relation);
}



/* Callback to destroy properties */
static void
destroy (GtkWidget *widget, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    gtk_widget_destroy (xpa->prop_window);
    xpa->prop_window = NULL;
}



/* Callback to respond to click on help/apply/ok/close */
static void
response (GtkDialog *dialog, gint id, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    switch (id)
    {
        case GTK_RESPONSE_HELP:
            help ();
            break;
        case GTK_RESPONSE_CLOSE:
        default:
            destroy (GTK_WIDGET (dialog), xpa);
            break;
    }
}



/* Display help dialog */
static void
help ()
{
    GError *error = NULL;

    gnome_help_display_desktop (NULL, PACKAGE, PACKAGE,
                                "props", &error);
    if (error)
    {
        g_warning ("help error: %s\n", error->message);
        g_error_free (error);
    }
}



static void
restart_timer (XPenguinsApplet *xpa)
{
    gtk_timeout_remove (xpa->timeout);
    xpa->timeout = gtk_timeout_add (
        default_divident / xpa->speed, xpa_service_xpenguins, xpa);
}



/* Convenience function for setting the info label on the themes page in case
 * of error
 */
static void
set_theme_error_label (
    XPenguinsApplet *xpa, const gchar* theme_name, const gchar *msg)
{
    gchar *str = g_strdup_printf (_("theme error: %s (%s)"), theme_name, msg);

    gtk_label_set_text (GTK_LABEL (xpa->info), str);
    g_free (str);
    if (xpa->preview_pixmap) gtk_widget_destroy (xpa->preview_pixmap);
}



/* Sets the number and speed widgets/variables, etc. enforced by the theme */
static void
set_theme_number_speed (XPenguinsApplet *xpa)
{
    guint delay = (xpa->theme).delay;

    if (delay != 0) xpa->speed = default_divident / delay;
    else xpa->speed = default_speed;
    xpa->number = (xpa->theme).total;

    if (xpa->prop_window)
    {
        gtk_adjustment_set_value (
            GTK_ADJUSTMENT (xpa->number_adj), xpa->number);
        gtk_adjustment_set_value (
            GTK_ADJUSTMENT (xpa->speed_adj), xpa->speed);
    }
}



/* Updates the override theme settings */
static void
update_overridetheme (XPenguinsApplet *xpa, gboolean state)
{
    xpa->overridetheme = state;

    if (xpa->loaded)
    {
        if (xpa->overridetheme)
        {
            xpa->number = panel_applet_gconf_get_int (
                xpa->applet, key_number, NULL);
            xpa->speed = panel_applet_gconf_get_int (
                xpa->applet, key_speed, NULL);
        }
        else
        {
            set_theme_number_speed (xpa);
        }
        xpenguins_set_number (xpa->number);
    }
}


/* Updates the number of toons */
static void
update_number (XPenguinsApplet *xpa, gint number)
{
    xpa->number = number;
    xpenguins_set_number (xpa->number);
}


/* Updates the speed of the toons */
static void
update_speed (XPenguinsApplet *xpa, gint speed)
{
    xpa->speed = speed;
}



/* Updates the property ignorepopups */
static void
update_ignorepopups (XPenguinsApplet *xpa, gboolean state)
{
    xpa->ignorepopups = state;
    xpenguins_ignorepopups (state);
}



/* Updates the property noblood */
static void
update_noblood (XPenguinsApplet *xpa, gboolean state)
{
    xpa->noblood = state;
    xpenguins_set_blood (!state);
}



/* Updates the property squish */
static void
update_squish (XPenguinsApplet *xpa, gboolean state)
{
    xpa->squish = state;
    xpenguins_squish (state);

    /* Restart xpenguins, see xpenguins.h */
    if (xpa->active && xpa->loaded)
    {
        xpenguins_exit ();
        xpenguins_start (NULL);
    }
}



/* Updates the property noangels */
static void
update_noangels (XPenguinsApplet *xpa, gboolean state)
{
    xpa->noangels = state;
    xpenguins_set_angels (!state);
}



/* Updates the widgets of the currently selected theme */
static void
update_theme_widgets (XPenguinsApplet *xpa)
{
    if (xpa->theme_info) /* Clear old theme information */
    {
        xpenguins_free_list (xpa->theme_info);
        xpa->theme_info = NULL;
    }

    /* Get information about the new theme */
    if ((xpa->theme_info = xpenguins_theme_info (xpa->focussed_theme_name)))
    {
        /* Create the text information string - first add the theme name */
        GString *string = g_string_new (xpa->focussed_theme_name);
        if (xpa->preview_pixmap) gtk_widget_destroy (xpa->preview_pixmap);
        /* Display the icon if it exists */
        if ((xpa->preview_pixmap = gtk_image_new_from_file (
            xpa->theme_info[PENGUIN_ICON])))
        {
            gtk_box_pack_end (
                GTK_BOX (xpa->preview), xpa->preview_pixmap, FALSE, FALSE, 5);
            gtk_widget_show (xpa->preview_pixmap);
        }
        if (*(xpa->theme_info[PENGUIN_DATE])) /* Add the date */
        {
            g_string_append (string, _(" (Last updated "));
            g_string_append (string, xpa->theme_info[PENGUIN_DATE]);
            g_string_append (string, ")");
        }
        if (*(xpa->theme_info[PENGUIN_ARTIST])) /* Add the artist(s) */
        {
            g_string_append (string, _("\nArtists: "));
            g_string_append (string, xpa->theme_info[PENGUIN_ARTIST]);
        }
        if (*(xpa->theme_info[PENGUIN_MAINTAINER])) /* Add the maintainer */
        {
            g_string_append (string, _("\nMaintainer: "));
            g_string_append (string, xpa->theme_info[PENGUIN_MAINTAINER]);
        }
        if (*(xpa->theme_info[PENGUIN_COPYRIGHT])) /* Add the copyright */
        {
            g_string_append (string, _("\nCopyright (C) "));
            g_string_append (string, xpa->theme_info[PENGUIN_COPYRIGHT]);
        }
        if (*(xpa->theme_info[PENGUIN_COMMENT])) /* Add the comment field */
        {
            g_string_append (string, "\n");
            g_string_append (string, xpa->theme_info[PENGUIN_COMMENT]);
        }
        /* Display the information */
        gtk_label_set_text (GTK_LABEL (xpa->info), string->str);
        g_string_free (string, FALSE);
    }
    else /* No "about" file found for theme */
    {
        set_theme_error_label (
            xpa, xpa->theme_name, _("no information available"));
    }
}



/* Updates the theme */
static gboolean
update_theme (XPenguinsApplet *xpa, const gchar *next)
{
    gchar *prev;

    g_return_val_if_fail (next != NULL, FALSE);

    /* Test for equality of current and next theme(s) */
    if (xpa->theme_name)
    {
        gint next_len = strlen (next);
        gint cur_len = strlen (xpa->theme_name);
        if (cur_len == next_len &&
            g_ascii_strncasecmp (xpa->theme_name, next, cur_len) == 0)
        {
            return FALSE;
        }
    }

    /* Stop xpenguins and free the old theme */
    if (xpa->active)
    {
        gtk_timeout_remove (xpa->timeout);
        xpenguins_exit ();
    }
    if (xpa->loaded)
    {
        xpenguins_free_theme (&(xpa->theme));
        xpa->loaded = FALSE;
    }

    /* Load new theme(s) */
    prev = xpa->theme_name;
    xpa->theme_name = g_strdup (next);
    if (!xpa_load_theme (xpa))
    {
        /* Reload old theme(s) */
        xpa->active = FALSE;
        g_free (xpa->theme_name);
        xpa->theme_name = prev;
        xpa_load_theme (xpa);
        return FALSE;
    }
    else
    {
        g_free (prev);
    }
    xpa->loaded = TRUE;

    /* Set number and speed if appropriate */
    if (!xpa->overridetheme) set_theme_number_speed (xpa);
    xpenguins_set_number (xpa->number);

    /* Update widgets */
    if (xpa->prop_window) update_theme_widgets (xpa);

    /* Start xpenguins and the service-timer */
    if (xpa->active)
    {
        gchar *error = xpenguins_start (NULL);
        if (error != NULL)
        {
            set_theme_error_label (xpa, xpa->theme_name, error);
            return FALSE;
        }
        restart_timer (xpa);
    }

    return TRUE;
}



/* Apply all the settings in the property box */
static void
cb_update (GtkWidget *widget, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;

    if (widget == xpa->overridetheme_check)
    {
        gboolean state;
        state = gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (xpa->overridetheme_check));
        panel_applet_gconf_set_bool (
            xpa->applet, key_overridetheme, state, NULL);
        gtk_widget_set_sensitive (GTK_WIDGET (xpa->table), state);
        update_overridetheme (xpa, state);
    }

    else if (widget == GTK_WIDGET (xpa->number_adj))
    {
        gint val = gtk_adjustment_get_value (GTK_ADJUSTMENT (xpa->number_adj));
        panel_applet_gconf_set_int (xpa->applet, key_number, val, NULL);
        update_number (xpa, val);
    }

    else if (widget == GTK_WIDGET (xpa->speed_adj))
    {
        gint val = gtk_adjustment_get_value (GTK_ADJUSTMENT (xpa->speed_adj));
        panel_applet_gconf_set_int (xpa->applet, key_speed, val, NULL);
        update_speed (xpa, val);
    }

    else if (widget == xpa->ignorepopups_check)
    {
        gboolean state = gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (xpa->ignorepopups_check));
        panel_applet_gconf_set_bool (
            xpa->applet, key_ignorepopups, state, NULL);
        update_ignorepopups (xpa, state);
    }

    else if (widget == xpa->noblood_check)
    {
        gboolean state = gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (xpa->noblood_check));
        panel_applet_gconf_set_bool (
            xpa->applet, key_noblood, state, NULL);
        update_noblood (xpa, state);
    }

    else if (widget == xpa->squish_check)
    {
        gboolean state = gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (xpa->squish_check));
        panel_applet_gconf_set_bool (
            xpa->applet, key_squish, state, NULL);
        update_squish (xpa, state);
    }

    else if (widget == xpa->noangels_check)
    {
        gboolean state = gtk_toggle_button_get_active (
            GTK_TOGGLE_BUTTON (xpa->noangels_check));
        panel_applet_gconf_set_bool (
            xpa->applet, key_noangels, state, NULL);
        update_noangels (xpa, state);
    }

    if (xpa->active) restart_timer (xpa);
}



/* A theme has been selected in the theme list - display the preview
 * icon and text for it */
static void
cb_update_theme (GtkTreeSelection *selection, gpointer data)
{
    XPenguinsApplet *xpa = (XPenguinsApplet *) data;
    GString *next = NULL;
    GtkTreeModel *model;
    GList *rows;
    guint rows_len, i;

    /* Get list of selected rows, loop thru it and build new next_theme_name */
    rows = gtk_tree_selection_get_selected_rows (selection, &model);
    rows_len = g_list_length (rows);
    for (i = 0; i < rows_len; i++)
    {
        gchar *name;
        GtkTreePath *path;
        GtkTreeIter iter;

        path = (GtkTreePath *) g_list_nth_data (rows, i);
        gtk_tree_model_get_iter (model, &iter, path);
        gtk_tree_model_get (model, &iter, 0, &name, -1);

        if (i > 0)
        {
            g_string_append (next, "/");
            g_string_append (next, name);
        }
        else
        {
            next = g_string_new (name);
        }

        g_free (name);
    }

    g_list_foreach (rows, (GFunc) gtk_tree_path_free, NULL);
    g_list_free (rows);

    if (next)
    {
        if (update_theme (xpa, next->str))
        {
            panel_applet_gconf_set_string (
                xpa->applet, key_themename, xpa->theme_name, NULL);
        }
        g_string_free (next, TRUE);
    }
}

