/***************************************************************************
 *
 * Copyright (c) 2000, 2001, 2002, 2003, 2004, 2005 BalaBit IT Ltd, Budapest, Hungary
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation.
 *
 * Note that this permission is granted for only version 2 of the GPL.
 *
 * As an additional exemption you are allowed to compile & link against the
 * OpenSSL libraries as published by the OpenSSL project. See the file
 * COPYING for details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: transfer.c,v 1.27 2004/07/01 16:55:25 bazsi Exp $
 *
 * Author  : bazsi
 * Auditor : 
 * Last audited version: 1.1
 * Notes:
 *   
 ***************************************************************************/

#include <zorp/transfer.h>
#include <zorp/log.h>
#include <zorp/source.h>

#define MAX_READ_AT_A_TIME 30

/**
 * z_transfer_src_read_method:
 * @self: not used
 * @stream: Stream to read from
 * @buf: Buffer to read into
 * @buf_len: Number of bytes to read
 * @bytes_read: Number of bytes read
 * @error: Error value
 *
 * Wrapper around z_stream_read, read from a stream into a buffer
 *
 * Returns:
 * The status of the operation
 */
static GIOStatus
z_transfer_src_read_method(ZTransfer *self G_GNUC_UNUSED, ZStream *stream, gchar *buf, gsize buf_len, gsize *bytes_read, GError **error)
{
  z_proxy_enter(self->owner);
  z_proxy_leave(self->owner);
  return z_stream_read(stream, buf, buf_len, bytes_read, error);
}

/**
 * z_transfer_src_write_method:
 * @self: not used
 * @stream: Stream to write into
 * @buf: Buffer to write from
 * @buf_len: Number of bytes to write
 * @bytes_written: Number of bytes written
 * @error: Error value
 *
 * Wrapper around z_stream write, write into a stream from a buffer
 *
 * Returns:
 * The status of the operation
 */
static GIOStatus
z_transfer_src_write_method(ZTransfer *self G_GNUC_UNUSED, ZStream *stream, const gchar *buf, gsize buf_len, gsize *bytes_written, GError **error)
{
  z_enter();
  z_leave();
  return z_stream_write(stream, buf, buf_len, bytes_written, error);
}

/**
 * z_transfer_src_shutdown_method:
 * @self: not used
 * @s: not used
 * @shutdown_mode: not used
 * @err: not used
 *
 * Dummy, simple ZStream doesn't need shutting down, but derived classes may.
 *
 * Returns:
 * G_IO_STATUS_NORMAL
 */
static GIOStatus 
z_transfer_src_shutdown_method(ZTransfer *self G_GNUC_UNUSED, ZStream *s G_GNUC_UNUSED, gint shutdown_mode G_GNUC_UNUSED, GError **err G_GNUC_UNUSED)
{
  z_enter();
  z_leave();
  return G_IO_STATUS_NORMAL;
}

static GIOStatus
z_transfer_dst_read_method(ZTransfer *self G_GNUC_UNUSED, ZStream *stream, gchar *buf, gsize buf_len, gsize *bytes_read, GError **error)
{
  z_enter();
  z_leave();
  return z_stream_read(stream, buf, buf_len, bytes_read, error);
}

static GIOStatus
z_transfer_dst_write_method(ZTransfer *self G_GNUC_UNUSED, ZStream *stream, const gchar *buf, gsize buf_len, gsize *bytes_written, GError **error)
{
  z_enter();
  z_leave();
  return z_stream_write(stream, buf, buf_len, bytes_written, error);
}

static GIOStatus 
z_transfer_dst_shutdown_method(ZTransfer *self G_GNUC_UNUSED, ZStream *s G_GNUC_UNUSED, gint shutdown_mode G_GNUC_UNUSED, GError **err G_GNUC_UNUSED)
{
  z_enter();
  z_leave();
  return G_IO_STATUS_NORMAL;
}

static gboolean
z_transfer_progress_method(ZTransfer *self G_GNUC_UNUSED, gint progress_reason G_GNUC_UNUSED)
{
  z_enter();
  z_leave();
  return TRUE;
}


/* actual code to copy data */
/* possible eofmask values */
#define ZTS_EOF_CLIENT_REMOVED   0x0010
#define ZTS_EOF_SERVER_REMOVED   0x0020

#define ZTS_EOF_ALL              0x000f

void
z_transfer_update_eof_mask(ZTransfer *self, guint add_mask)
{
  gulong old_mask = self->status & ZTS_EOF_BITS;
 
  z_proxy_enter(self->owner);
  add_mask = add_mask & ZTS_EOF_BITS;
  self->status |= add_mask;
  
  if ((self->status & (ZTS_EOF_CLIENT_R | ZTS_EOF_CLIENT_W | ZTS_EOF_CLIENT_REMOVED)) == (ZTS_EOF_CLIENT_R | ZTS_EOF_CLIENT_W))
    {
      self->status |= ZTS_EOF_CLIENT_REMOVED;
    }

  if ((self->status & (ZTS_EOF_SERVER_R | ZTS_EOF_SERVER_W | ZTS_EOF_SERVER_REMOVED)) == (ZTS_EOF_SERVER_R | ZTS_EOF_SERVER_W))
    {
      self->status |= ZTS_EOF_SERVER_REMOVED;
    }
  
  if ((self->status & (ZTS_FINISHED | ZTS_EOF_CLIENT_REMOVED | ZTS_EOF_SERVER_REMOVED)) == (ZTS_EOF_CLIENT_REMOVED | ZTS_EOF_SERVER_REMOVED))
    {
#if 0
      /* FIXME !! */
      self->session_data->destroy_queue = g_list_prepend(self->session_data->destroy_queue, self);
#endif
      self->status |= ZTS_FINISHED;
    }
  /*LOG
    This message reports that the data-transfer to or from some endpoint is closed.
   */
  z_proxy_log(self->owner, CORE_DEBUG, 7, "Eofmask is updated; old_mask='%04lx', eof_mask='%04lx'", old_mask, self->status & ZTS_EOF_BITS);
  z_proxy_leave(self->owner);
}

static GIOStatus
z_transfer_shutdown(ZTransfer *self, gint endpoint, gint shutdown_mode, GError **err)
{
  GIOStatus res = G_IO_STATUS_NORMAL;
  gboolean nonblocking;
  guint old_shutdown_flags, shutdown_flags;
  
  z_proxy_enter(self->owner);
  /*
   * self->shutdown mask contains two bits for each endpoints, the lower
   * representing read, the higher write side shutdown. The first
   * expression masks and shifts the bits corresponding to our endpoint.
   * (old_shutdown_flags)
   */
  old_shutdown_flags = (self->shutdown_mask & (3 << (endpoint * 2))) >> (endpoint * 2);
  
  /*
   * shutdown_mode is converted to r/w flags by adding one. The difference
   * is calculated, thus shutdown_flags will contain previously unshut
   * directions.
   */
  shutdown_flags = (~old_shutdown_flags & (shutdown_mode + 1));
  
  if (shutdown_flags)
    {
      /* we store directions shut in this call */
      self->shutdown_mask |= shutdown_flags << (endpoint * 2);
      
      /* convert back to shutdown mode */
      shutdown_mode = shutdown_flags - 1;
      
      nonblocking = z_stream_get_nonblock(self->endpoints[endpoint]);
      
      z_stream_set_nonblock(self->endpoints[endpoint], FALSE);
      if (endpoint & ZTE_STACKED)
        {
          res = z_stream_shutdown(self->endpoints[endpoint], shutdown_mode, err);
        }
      else if (endpoint == ZTE_CLIENT)
        {
          res = z_transfer_src_shutdown(self, self->endpoints[endpoint], shutdown_mode, err);
        }
      else
        {
          res = z_transfer_dst_shutdown(self, self->endpoints[endpoint], shutdown_mode, err);
        }
      z_stream_set_nonblock(self->endpoints[endpoint], nonblocking);
    }
  z_proxy_leave(self->owner);
  return res;
}

static GIOStatus
z_transfer_read_input(ZTransfer *self, gint endpoint, ZTransferIOBuffer *buf)
{
  GIOStatus rc;

  z_proxy_enter(self->owner);
  if (endpoint & ZTE_STACKED)
    {
      rc = z_stream_read(self->endpoints[endpoint], buf->buf, self->buffer_size, &buf->end, NULL);
    }
  else if (endpoint == ZTE_CLIENT)
    {
      rc = z_transfer_src_read(self, self->endpoints[endpoint], buf->buf, self->buffer_size, &buf->end, NULL);
    }
  else 
    {
      rc = z_transfer_dst_read(self, self->endpoints[endpoint], buf->buf, self->buffer_size, &buf->end, NULL);
    }
  if (rc == G_IO_STATUS_NORMAL)
    {
      buf->packet_bytes += buf->end;
      buf->packet_count++;
      self->global_packet_count++;
#if 0
      FIXME: progress
      if (self->session_data->packet_stats_interval_packet &&
         (self->global_packet_count % self->session_data->packet_stats_interval_packet) == 0)
        {
          if (!self->session_data->packet_stats(self->owner, self->vars, 
                                                self->buffers[EP_CLIENT].packet_bytes, 
                                                self->buffers[EP_CLIENT].packet_count,
                                                self->buffers[EP_SERVER].packet_bytes,
                                                self->buffers[EP_SERVER].packet_count))
            z_plug_update_eof_mask(self, ZTS_EOF_ALL);
        }
#endif
    }
  z_proxy_leave(self->owner);
  return rc;
}

static GIOStatus
z_transfer_write_output(ZTransfer *self, ZTransferIOBuffer *buf, gint endpoint)
{
  GIOStatus rc;
  gsize bytes_written;
  
  z_proxy_enter(self->owner);
  if (buf->ofs != buf->end)
    {
      /* buffer not empty */
      if (endpoint & ZTE_STACKED)
        {
          rc = z_stream_write(self->endpoints[endpoint], &buf->buf[buf->ofs], buf->end - buf->ofs, &bytes_written, NULL);
        }
      else if (endpoint == ZTE_CLIENT)
        {
          rc = z_transfer_src_write(self, self->endpoints[endpoint], &buf->buf[buf->ofs], buf->end - buf->ofs, &bytes_written, NULL);
        }
      else
        {
          rc = z_transfer_dst_write(self, self->endpoints[endpoint], &buf->buf[buf->ofs], buf->end - buf->ofs, &bytes_written, NULL);
        }
      switch (rc) 
        {
        case G_IO_STATUS_NORMAL:
          buf->ofs += bytes_written;
          break;
        case G_IO_STATUS_AGAIN:
          break;
        default:
          z_proxy_leave(self->owner);
          return rc;
        }
      if (buf->ofs != buf->end)
        {
          z_proxy_leave(self->owner);
          return G_IO_STATUS_AGAIN;
        }
      else
        {
          z_proxy_leave(self->owner);
          return G_IO_STATUS_NORMAL;
        }
    }
  z_proxy_leave(self->owner);
  return G_IO_STATUS_NORMAL;
}


static GIOStatus
z_transfer_copy_data(ZTransfer *self, gint ep_from, gint ep_to, ZTransferIOBuffer *buf, gboolean *error_on_flush)
{
  GIOStatus rc = G_IO_STATUS_ERROR;
  int pkt_count = 0;
  ZStream *from = self->endpoints[ep_from];
  ZStream *to = ep_to != -1 ? self->endpoints[ep_to] : NULL;

  z_proxy_enter(self->owner);
  *error_on_flush = FALSE;

  if (self->timeout_source)
    z_timeout_source_set_timeout(self->timeout_source, self->timeout);

  z_stream_set_cond(from, Z_STREAM_FLAG_READ, FALSE);

  if (to)
    {
      z_stream_set_cond(to, Z_STREAM_FLAG_WRITE, FALSE);
      rc = z_transfer_write_output(self, buf, ep_to);
      if (rc == G_IO_STATUS_AGAIN)
        {
          z_proxy_leave(self->owner);
          return rc;
        }
      else if (rc != G_IO_STATUS_NORMAL)
        {
          *error_on_flush = TRUE;
          self->status |= ZTS_FAILED;
          z_proxy_leave(self->owner);
          return rc;
        }
    }
  while (pkt_count < MAX_READ_AT_A_TIME)
    {  
      buf->ofs = buf->end = 0;
      rc = z_transfer_read_input(self, ep_from, buf);
      if (rc == G_IO_STATUS_NORMAL)
        {
          if (to)
            {
              rc = z_transfer_write_output(self, buf, ep_to);
              if (rc == G_IO_STATUS_AGAIN)
                break;
              else if (rc != G_IO_STATUS_NORMAL)
                {
                  *error_on_flush = TRUE;
                  self->status |= ZTS_FAILED;
                  z_proxy_leave(self->owner);
                  return rc;
                }
            }
        }
      else if (rc == G_IO_STATUS_AGAIN)
        break;
      else if (rc == G_IO_STATUS_EOF)
        {
          z_proxy_leave(self->owner);
          return rc;
        }
      else
        {
          self->status |= ZTS_FAILED;
          z_proxy_leave(self->owner);
          return G_IO_STATUS_ERROR;
        }
      pkt_count++;
    }

  z_proxy_leave(self->owner);
  return rc;
}

static gboolean
z_transfer_copy_client_to_server(ZStream      *stream G_GNUC_UNUSED,
                                 GIOCondition cond G_GNUC_UNUSED,
                                 gpointer     user_data)
{
  ZTransfer *self = Z_CAST(user_data, ZTransfer);
  GIOStatus ret;
  gboolean error_on_flush;

  z_proxy_enter(self->owner);
  if (self->flags & ZTF_COPY_TO_SERVER)
    ret = z_transfer_copy_data(self, ZTE_CLIENT, ZTE_SERVER, &self->buffers[ZTE_SERVER], &error_on_flush);
  else
    ret = z_transfer_copy_data(self, ZTE_CLIENT, -1, &self->buffers[ZTE_SERVER], &error_on_flush);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      z_transfer_enable_copy(self, ZTE_SERVER);
      break;
    case G_IO_STATUS_EOF:
      if (self->flags & ZTF_SEPARATE_SHUTDOWN)
        {
          z_transfer_shutdown(self, ZTE_CLIENT, SHUT_RD, NULL);
          z_transfer_shutdown(self, ZTE_SERVER, SHUT_WR, NULL);
          z_transfer_update_eof_mask(self, ZTS_EOF_CLIENT_R | ZTS_EOF_SERVER_W);
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
      break;
    default:
      if ((self->flags & ZTF_COMPLETE_COPY) && error_on_flush)
        {
          z_transfer_shutdown(self, ZTE_SERVER, SHUT_WR, NULL);
          z_transfer_update_eof_mask(self, ZTS_EOF_SERVER_W);
          z_transfer_enable_copy(self, ZTE_SERVER);
          self->flags &= ~ZTF_COPY_TO_SERVER;
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
//      z_proxy_leave(self->owner);
//      return FALSE;
    }
  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_transfer_copy_server_to_client(ZStream      *stream G_GNUC_UNUSED,
                                 GIOCondition cond G_GNUC_UNUSED,
                                 gpointer     user_data)
{
  ZTransfer *self = Z_CAST(user_data, ZTransfer);
  GIOStatus ret;
  gboolean error_on_flush;

  z_proxy_enter(self->owner);
  if (self->flags & ZTF_COPY_TO_CLIENT)
    ret = z_transfer_copy_data(self, ZTE_SERVER, ZTE_CLIENT, &self->buffers[ZTE_CLIENT], &error_on_flush);
  else
    ret = z_transfer_copy_data(self, ZTE_SERVER, -1, &self->buffers[ZTE_CLIENT], &error_on_flush);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      z_transfer_enable_copy(self, ZTE_CLIENT);
      break;
    case G_IO_STATUS_EOF:
      if (self->flags & ZTF_SEPARATE_SHUTDOWN)
        {
          z_transfer_shutdown(self, EP_SERVER, SHUT_RD, NULL);
          z_transfer_shutdown(self, EP_CLIENT, SHUT_WR, NULL);
          z_transfer_update_eof_mask(self, ZTS_EOF_SERVER_R | ZTS_EOF_CLIENT_W);
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
      break;
    default:
      if ((self->flags & ZTF_COMPLETE_COPY) && error_on_flush)
        {
          z_transfer_shutdown(self, ZTE_CLIENT, SHUT_WR, NULL);
          z_transfer_update_eof_mask(self, ZTS_EOF_CLIENT_W);
          z_transfer_enable_copy(self, ZTE_CLIENT);
          self->flags &= ~ZTF_COPY_TO_CLIENT;
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
//      z_proxy_leave(self->owner);
//      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

/* I/O callbacks when a proxy is stacked */
static gboolean
z_transfer_copy_client_to_down(ZStream      *stream G_GNUC_UNUSED,
                               GIOCondition cond G_GNUC_UNUSED,
	      		       gpointer     user_data)
{
  ZTransfer *self = Z_CAST(user_data, ZTransfer);
  GIOStatus ret;
  gboolean error_on_flush;

  z_proxy_enter(self->owner);
  if (self->flags & ZTF_COPY_TO_SERVER)
    ret = z_transfer_copy_data(self, ZTE_CLIENT, ZTE_DOWN_CLIENT, &self->buffers[ZTE_DOWN_CLIENT], &error_on_flush);
  else
    ret = z_transfer_copy_data(self, ZTE_CLIENT, -1, &self->buffers[ZTE_DOWN_CLIENT], &error_on_flush);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      z_transfer_enable_copy(self, ZTE_SERVER);
      break;
    case G_IO_STATUS_EOF:
      if (self->flags & ZTF_SEPARATE_SHUTDOWN)
        {
          z_transfer_shutdown(self, ZTE_CLIENT, SHUT_RD, NULL);
	  z_transfer_shutdown(self, ZTE_DOWN_CLIENT, SHUT_WR, NULL);
	  z_transfer_update_eof_mask(self, ZTS_EOF_CLIENT_R);
	}
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
	}
      break;
    default:
      if ((self->flags & ZTF_COMPLETE_COPY) && error_on_flush)
        {
          z_transfer_shutdown(self, ZTE_DOWN_CLIENT, SHUT_WR, NULL);
          z_transfer_enable_copy(self, ZTE_SERVER);
          self->flags &= ~ZTF_COPY_TO_SERVER;
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
//      z_proxy_leave(self->owner);
//      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_transfer_copy_down_to_client(ZStream      *stream G_GNUC_UNUSED,
                               GIOCondition cond G_GNUC_UNUSED,
                               gpointer     user_data)
{
  ZTransfer *self = Z_CAST(user_data, ZTransfer);
  GIOStatus ret;
  gboolean error_on_flush;

  z_proxy_enter(self->owner);
  if (self->flags & ZTF_COPY_TO_CLIENT)
    ret = z_transfer_copy_data(self, ZTE_DOWN_CLIENT, ZTE_CLIENT, &self->buffers[ZTE_CLIENT], &error_on_flush);
  else
    ret = z_transfer_copy_data(self, ZTE_DOWN_CLIENT, -1, &self->buffers[ZTE_CLIENT], &error_on_flush);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN:
      z_transfer_enable_copy(self, ZTE_CLIENT);
      break;
    case G_IO_STATUS_EOF:
      if (self->flags & ZTF_SEPARATE_SHUTDOWN)
        {
          z_transfer_shutdown(self, ZTE_DOWN_CLIENT, SHUT_RD, NULL);
          z_transfer_shutdown(self, ZTE_CLIENT, SHUT_WR, NULL);
	  z_transfer_update_eof_mask(self, ZTS_EOF_CLIENT_W);
        }
      else
        {
	  z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
 	}
      break;
    default:
      if ((self->flags & ZTF_COMPLETE_COPY) && error_on_flush)
        {
          z_transfer_shutdown(self, ZTE_CLIENT, SHUT_WR, NULL);
          z_transfer_update_eof_mask(self, ZTS_EOF_CLIENT_W);
          z_transfer_enable_copy(self, ZTE_CLIENT);
          self->flags &= ~ZTF_COPY_TO_CLIENT;
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
//      z_proxy_leave(self->owner);
//      return FALSE;
    }
  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_transfer_copy_server_to_down(ZStream      *stream G_GNUC_UNUSED,
                               GIOCondition cond G_GNUC_UNUSED,
                               gpointer     user_data)
{
  ZTransfer *self = Z_CAST(user_data, ZTransfer);
  GIOStatus ret;
  gboolean error_on_flush;

  z_proxy_enter(self->owner);
  if (self->flags & ZTF_COPY_TO_CLIENT)
    ret = z_transfer_copy_data(self, ZTE_SERVER, ZTE_DOWN_SERVER, &self->buffers[ZTE_DOWN_SERVER], &error_on_flush);
  else
    ret = z_transfer_copy_data(self, ZTE_SERVER, -1, &self->buffers[ZTE_DOWN_SERVER], &error_on_flush);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN: 
      z_transfer_enable_copy(self, ZTE_CLIENT);
      break;
    case G_IO_STATUS_EOF:
      if (self->flags & ZTF_SEPARATE_SHUTDOWN)
        {
          z_transfer_shutdown(self, ZTE_SERVER, SHUT_RD, NULL);  
          z_transfer_shutdown(self, ZTE_DOWN_SERVER, SHUT_WR, NULL);  
          z_transfer_update_eof_mask(self, ZTS_EOF_SERVER_R);
        }
      else
        { 
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
      break;
    default:
      if ((self->flags & ZTF_COMPLETE_COPY) && error_on_flush)
        {
          z_transfer_shutdown(self, ZTE_DOWN_SERVER, SHUT_WR, NULL);
          z_transfer_enable_copy(self, ZTE_CLIENT);
          self->flags &= ~ZTF_COPY_TO_CLIENT;
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
//      z_proxy_leave(self->owner);
//      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

static gboolean
z_transfer_copy_down_to_server(ZStream      *stream G_GNUC_UNUSED,
                               GIOCondition cond G_GNUC_UNUSED,
                               gpointer     user_data)
{
  ZTransfer *self = Z_CAST(user_data, ZTransfer);
  GIOStatus ret;
  gboolean error_on_flush;

  z_proxy_enter(self->owner);
  if (self->flags & ZTF_COPY_TO_SERVER)
    ret = z_transfer_copy_data(self, ZTE_DOWN_SERVER, ZTE_SERVER, &self->buffers[ZTE_SERVER], &error_on_flush);
  else
    ret = z_transfer_copy_data(self, ZTE_DOWN_SERVER, -1, &self->buffers[ZTE_SERVER], &error_on_flush);

  switch (ret)
    {
    case G_IO_STATUS_NORMAL:
    case G_IO_STATUS_AGAIN: 
      z_transfer_enable_copy(self, ZTE_SERVER);
      z_proxy_leave(self->owner);
      return TRUE;
    case G_IO_STATUS_EOF:
      if (self->flags & ZTF_SEPARATE_SHUTDOWN)
        {
          z_transfer_shutdown(self, ZTE_DOWN_SERVER, SHUT_RD, NULL);  
          z_transfer_shutdown(self, ZTE_SERVER, SHUT_WR, NULL); 
          z_transfer_update_eof_mask(self, ZTS_EOF_SERVER_W);
        }
      else
        { 
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
      break;
    default:
      if ((self->flags & ZTF_COMPLETE_COPY) && error_on_flush)
        {
          self->flags &= ~ZTF_COPY_TO_SERVER;
          z_transfer_shutdown(self, ZTE_SERVER, SHUT_WR, NULL);
          z_transfer_update_eof_mask(self, ZTS_EOF_SERVER_W);
        }
      else
        {
          z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
        }
//      z_proxy_leave(self->owner);
//      return FALSE;
    }

  z_proxy_leave(self->owner);
  return TRUE;
}

void
z_transfer_disable_copy(ZTransfer *self, guint direction)
{
  if (direction == ZTE_CLIENT)
    {
      z_stream_set_cond(self->endpoints[ZTE_SERVER], Z_STREAM_FLAG_READ, FALSE);
      z_stream_set_cond(self->endpoints[ZTE_CLIENT], Z_STREAM_FLAG_WRITE, FALSE);
      if (self->endpoints[ZTE_DOWN_CLIENT])
        {
          z_stream_set_cond(self->endpoints[ZTE_DOWN_SERVER], Z_STREAM_FLAG_WRITE, FALSE);
          z_stream_set_cond(self->endpoints[ZTE_DOWN_CLIENT], Z_STREAM_FLAG_READ, FALSE);
        }
    }
  else if (direction == ZTE_SERVER)
    {
      z_stream_set_cond(self->endpoints[ZTE_CLIENT], Z_STREAM_FLAG_READ, FALSE);
      z_stream_set_cond(self->endpoints[ZTE_SERVER], Z_STREAM_FLAG_WRITE, FALSE);
      if (self->endpoints[ZTE_DOWN_CLIENT])
        {
          z_stream_set_cond(self->endpoints[ZTE_DOWN_CLIENT], Z_STREAM_FLAG_WRITE, FALSE);
          z_stream_set_cond(self->endpoints[ZTE_DOWN_SERVER], Z_STREAM_FLAG_READ, FALSE);
        }
    }
}

void 
z_transfer_enable_copy(ZTransfer *self, guint direction)
{
  if (direction == ZTE_CLIENT)
    {
      if (self->endpoints[ZTE_DOWN_CLIENT])
        {
          /* we have a stacked proxy */
          
          /* server -> down_server */
          if ((self->status & ZTS_EOF_SERVER_R) == 0)
            {
              if (z_transfer_io_buffer_empty(&self->buffers[ZTE_DOWN_SERVER]))
                {
                  z_stream_set_cond(self->endpoints[ZTE_SERVER], Z_STREAM_FLAG_READ, TRUE);
                }
              else
                {
                  z_stream_set_cond(self->endpoints[ZTE_DOWN_SERVER], Z_STREAM_FLAG_WRITE, TRUE);
                }
            }

          /* down_client -> client */
          if ((self->status & ZTS_EOF_CLIENT_W) == 0)
            {
              if (z_transfer_io_buffer_empty(&self->buffers[ZTE_CLIENT]))
                {
                  z_stream_set_cond(self->endpoints[ZTE_DOWN_CLIENT], Z_STREAM_FLAG_READ, TRUE);
                }
              else
                {
                  z_stream_set_cond(self->endpoints[ZTE_CLIENT], Z_STREAM_FLAG_WRITE, TRUE);
                }
            }
        }
      else  
        {
          /* server -> client */
          if ((self->status & ZTS_EOF_SERVER_R) == 0)
            {
              if (z_transfer_io_buffer_empty(&self->buffers[ZTE_CLIENT]) || (self->status & ZTS_EOF_CLIENT_W) != 0)
                {
                  z_stream_set_cond(self->endpoints[ZTE_SERVER], Z_STREAM_FLAG_READ, TRUE);
                }
              else
                {
                  z_stream_set_cond(self->endpoints[ZTE_CLIENT], Z_STREAM_FLAG_WRITE, TRUE);
                }
            }
        }
    }
  else if (direction == ZTE_SERVER)
    {
      if (self->endpoints[ZTE_DOWN_CLIENT])
        {
          /* we have a stacked proxy */
          
          /* client -> down_client */
          if ((self->status & ZTS_EOF_CLIENT_R) == 0)
            {
              if (z_transfer_io_buffer_empty(&self->buffers[ZTE_DOWN_CLIENT]))
                {
                  z_stream_set_cond(self->endpoints[ZTE_CLIENT], Z_STREAM_FLAG_READ, TRUE);
                }
              else
                {
                  z_stream_set_cond(self->endpoints[ZTE_DOWN_CLIENT], Z_STREAM_FLAG_WRITE, TRUE);
                }
            }

          /* down_server -> server */
          if ((self->status & ZTS_EOF_SERVER_W) == 0)
            {
              if (z_transfer_io_buffer_empty(&self->buffers[ZTE_SERVER]))
                {
                  z_stream_set_cond(self->endpoints[ZTE_DOWN_SERVER], Z_STREAM_FLAG_READ, TRUE);
                }
              else
                {
                  z_stream_set_cond(self->endpoints[ZTE_SERVER], Z_STREAM_FLAG_WRITE, TRUE);
                }
            }
        }
      else  
        {
          /* client -> server */
          if ((self->status & ZTS_EOF_CLIENT_R) == 0)
            {
              if (z_transfer_io_buffer_empty(&self->buffers[ZTE_SERVER]) || (self->status & ZTS_EOF_SERVER_W) != 0)
                {
                  z_stream_set_cond(self->endpoints[ZTE_CLIENT], Z_STREAM_FLAG_READ, TRUE);
                }
              else
                {
                  z_stream_set_cond(self->endpoints[ZTE_SERVER], Z_STREAM_FLAG_WRITE, TRUE);
                } 
            }
        }
    }
}

static gboolean
z_transfer_timeout(gpointer user_data)
{
  ZTransfer *self = Z_CAST(user_data, ZTransfer);
  
  z_proxy_enter(self->owner);
  /*LOG
    This message indicates that data transfer timed out and proxy is closing connection.
   */
  z_proxy_log(self->owner, CORE_ERROR, 3, "Data transfer timed out; timeout='%ld'", self->timeout);
  z_transfer_update_eof_mask(self, ZTS_EOF_ALL);
  z_proxy_leave(self->owner);
  return FALSE;
}

gboolean
z_transfer_init_streams(ZTransfer *self)
{
  z_proxy_enter(self->owner);
 
  self->buffers[ZTE_CLIENT].buf = g_new0(char, self->buffer_size);
  self->buffers[ZTE_SERVER].buf = g_new0(char, self->buffer_size);
  
  
  if (self->flags & ZTF_COPY_TO_SERVER)
    {
      z_stream_set_callback(self->endpoints[ZTE_CLIENT],
                            Z_STREAM_FLAG_READ, 
                            z_transfer_copy_client_to_server,
                            self,
                            NULL);
      z_stream_set_callback(self->endpoints[ZTE_SERVER],
                            Z_STREAM_FLAG_WRITE,
                            z_transfer_copy_client_to_server,
                            self,
                            NULL);

      z_stream_set_cond(self->endpoints[ZTE_CLIENT],
                       Z_STREAM_FLAG_READ,
                       TRUE);
    }
  else
    {
      self->status |= ZTS_EOF_CLIENT_R | ZTS_EOF_SERVER_W;
    }
    
  z_stream_set_timeout(self->endpoints[ZTE_CLIENT], -2);
  
  
  if (self->flags & ZTF_COPY_TO_CLIENT)
    {
      z_stream_set_callback(self->endpoints[ZTE_SERVER],
                            Z_STREAM_FLAG_READ,
                            z_transfer_copy_server_to_client,
                            self,
                            NULL);
      z_stream_set_callback(self->endpoints[ZTE_CLIENT],
                            Z_STREAM_FLAG_WRITE,
                            z_transfer_copy_server_to_client,
                            self,
                            NULL);
      z_stream_set_cond(self->endpoints[ZTE_SERVER],
                       Z_STREAM_FLAG_READ,
                       TRUE);
    }
  else
    {
      self->status |= ZTS_EOF_CLIENT_W | ZTS_EOF_SERVER_R;
    }
  z_stream_set_timeout(self->endpoints[ZTE_SERVER], -2);

  z_proxy_leave(self->owner);
  return TRUE;

}

static gboolean
z_transfer_init_stacked_streams(ZTransfer *self)
{
  z_proxy_enter(self->owner);
  
  if (self->endpoints[ZTE_DOWN_CLIENT] && self->endpoints[ZTE_DOWN_SERVER])
    {
      self->buffers[ZTE_DOWN_CLIENT].buf = g_new0(char, self->buffer_size);
      self->buffers[ZTE_DOWN_SERVER].buf = g_new0(char, self->buffer_size);

      z_stream_set_nonblock(self->endpoints[ZTE_DOWN_CLIENT], TRUE);
      z_stream_set_nonblock(self->endpoints[ZTE_DOWN_SERVER], TRUE);

      if (self->flags & ZTF_COPY_TO_SERVER)
        {
          z_stream_set_callback(self->endpoints[ZTE_CLIENT],
                                Z_STREAM_FLAG_READ,
                                z_transfer_copy_client_to_down,
                                self,
                                NULL);

          z_stream_set_callback(self->endpoints[ZTE_DOWN_CLIENT],
                                Z_STREAM_FLAG_WRITE,
                                z_transfer_copy_client_to_down,
                                self, 
                                NULL);

          z_stream_set_callback(self->endpoints[ZTE_DOWN_SERVER],
                                Z_STREAM_FLAG_READ,
                                z_transfer_copy_down_to_server,
                                self,
                                NULL);

          z_stream_set_callback(self->endpoints[ZTE_SERVER],
                                Z_STREAM_FLAG_WRITE,
                                z_transfer_copy_down_to_server,
                                self,
                                NULL);

          z_stream_set_cond(self->endpoints[ZTE_DOWN_SERVER],
                           Z_STREAM_FLAG_READ,
                           TRUE);
        }
      
      if (self->flags & ZTF_COPY_TO_CLIENT)
        {
          z_stream_set_callback(self->endpoints[ZTE_SERVER],
                                Z_STREAM_FLAG_READ,
                                z_transfer_copy_server_to_down,
                                self,
                                NULL);

          z_stream_set_callback(self->endpoints[ZTE_DOWN_SERVER],
                                Z_STREAM_FLAG_WRITE,
                                z_transfer_copy_server_to_down,
                                self,
                                NULL);

          z_stream_set_callback(self->endpoints[ZTE_CLIENT],
                                Z_STREAM_FLAG_WRITE,
                                z_transfer_copy_down_to_client,
                                self,
                                NULL);
          z_stream_set_callback(self->endpoints[ZTE_DOWN_CLIENT],
                                Z_STREAM_FLAG_READ,
                                z_transfer_copy_down_to_client,
                                self, 
                                NULL);

          z_stream_set_cond(self->endpoints[ZTE_DOWN_CLIENT],
                           Z_STREAM_FLAG_READ,
                           TRUE);

        }
        
    }
  z_proxy_leave(self->owner);
  return TRUE;
}


gboolean
z_transfer_run(ZTransfer *self)
{
  gint i;
  
  z_proxy_enter(self->owner);

  for (i = 0; i <= ZTE_MAX; i++)
    {
      if (self->endpoints[i])
        {
          z_stream_save_context(self->endpoints[i], &self->contexts[i]);
          if (i & ZTE_STACKED ||
              !(self->flags & ZTF_PROXY_STREAMS_POLLED))
            {
              z_poll_add_stream(self->poll, self->endpoints[i]);
            }
        }
    }
  z_stream_set_nonblock(self->endpoints[ZTE_CLIENT], TRUE);
  z_stream_set_nonblock(self->endpoints[ZTE_SERVER], TRUE);
  
  if (!z_transfer_init_streams(self) || !z_transfer_init_stacked_streams(self))
    {
      self->status |= ZTS_FAILED;
      goto restore_streams;
    }

  z_transfer_update_eof_mask(self, 0);
  if (self->timeout > 0)
    {
      GMainContext *context;
         
      self->timeout_source = z_timeout_source_new(self->timeout);
      g_source_set_callback(self->timeout_source, z_transfer_timeout, self, NULL);
      context = z_poll_get_context(self->poll);
      g_source_attach(self->timeout_source, context);
    }
  z_transfer_progress(self, ZTP_STARTUP);
  while (((self->status & ZTS_FINISHED) == 0) && z_poll_iter_timeout(self->poll, -1))
    {
      ;
    }
  z_transfer_shutdown(self, ZTE_CLIENT, SHUT_RD, NULL);
  z_transfer_shutdown(self, ZTE_CLIENT, SHUT_WR, NULL);
  z_transfer_shutdown(self, ZTE_SERVER, SHUT_RD, NULL);
  z_transfer_shutdown(self, ZTE_SERVER, SHUT_WR, NULL);
  

 restore_streams:

  for (i = 0; i <= ZTE_MAX; i++)
    {
      if (self->endpoints[i])
        {
          if (i & ZTE_STACKED ||
              !(self->flags & ZTF_PROXY_STREAMS_POLLED))
            {
              z_poll_remove_stream(self->poll, self->endpoints[i]);
            }
          z_stream_restore_context(self->endpoints[i], &self->contexts[i]);
        }
    }

  z_proxy_leave(self->owner);
  return !(self->status & ZTS_FAILED);
}

/**
 * z_transfer_set_stacked_proxy:
 * @self: ZTransfer instance
 * @proxy: stacked proxy streams
 *
 * This function must be called prior to z_transfer_run() to set up stacked
 * proxy streams.
 **/
void
z_transfer_set_stacked_proxy(ZTransfer *self, ZStackedProxy *stacked)
{
  z_proxy_enter(self->owner);
  if (stacked)
    {
      if (self->stacked)
        {
          z_stacked_proxy_destroy(self->stacked);
        }
      z_stream_unref(self->endpoints[ZTE_DOWN_CLIENT]);
      z_stream_unref(self->endpoints[ZTE_DOWN_SERVER]);
      
      self->endpoints[ZTE_DOWN_CLIENT] = z_stream_ref(stacked->downstreams[EP_CLIENT]);
      self->endpoints[ZTE_DOWN_SERVER] = z_stream_ref(stacked->downstreams[EP_SERVER]);
      self->stacked = stacked;
    }
  z_proxy_leave(self->owner);
}

/**
 * z_transfer_new:
 * @class: instantiate this class (must be derived from ZTransfer)
 * @owner: owner ZProxy instance
 * @poll: a poll loop to use for transfer purposes
 * @client: source stream
 * @server: destination stream
 * @stacked: stacked proxy information
 * @buffer_size: buffer size to use
 * @timeout: transfer I/O timeout in milliseconds
 * @flags: misc flags (one of ZTF_*)
 *
 * This function is the ZTransfer constructor, allocates and initializes the
 * instance structure based on its parameters.
 **/
ZTransfer *
z_transfer_new(ZClass *class, 
               ZProxy *owner, 
               ZPoll *poll, 
               ZStream *client, ZStream *server, 
               ZStackedProxy *stacked,
               gsize buffer_size, 
               glong timeout,
               gulong flags)
{
  ZTransfer *self;
  

  z_proxy_enter(owner);
  self = Z_NEW_COMPAT(class, ZTransfer);
  
  self->owner = z_proxy_ref(owner);
  
  z_poll_ref(poll);
  self->poll = poll;
  self->endpoints[ZTE_CLIENT] = z_stream_ref(client);
  self->endpoints[ZTE_SERVER] = z_stream_ref(server);
  z_transfer_set_stacked_proxy(self, stacked);
  self->buffer_size = buffer_size;
  self->timeout = timeout;
  self->flags = flags;

  z_proxy_leave(self->owner);
  return self;
}

void
z_transfer_free_method(ZObject *s)
{
  ZTransfer *self = Z_CAST(s, ZTransfer);
  gint i;
  
  z_enter();
  if (self->timeout_source)
    {
      g_source_destroy(self->timeout_source);
      g_source_unref(self->timeout_source);
      self->timeout_source = NULL;
    }
  for (i = 0; i <= ZTE_MAX; i++)
    {
      z_stream_unref(self->endpoints[i]);
      if (self->buffers[i].buf)
        g_free(self->buffers[i].buf);
    }
  z_proxy_unref(self->owner);
  if (self->stacked)
    {
      z_stacked_proxy_destroy(self->stacked);
    }
  z_poll_unref(self->poll);
  z_object_free_method(s);
  z_leave();
}

ZTransferFuncs z_transfer_funcs =
{
  {
    Z_FUNCS_COUNT(ZTransfer),
    z_transfer_free_method,
  },
  z_transfer_src_read_method,
  z_transfer_src_write_method,
  z_transfer_src_shutdown_method,
  z_transfer_dst_read_method,
  z_transfer_dst_write_method,
  z_transfer_dst_shutdown_method,
  z_transfer_progress_method
};

ZClass ZTransfer__class = 
{
  Z_CLASS_HEADER,
  &ZObject__class,
  "ZTransfer",
  sizeof(ZTransfer),
  &z_transfer_funcs.super
};
