# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core.utils.i18n import install_translation
_ = install_translation('database')

from twisted.internet import defer

from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.link import Link
from elisa.plugins.base.models.image import ImageModel

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.poblesec.base.list import  GenericListViewMode

from elisa.plugins.database.models import Image, PhotoAlbum
from elisa.core import common
from elisa.core.media_uri import MediaUri

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.poblesec.actions import ViewSlideshowAction
from elisa.plugins.database.actions import PhotoAddToFavoritesAction
from elisa.plugins.database.actions import ViewPhotoAlbumSlideshowAction

from elisa.plugins.favorites.models import FavoritesItem

import os
import os.path


def photo_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/photo_library'
    link.label = _('Photo Library')
    link.icon = 'elisa.plugins.poblesec.photo_library'
    controller.model.append(link)

    return defer.succeed(None)

def photo_lib_albums_decorator(controller):
    albums = Link()
    albums.controller_path = '/poblesec/database/photo/albums'
    albums.label = _('Browse by Albums')
    albums.icon = 'elisa.plugins.poblesec.photo_album'
    controller.model.append(albums)
    return defer.succeed(None)


def photo_lib_images_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/photo/list'
    tracks.label = _('Browse all Photos')
    tracks.icon = 'elisa.plugins.poblesec.photo'
    controller.model.append(tracks)
    return defer.succeed(None)


def photo_lib_times_decorator(controller):
    times = Link()
    times.controller_path = '/poblesec/database/photo/by_time'
    times.label = _('By time')
    times.icon = 'elisa.plugins.poblesec.by_genre' # FIXME
    controller.model.append(times)
    return defer.succeed(None)

class ThumbnailMixin(object):

    def __init__(self):
        # FIXME: we need the frontend to get a reference to the gst_metadata
        # instance. This a cheap - UGLY - way to get the frontend without
        # changing a lot of client code. It is really ugly as we assume there
        # is only one frontend, which might not be the case in the future...
        frontend = common.application.interface_controller.frontends.values()[0]
        # Retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        self.gst_metadata = controllers[0].gst_metadata

    def _updated_thumbnail(self, thumbnail, item):
        item.thumbnail = thumbnail
        return thumbnail

    def _request_thumbnail(self, path):
        def got_metadata(metadata):
            return metadata['thumbnail']

        metadata = {'uri': MediaUri("file://%s" % path), 'thumbnail': None,}

        dfr = self.gst_metadata.get_metadata(metadata)
        dfr.addCallback(got_metadata)
        return dfr

class PhotoViewMode(GenericListViewMode, ThumbnailMixin):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return defer.succeed(os.path.basename(item.file_path))

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.photo'
        return resource

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None

        if hasattr(item, 'thumbnail') and item.thumbnail != None:
            return defer.succeed(item.thumbnail)

        dfr = self._request_thumbnail(item.file_path)
        dfr.addCallback(self._updated_thumbnail, item)
        return dfr

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None

        if hasattr(item, 'thumbnail'):
            return item.thumbnail


class GenericPhotosController(HierarchyController):

    def initialize(self, album=None, photo=None, photos=None):
        self.album = album
        self.photo = photo
        self.photos = photos

        dfr = super(GenericPhotosController, self).initialize()
        return dfr

    def make_actions(self):
        actions = []

        if self.album:
            action = ViewSlideshowAction(self)
            action.title = _('Play Slideshow')
            action.subtitle = _('Slideshow of All Photos')
            actions.append(action)

            action = PhotoAddToFavoritesAction(self)
            action.untoggled_title = _('Add Album To Favorites')
            action.untoggled_subtitle = _("Add '%s' To Your Favorites") % self.album.name
            action.toggled_title = _('Remove Album From Favorites')
            action.toggled_subtitle = _("Remove '%s' From Your Favorites") % self.album.name
            actions.append(action)

            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.foreign_id == self.album.name,
                                                FavoritesItem.foreign_class == u'PhotoAlbum')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        elif self.photo:
            action = ViewSlideshowAction(self)
            action.title = _('View Photo')
            action.subtitle = _("View '%s'") % os.path.basename(self.photo.file_path)
            actions.append(action)

            action = PhotoAddToFavoritesAction(self)
            action.untoggled_title = _('Add Photo To Favorites')
            action.untoggled_subtitle = _("Add '%s' To Your Favorites") % os.path.basename(self.photo.file_path)
            action.toggled_title = _('Remove From Favorites')
            action.toggled_subtitle = _("Remove '%s' From Your Favorites") % os.path.basename(self.photo.file_path)
            actions.append(action)

            dfr = common.application.store.find(FavoritesItem,
                                                FavoritesItem.foreign_id == self.photo.file_path,
                                                FavoritesItem.foreign_class == u'Image')
            dfr.addCallback(lambda rs: rs.all())
            dfr.addCallback(action.setup)
        else:
            action = ViewSlideshowAction(self)
            action.title = _('Play Slideshow')
            action.sub_title = _('Slideshow of All Photos')
            actions.append(action)

        return actions

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(model, Action):
                model.run()
            else:
                browser = self.frontend.retrieve_controllers('/poblesec/browser')[0]
                photos = filter(lambda p: not isinstance(p, Action), self.model)
                title = os.path.basename(model.file_path)
                dfr = browser.history.append_controller('/poblesec/database/photo/photo_actions',
                                                        title,
                                                        photo=model,
                                                        photos=photos)

class PhotosController(GenericPhotosController):

    def initialize(self, album=None, photo=None, photos=None):
        dfr = super(PhotosController, self).initialize(album=album, photo=photo, photos=photos)

        def get_images(result):
            store = common.application.store

            if self.album:
                dfr = store.find(Image, Image.album_name == album.name)
            else:
                dfr = store.find(Image)

            return dfr

        def got_images(images):
            self.model.extend(images)
            return self

        def sort_images(result_set):
            result_set.order_by(Image.shot_time, Image.file_path)
            return result_set.all()

        if photo is None:
            dfr.addCallback(get_images)
            dfr.addCallback(sort_images)
            dfr.addCallback(got_images)

        return dfr


class PhotoAlbumViewMode(GenericListViewMode, ThumbnailMixin):

    def get_label(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.title)
        return defer.succeed(item.name)

    def get_default_image(self, item):
        if isinstance(item, Action):
            return item.icon
        resource = 'elisa.plugins.poblesec.photo_album'
        return resource

    def _retrieve_file_path(self, album):
        def got_one(one):
            return one.file_path

        dfr = album.photos.any()
        dfr.addCallback(got_one)
        return dfr

    def _got_thumbnail(self, thumbnail, item):
        item.thumbnail = thumbnail
        return None

    def get_image(self, item, theme):
        if isinstance(item, Action):
            return None

        if hasattr(item, 'thumbnail') and item.thumbnail != None:
            return defer.succeed(item.thumbnail)

        dfr = self._retrieve_file_path(item)
        dfr.addCallback(self._request_thumbnail)
        dfr.addCallback(self._got_thumbnail, item)
        return dfr

    def get_preview_image(self, item, theme):
        if isinstance(item, Action):
            return None
        if hasattr(item, 'thumbnail'):
            return item.thumbnail


class GenericPhotoAlbumsController(HierarchyController):

    def initialize(self):
        dfr = super(GenericPhotoAlbumsController, self).initialize()
        return dfr

    def make_actions(self):
        actions = []

        action = ViewPhotoAlbumSlideshowAction(self)
        action.title = _('Play Slideshow')
        action.sub_title = _('Slideshow of All Photos')
        actions.append(action)

        return actions

     # copy-paste-code again :(
    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(model, Action):
                model.run()
            else:
                controllers = self.frontend.retrieve_controllers('/poblesec/browser')
                history = controllers[0].history
                dfr = history.append_controller('/poblesec/database/photo/list',
                                                        model.name, album=model)


class PhotoAlbumsController(GenericPhotoAlbumsController):

    def initialize(self):
        dfr = super(PhotoAlbumsController, self).initialize()

        def get_albums(self):
            store = common.application.store
            dfr = store.find(PhotoAlbum)
            return dfr

        def got_albums(albums):
            self.model.extend(albums)
            return self

        def sort_albums(result_set):
            result_set.order_by(PhotoAlbum.name)
            return result_set.all()

        dfr.addCallback(get_albums)
        dfr.addCallback(sort_albums)
        dfr.addCallback(got_albums)

        return dfr


# Photo
class PhotoVerticalWithPreview(PhotosController, MenuItemPreviewListController):
    view_mode = PhotoViewMode

class PhotoCoverflow(PhotosController, ImageWithReflectionCoverflowController):
    view_mode = PhotoViewMode

class PhotoGrid(PhotosController, GridItemGridController):
    view_mode = PhotoViewMode

class PhotoListSwitcherController(ListSwitcherController):
    modes = [PhotoVerticalWithPreview,
             PhotoCoverflow,
             PhotoGrid]
    default_mode = PhotoGrid

# Albums
class AlbumsVerticalWithPreview(PhotoAlbumsController, MenuItemPreviewListController):
    view_mode = PhotoAlbumViewMode

class AlbumsCoverflow(PhotoAlbumsController, ImageWithReflectionCoverflowController):
    view_mode = PhotoAlbumViewMode

class AlbumsGrid(PhotoAlbumsController, GridItemGridController):
    view_mode = PhotoAlbumViewMode

class AlbumsListSwitcherController(ListSwitcherController):

    modes = [AlbumsVerticalWithPreview,
             AlbumsCoverflow,
             AlbumsGrid]
    default_mode = AlbumsVerticalWithPreview
