# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

"""
Component instances management
"""


__maintainer__ = 'Philippe Normand <philippe@fluendo.com>'

from elisa.core import log
from elisa.core.utils import classinit
from elisa.core import common
from twisted.internet import defer, task

class AlreadyRegistered(Exception):
    pass

class CannotUnregister(Exception):
    pass

class Manager(log.Loggable):
    """ A Manager is a Component container

    Components can be registered and unregistered from the
    Manager.

    Optionnally the manager can implement start/stop methods if it
    needs to handle any kind of loop (example: media sources scanning,
    input events polling, etc). start/stop methods are called by the
    parent object (application).

    @ivar _components: Components currently registered in the manager
    @type _components: L{elisa.core.component.Component} list
    """

    # Allows property fget/fset/fdel/doc overriding
    __metaclass__ = classinit.ClassInitMeta
    __classinit__ = classinit.build_properties

    def __init__(self):
        """ Initialize the _components instance variable and the
        Manager as a Loggable object.
        """
        log.Loggable.__init__(self)
        self.debug("Creating")

        self._components = []

    def load_components(self, components_names):
        """
        Load a list of components in sequence.

        @param  components_names:    list of strings with the names of
                                     components to create
        @type   components_names:    list
        """
        application = common.application
        plugin_registry = application.plugin_registry

        def load_components_iter(loaded):
            def create_component_done(component, component_name):
                self.debug("Loaded provider %s" % component_name)
                self.register_component(component)
                return (True, component)

            def create_component_failure(failure, component_name):
                log_path = common.application.log_failure(failure)
                self.warning("Creating %s failed. A full traceback can"
                             " be found at %s" % (component_name, log_path))
                return (False, failure)

            for component_name in components_names:
                res = plugin_registry.create_component(component_name)
                res.addCallback(create_component_done, component_name)
                res.addErrback(create_component_failure, component_name)
                res.addCallback(loaded.append)

                yield res

        def load_components_iter_done(iter, loaded):
            really_loaded = filter(lambda x: x[0], loaded)
            self.debug("Loaded %s providers" % len(really_loaded))
            return loaded

        loaded = []
        res_dfr = task.coiterate(load_components_iter(loaded))
        res_dfr.addCallback(load_components_iter_done, loaded)

        return res_dfr

    def start(self):
        """ Start a loop or something to initialize the Manager. Can
        also look for new components to register in this method.

        Does nothing by default, override if needed.
        """

    def stop(self):
        """ Stop clean and remove all registered components.

            @rtype: L{twisted.internet.defer.DeferredList}
        """
        deferreds = []
        for component in self._components:
            deferreds.append(defer.maybeDeferred(component.clean))
        self._components = []

        return defer.DeferredList(deferreds)


    def register_component(self, component):
        """ Register a new Component

        Store a new Component in our components list. Returns the
        result of the operation. If the component is already
        registered, don't register it twice.

        @param component: the Component to register
        @type component:  L{elisa.core.component.Component}

        @raise AlreadyRegistered : when the component has already
                                   been registered
        @raise TypeError:          when the given L{component} is *not* of the
                                   type L{supported_components}
        """
        if component in self._components:
            self.debug("Component %s already registered" % component)
            raise AlreadyRegistered()
        else:
            self.debug("Registering component %s" % component)
            self._components.append(component)

    def unregister_component(self, component):
        """ Unregister a component from the Manager

        Remove the Component instance from our components list if it's
        there. Returns the result of the operation.

        @param component: the Component to register
        @type component:  L{elisa.core.component.Component}

        @raise CannotUnregister : raised when the component cannot be removed
        """

        if component in self._components:
            self.debug("Unregistering component %s" % component)
            self._components.remove(component)
        else:
            self.debug("Impossible to unregister component %s" % component)
            raise CannotUnregister()
