/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <config.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <gio/gio.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#define LAUNCHER_NAME    "ume-launcher"
#define LAUNCHER_DESKTOP LAUNCHER_NAME".desktop"
#define SHARE_LAUNCHER PKGDATADIR"/"LAUNCHER_DESKTOP

#define SHARE_CLASSIC_PANEL_CONFIG PKGDATADIR"/classic-panel-config.xml"
#define SHARE_NETBOOK_PANEL_CONFIG PKGDATADIR"/netbook-panel-config.xml"

#define NAUTILUS_SHOW_DESKTOP "/apps/nautilus/preferences/show_desktop"
#define THEME "/desktop/gnome/interface/gtk_theme"

#define GCONF_STOP  "gconftool-2 --shutdown"
#define GCONF_UNSET "gconftool-2 --recursive-unset /apps/panel"
#define GCONF_LOAD  "gconftool-2 --load "

enum
{
  NETBOOK_MODE = 0,
  CLASSIC_MODE
};

static gboolean create_dir (const gchar *dir);


static gint current_mode = NETBOOK_MODE;
static gint selected_mode = NETBOOK_MODE;

gint
main (gint argc, gchar *argv[])
{
  GtkBuilder *builder;
  GError     *error = NULL;
  GtkWidget  *dialog;
  GtkWidget  *netbook_radio;
  GtkWidget  *classic_radio;
  gchar      *autostart; 
  
  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  g_thread_init (NULL);
  g_type_init ();
  g_set_application_name ("Desktop Switcher");

  gtk_init (&argc, &argv);

  /* Load the user interface */
  builder = gtk_builder_new ();
  gtk_builder_add_from_file (builder, PKGDATADIR"/ui.xml", &error);
  if (error)
  {
    g_warning ("Unable to load ui: %s\n", error->message);
    g_error_free (error);
    return EXIT_FAILURE;
  }
  gtk_builder_set_translation_domain (builder, GETTEXT_PACKAGE);

  /* Make sure the autostart directory exists */
  autostart = g_build_filename (g_get_home_dir (), ".config","autostart", NULL);
  create_dir (autostart);
  g_free (autostart);

  /* Get all the widgets */
  dialog = GTK_WIDGET (gtk_builder_get_object (builder, "dialog1"));
  netbook_radio = GTK_WIDGET (gtk_builder_get_object (builder, "netbookradio"));
  classic_radio = GTK_WIDGET (gtk_builder_get_object (builder, "classicradio"));

  /* Check which mode we're in*/
  /*FIXME: We need to do a better check */
  autostart = g_build_filename (g_get_home_dir (), 
                                ".config", "autostart", 
                                LAUNCHER_DESKTOP,
                                NULL);
  if (g_file_test (autostart, G_FILE_TEST_EXISTS))
  {
    g_object_set (classic_radio, "active", TRUE, NULL);
    current_mode = selected_mode = CLASSIC_MODE;
  }
  else
  {
    g_object_set (netbook_radio, "active", TRUE, NULL);  
  }
  g_free (autostart);

  /* Connect up the signals & show the window */
  gtk_builder_connect_signals (builder, NULL);
  gtk_widget_show_all (dialog);

  gtk_main ();

  return EXIT_SUCCESS;
}

static gboolean
create_dir (const gchar *dir)
{
  if (!g_file_test (dir, G_FILE_TEST_EXISTS))
  {
    g_mkdir_with_parents (dir, 0700);
    return TRUE;
  }
  return FALSE;
}

void
on_netbookradio_toggled (GtkToggleButton *button)
{
  if (gtk_toggle_button_get_active (button))
    selected_mode = NETBOOK_MODE;
}

void
on_classicradio_toggled (GtkToggleButton *button)
{
  if (gtk_toggle_button_get_active (button))
    selected_mode = CLASSIC_MODE;
}

static gchar *
launcher_autostart_file (void)
{
  return g_build_filename (g_get_home_dir (),
                               ".config", "autostart",
                               LAUNCHER_DESKTOP,
                               NULL);
}

static void
enable_netbook (void)
{
  GFile *file;
  gchar *filename;
  GError *error = NULL;
  GConfClient *client = gconf_client_get_default ();
  GdkScreen *screen = gdk_screen_get_default ();

  g_debug ("Enabling netbook mode");
  
  /* Make sure the launcher autostarts */
  g_debug ("Enabling launcher autostart");
  filename = launcher_autostart_file ();
  file = g_file_new_for_path (filename);
  g_file_delete (file, NULL, &error);
  if (error)
  { 
    g_warning ("Unable to remove launcher autostart suppression file: %s",
               error->message);
    g_error_free (error);
  }
  g_object_unref (file);
  g_free (filename);

  /* Tell nautilus not to draw the desktop */
  error = NULL;
  gconf_client_set_bool (client, NAUTILUS_SHOW_DESKTOP, FALSE, &error);
  if (error)
  {
    g_warning ("Can't stop Nautilus from drawing the desktop: %s", 
               error->message);
    g_error_free (error);
    error = NULL;
  }

  /* Set the theme */
  error = NULL;
  gconf_client_set_string (client, THEME, "Human-Murrine-Netbook", &error);
  if (error)
  {
    g_warning ("Can't change the theme: %s", 
               error->message);
    g_error_free (error);
    error = NULL;
  }

  /* Launch the launcher */
  g_debug ("Launching the, er, Launcher");
  gdk_spawn_command_line_on_screen (screen, LAUNCHER_NAME, NULL);

  /* Set the panel to the netbook layout */
  g_debug ("Restoring panel settings");
  gdk_spawn_command_line_on_screen (screen, GCONF_STOP, NULL);
  gdk_spawn_command_line_on_screen (screen, GCONF_UNSET, NULL);
  gdk_spawn_command_line_on_screen (screen, 
                                    GCONF_LOAD SHARE_NETBOOK_PANEL_CONFIG,
                                    NULL);
  gdk_spawn_command_line_on_screen (screen, "killall gnome-panel", NULL);
  gdk_spawn_command_line_on_screen (screen, "gnome-panel", NULL);
 
  g_object_unref (client);
}
  
static void
enable_classic ()
{
  GFile *src, *dest;
  GError *error = NULL;
  gchar *filename;
  GConfClient *client = gconf_client_get_default ();
  GdkScreen *screen = gdk_screen_get_default ();

  g_debug ("Enabling classic mode");

  /* Stop the launcher from autostarting */
  g_debug ("Suppressing Launcher autostart");
  filename = launcher_autostart_file ();
  src = g_file_new_for_path (SHARE_LAUNCHER);
  dest = g_file_new_for_path (filename);
  g_file_copy (src, dest, 0, NULL, NULL, NULL, &error);
  if (error)
  {
    g_debug ("Unable to suppress launcher autostart: %s", error->message);
    g_error_free (error);
  }
  g_object_unref (dest);
  g_object_unref (src);
  g_free (filename);

  /* Tell nautilus to draw the desktop */
  error = NULL;
  gconf_client_set_bool (client, NAUTILUS_SHOW_DESKTOP, TRUE, &error);
  if (error)
  {
    g_warning ("Can't make Nautilus draw the desktop: %s", 
               error->message);
    g_error_free (error);
    error = NULL;
  }
  gdk_spawn_command_line_on_screen (screen, "nautilus -n", NULL);

  /* Set the theme */
  error = NULL;
  gconf_client_set_string (client, THEME, "Human", &error);
  if (error)
  {
    g_warning ("Can't change the theme: %s", 
               error->message);
    g_error_free (error);
    error = NULL;
  }

  /* Kill the launcher */
  g_debug ("Killing the launcher");
  gdk_spawn_command_line_on_screen (screen, "killall "LAUNCHER_NAME, NULL);

  /* Set the panel to the netbook layout */
  g_debug ("Restoring panel settings");
  gdk_spawn_command_line_on_screen (screen, GCONF_STOP, NULL);
  gdk_spawn_command_line_on_screen (screen, GCONF_UNSET, NULL);
  gdk_spawn_command_line_on_screen (screen, 
                                    GCONF_LOAD SHARE_CLASSIC_PANEL_CONFIG,
                                    NULL);
  gdk_spawn_command_line_on_screen (screen, "killall gnome-panel", NULL);
  gdk_spawn_command_line_on_screen (screen, "gnome-panel", NULL);  


  g_object_unref (client);
}

void
on_apply_clicked (GtkButton *button)
{
  if (current_mode == selected_mode)
    gtk_main_quit ();
  else if (selected_mode == NETBOOK_MODE)
    enable_netbook ();
  else
    enable_classic ();

  gtk_main_quit ();
}
