/*
 * Copyright (C) 2004-2008 Geometer Plus <contact@geometerplus.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#include <ZLTextParagraph.h>

#include "ContentsView.h"
#include "BookTextView.h"
#include "FBReader.h"

#include "../bookmodel/BookModel.h"

ContentsView::ContentsView(FBReader &reader, shared_ptr<ZLPaintContext> context) : FBTextView(reader, context), myMouseButtonPressed(false) {
}

ContentsView::~ContentsView() {
}

bool ContentsView::onStylusMovePressed(int x, int y) {
	return myMouseButtonPressed ? FBTextView::onStylusMovePressed(x, y) : false;
}

bool ContentsView::onStylusRelease(int x, int y) {
	if (myMouseButtonPressed) {
		myMouseButtonPressed = false;
		return FBTextView::onStylusMovePressed(x, y);
	}
	return false;
}

bool ContentsView::onStylusMove(int x, int y) {
	if (FBTextView::onStylusMove(x, y)) {
		return false;
	}

	int index = paragraphIndexByCoordinates(x, y);
	if ((index < 0) || (index >= (int)model()->paragraphsNumber())) {
		fbreader().setHyperlinkCursor(false);
		return true;
	}

	const ContentsModel &contentsModel = (const ContentsModel&)*model();
	const ZLTextTreeParagraph *paragraph = (const ZLTextTreeParagraph*)contentsModel[index];
	
	fbreader().setHyperlinkCursor(contentsModel.reference(paragraph) >= 0);

	return true;
}

bool ContentsView::_onStylusPress(int x, int y) {
	int index = paragraphIndexByCoordinates(x, y);
	if ((index >= 0) && (index < (int)model()->paragraphsNumber())) {
		const ContentsModel &contentsModel = (const ContentsModel&)*model();
		const ZLTextTreeParagraph *paragraph = (const ZLTextTreeParagraph*)contentsModel[index];
		
		int reference = contentsModel.reference(paragraph);

		if (reference >= 0) {
			fbreader().bookTextView().gotoParagraph(reference);
			fbreader().showBookTextView();
			return true;
		}
	}

	myMouseButtonPressed = true;
	return FBTextView::_onStylusPress(x, y);
}

bool ContentsView::isEmpty() const {
	return (model() == 0) || (model()->paragraphsNumber() == 0);
}

size_t ContentsView::currentTextViewParagraph(bool includeStart) const {
	const ZLTextWordCursor &cursor = fbreader().bookTextView().startCursor();
	if (!cursor.isNull()) {
		long reference = cursor.paragraphCursor().index();
		bool startOfParagraph = cursor.elementIndex() == 0;
		if (cursor.isEndOfParagraph()) {
			++reference;
			startOfParagraph = true;
		}
		size_t length = model()->paragraphsNumber();
		const ContentsModel &contentsModel = (const ContentsModel&)*model();
		for (size_t i = 1; i < length; ++i) {
			long contentsReference =
				contentsModel.reference(((const ZLTextTreeParagraph*)contentsModel[i]));
			if ((contentsReference > reference) ||
					(!includeStart && startOfParagraph && (contentsReference == reference))) {
				return i - 1;
			}
		}
		return length - 1;
	}
	return (size_t)-1;
}

void ContentsView::gotoReference() {
	model()->removeAllMarks();
	const size_t selected = currentTextViewParagraph();
	highlightParagraph(selected);
	gotoParagraph(selected);
	scrollPage(false, ZLTextView::SCROLL_PERCENTAGE, 40);
}

bool ContentsView::isSelectionEnabled() const {
	return false;
}
