/****************************************************************************

 * Copyright (c) 2008-2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.

File Name:          PMMenu.cpp

Description:

    This file implements class PMMenu.

Environment (opt):

    OS: Ubuntu,
    SE: Code::Blocks8.02

Notes (opt):

  =====================================================================

  Revision   Revision History               Author     Date

  =====================================================================

  0.1        Create                         Zhang Hui   2008-7-20

  =====================================================================

****************************************************************************/

#include "include/PMMenu.h"
#include "include/DataMiddle.h"

BEGIN_EVENT_TABLE(PMMenu, wxMenu)
    EVT_MENU(MENU_BATTERY_INFO, PMMenu::OpenBatteryInfoDlg)
    EVT_MENU(MENU_PM_DLG,  PMMenu::OpenPMDialog)
    EVT_MENU(MENU_EXIT,PMMenu::ExitApp)
END_EVENT_TABLE()

/**constructor*/
PMMenu::PMMenu(const wxString& title, long style) : wxMenu(title, style)
{
}

/**remove all menu items*/
void PMMenu::DestroyItems()
{
    wxList list = this->GetMenuItems();
    wxList::iterator iter;

    for (iter = list.begin(); iter != list.end(); ++iter)
    {
        this->Destroy(static_cast<wxMenuItem *>(*iter));
    }
}

/**initial the menu*/
void PMMenu::InitMenu()
{
    DestroyItems();

    InitSchemeList();

    int schemeIdTemp = 0;

    PowerSchemeMgmtProxy::GetInstance()->GetActivePwrScheme(schemeIdTemp);

    int item_id = MENU_CLASS_DEFAUT;
    wxMenuItem* menuItem = NULL;

    DCS_Battery_Info batteryInfo;
    int index = 0;
    bool res = BatteryInfoMgmtProxy::PM_GetBatteriesInfo(&batteryInfo, index);
    int ac_adapter = BatteryInfoMgmtProxy::PM_GetACAdapterStatus();
    int isAC = 0;
    if (1 == ac_adapter)
    {
        isAC = 1;
    }

    wxString label;
    wxString time;
    wxBitmap bitmap;
    if (!res)
    {
        label = _("AC Power");
        time = _("No system battery.");
        bitmap = wxBitmap(wxImage(PIC_PATH + _T("Menu_AC.png"),wxBITMAP_TYPE_PNG));
    }
    else
    {

        int remPer = 0;
        if (batteryInfo.RemainPercent != INVALID_VALUE && batteryInfo.RemainPercent <= BAT_FULL_CAP)
        {
            remPer = batteryInfo.RemainPercent;
        }


        if (1 == ac_adapter)
        {
            if (remPer < 100)
            {

                label = _("Charging ") + wxString::Format(wxT("%d%%"), remPer);
                if (batteryInfo.RemainTime == INVALID_VALUE)
                {
                    time = _("Unknown time");
                }
                else if (batteryInfo.RemainTime > BAT_OVER_TIME)
                {
                    wxString ws = _("hours remaining");
                    time = wxT("04:28 ") + ws;
                }
                else
                {
                    time = wxString::Format(wxT("%02d:%02d"), batteryInfo.RemainTime / 60, batteryInfo.RemainTime % 60) + wxT(" ") +  _("hours remaining");
                }

                bitmap = wxBitmap(wxImage(PIC_PATH + _T("Menu_Charging.png"),wxBITMAP_TYPE_PNG));
            }
            else
            {
                label = _("AC Power");
                time = _("Battery Fully Charged");
                bitmap = wxBitmap(wxImage(PIC_PATH + _T("Menu_AC.png"),wxBITMAP_TYPE_PNG));
            }



        }

        if (2 == ac_adapter)
        {

            if (remPer >= 60 && remPer <= 100)
            {
                bitmap = wxBitmap(wxImage(PIC_PATH + _T("Menu_High.png"),wxBITMAP_TYPE_PNG));
            }
            else if (remPer >= 30 && remPer < 60)
            {
                bitmap = wxBitmap(wxImage(PIC_PATH + _T("Menu_Mid.png"),wxBITMAP_TYPE_PNG));
            }
            else
            {
                bitmap = wxBitmap(wxImage(PIC_PATH + _T("Menu_Low.png"),wxBITMAP_TYPE_PNG));
            }

            label = _("Battery ") + wxString::Format(wxT("%d%%"), remPer);
            if (batteryInfo.RemainTime == INVALID_VALUE)
            {
                time = _("Unknown time");
            }
            else
            {
                time = wxString::Format(wxT("%02d:%02d"), batteryInfo.RemainTime / 60, batteryInfo.RemainTime % 60) + wxT(" ") + _("hours remaining");
            }
        }
    }

    menuItem = new wxMenuItem(this, MENU_BATTERY_INFO, label + wxT("\n") + time);
    menuItem->SetBitmap(bitmap);
    this->Append(menuItem);



    this->AppendSeparator();

    if (m_schemeList.size() != 0)
    {
        for (vector<PSchemeList_t>::iterator iter = m_schemeList.begin(); iter != m_schemeList.end(); iter++)
        {
            PSchemeList_t schemeTemp = *iter;
            int value = schemeTemp->value;
            if (value != 0)
            {
                wxString string = PMUtility::StringToWxString(schemeTemp->name);
                menuItem = new wxMenuItem(this, item_id, string);

                if (schemeIdTemp == schemeTemp->id)
                {
                    menuItem->SetBitmap(wxBitmap(wxImage(PIC_PATH + _T("TrayMenu_Icon_ActiveScheme.png"),wxBITMAP_TYPE_PNG)));
                }
                else
                {
                    menuItem->SetBitmap(wxBitmap(wxImage(PIC_PATH + _T("TrayMenu_Icon_ActiveScheme_none.png"),wxBITMAP_TYPE_PNG)));
                }

                Connect(item_id, wxEVT_COMMAND_MENU_SELECTED,(wxObjectEventFunction)&PMMenu::ChangeScheme );


                this->Append(menuItem);
                item_id++;
            }
        }
    }

    this->AppendSeparator();

    menuItem = new wxMenuItem(this, MENU_PM_DLG, _("Launch Power Manager"));
    menuItem->SetBitmap(wxBitmap(wxImage(PIC_PATH + _T("TrayMenu_Icon_LaunchPM.png"),wxBITMAP_TYPE_PNG)));
    this->Append(menuItem);

//    menuItem = new wxMenuItem(this, MENU_BATTERY_INFO, _("View Battery Information"));
//    menuItem->SetBitmap(wxBitmap(wxImage(PIC_PATH + _T("TrayMenu_Icon_BattInfo.png"),wxBITMAP_TYPE_PNG)));
//    this->Append(menuItem);

    menuItem = new wxMenuItem(this, MENU_EXIT, _("Exit"));
    menuItem->SetBitmap(wxBitmap(wxImage(PIC_PATH + _T("TrayMenu_Icon_Exit.png"),wxBITMAP_TYPE_PNG)));
    this->Append(menuItem);

}

/**menu item click event handler*/
void PMMenu::OpenPMDialog(wxCommandEvent& event)
{
    DialogRegister* p = DialogRegister::GetInstance();
    PMTrayIcon* task = p->GetPtrTask();
    task->ShowPMDlg();
}

/**menu item click event handler*/
void PMMenu::OpenBatteryInfoDlg(wxCommandEvent& event)
{
    DialogRegister* p = DialogRegister::GetInstance();
    PMTrayIcon* task = p->GetPtrTask();
    task->ShowBatteryDlg();
}

/**menu item click event handler*/
void PMMenu::ExitApp(wxCommandEvent& event)
{
    ::wxExit();
}

/**menu item click event handler*/
void PMMenu::ChangeScheme(wxCommandEvent& event)
{
    if ( DeviceDisableDialog::m_IsExist )
    {
        return;
    }

    int id = event.GetId();
    wxMenuItem* pItem = this->FindItem(id);
    wxString str = pItem->GetText();
    PSchemeList_t schemeTemp = NULL;

    for (vector<PSchemeList_t>::iterator iter = m_schemeList.begin(); iter != m_schemeList.end(); iter++)
    {
        schemeTemp = *iter;
        string name = schemeTemp->name;

        if ( 0 == name.compare(PMUtility::WxStringToString(str)) )
        {
            int schemeId = 0;

            PowerSchemeMgmtProxy::GetInstance()->GetActivePwrScheme(schemeId);
            if (schemeId ==  schemeTemp->id)
            {
                return;
            }
            schemeId = schemeTemp->id;
            int isWarning = 0;
            int isAC = 0;                // power type will be get later
            int ac_adapter = BatteryInfoMgmtProxy::PM_GetACAdapterStatus();

            if (1 == ac_adapter)
            {
                isAC = 1;
            }

//            DataMiddle::GetInstance()->UpdatePMDlgEditingScheme(schemeId);

            ConfigurationFileEditor file;
            file.ReadSwitchWarning(isWarning);

            vector<string> devicesVec;
            PowerSchemeMgmtProxy::GetInstance()->GetDisabledDevices(schemeId, isAC, devicesVec);
            if ( (1 == isWarning) && (devicesVec.size() > 0) )
            {
                DeviceDisableDialog* dlg = new DeviceDisableDialog(NULL);
 				if (dlg->InitDlg(schemeId, isAC, devicesVec))
                {
                    dlg->ShowModal();
                }
            }
            else
            {
                    PowerSchemeMgmtProxy::GetInstance()->ApplyPwrScheme(schemeId);
                    DataMiddle::GetInstance()->UpdatePMDlgButtonStatusAfterSwitching(schemeId);

            }

            break;
        }
    }
}

/**initial the scheme list*/
int PMMenu::InitSchemeList()
{
    m_schemeList.clear();
    ConfigurationFileEditor file;
    int result = file.Read(m_schemeList);
    return result;
}
