/*
** ucview.c
** 
** Made by (Arne Caspari)
** Login   <arne@arne-laptop>
** 
*/

/*
  Copyright (C) 2007-2008  Arne Caspari <arne@unicap-imaging.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
  Main application window and mainloop / callback handling
 */

#include "config.h"

#define _GNU_SOURCE

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gconf/gconf-client.h>

#include <unicap.h>
#include <unicapgtk.h>
#include <ucil.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <locale.h>


#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <signal.h>

#include "ucview.h"
#include "device_dialog.h"
#include "settings_dialog.h"
#include "info_box.h"
#include "checkperm.h"
#include "callbacks.h"
#include "gui.h"
#include "user_config.h"
#if HAVE_DBUS
#include <dbus/dbus-glib.h>
#include "ucview_dbus_server.h"
#endif
#include "plugin_loader.h"
#include "preferences.h"
#include "worker.h"

#include "icon-ucview.h"
#include "icons.h"
#include "sidebar.h"

#include "marshal.h"




enum
{
   PROP_0 = 0, 
   PROP_UNICAP_HANDLE, 
   PROP_KEEP_DBUS_ALIVE, 
};

static void ucview_window_class_init( UCViewWindowClass *klass );
static void ucview_window_init( UCViewWindow *ucv );
static void ucview_window_set_property( GObject *obj, guint property_id, const GValue *value, GParamSpec *pspec );
static void ucview_window_get_property( GObject *obj, guint property_id, GValue *value, GParamSpec *pspec );
static void ucview_window_destroy( GtkObject *obj );
static void ucview_window_realize( UCViewWindow *window );
static void ucview_window_unrealize( UCViewWindow *window );


static void ucview_set_handle( UCViewWindow *window, unicap_handle_t handle );
static void build_main_window( UCViewWindow *window );
static gboolean resize_window( UCViewWindow *ucv );
static GtkWidget *create_display_widget( unicap_handle_t handle );
static void new_frame_cb( unicap_event_t event, unicap_handle_t handle, unicap_data_buffer_t *buffer, UCViewWindow *window );
static void predisplay_cb( UnicapgtkVideoDisplay *ugtk, unicap_data_buffer_t *buffer, UCViewWindow *window );
static void ucview_register_object( UCViewWindow *window, unicap_handle_t handle );

#if HAVE_DBUS
static gchar *build_dbus_string( gchar *name );
#endif

static gboolean ucview_window_enable_record( UCViewWindow *window, gboolean enable, GError **error );
static gboolean ucview_window_record_video_file( UCViewWindow *window, gchar *filename, GError **error );
static gboolean ucview_window_save_still_image( UCViewWindow *window, GError **error );

#if HAVE_DBUS
#include "ucview-server-bindings.h"
#include "client-bindings.h"
#endif

#define UCVIEW_WINDOW_GET_PRIVATE( obj ) ( G_TYPE_INSTANCE_GET_PRIVATE( ( obj ), UCVIEW_WINDOW_TYPE, UCViewWindowPrivate ) )


G_DEFINE_TYPE( UCViewWindow, ucview_window, GTK_TYPE_WINDOW );

guint ucview_signals[ UCVIEW_LAST_SIGNAL ] = { 0 };


typedef struct _CrashData CrashData;

struct _CrashData
{
      GConfClient *client;
      UCViewPluginData *plugin;
};

CrashData crash_data;


struct _UCViewWindowPrivate
{
      GtkUIManager *ui;
      GtkWidget *statusbar;
};



static void crash_handler( int signal, siginfo_t *info, void *_data )
{
   gchar *gconf_path;
   
   if( crash_data.plugin )
   {
      gchar *keyname = g_path_get_basename( g_module_name( crash_data.plugin->module ) );
   
      gconf_path = g_build_path( "/", UCVIEW_GCONF_DIR, "plugins", keyname, NULL );
      gconf_client_set_bool( crash_data.client, gconf_path, FALSE, NULL );
      gconf_client_set_string( crash_data.client, UCVIEW_GCONF_DIR "/crash_module", g_module_name( crash_data.plugin->module ), NULL );
      g_free( gconf_path );
   }
   else
   {
      gconf_client_unset( crash_data.client, UCVIEW_GCONF_DIR "/crash_module", NULL );
   }
   
   gconf_client_set_bool( crash_data.client, UCVIEW_GCONF_DIR "/crash_detected", TRUE, NULL );
};


static void ucview_window_class_init( UCViewWindowClass *klass )
{
   GConfClient *client;
   struct sigaction sigact;
   GObjectClass *object_class = G_OBJECT_CLASS( klass );
   GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS( klass );

   g_type_class_add_private( object_class, sizeof( UCViewWindowPrivate ) );
   
   object_class->set_property = ucview_window_set_property;
   object_class->get_property = ucview_window_get_property;
   gtk_object_class->destroy = ucview_window_destroy;
   
   g_object_class_install_property( object_class, 
				    PROP_UNICAP_HANDLE, 
				    g_param_spec_pointer( "unicap-handle", NULL, NULL, 
							  G_PARAM_READABLE | G_PARAM_WRITABLE /* | G_PARAM_CONSTRUCT_ONLY */ ) );

   g_object_class_install_property( object_class, 
				    PROP_KEEP_DBUS_ALIVE, 
				    g_param_spec_boolean( "keep-dbus-alive", NULL, NULL, FALSE, 
							  G_PARAM_READABLE | G_PARAM_WRITABLE ) );

   client = gconf_client_get_default();
   gconf_client_add_dir( client, UCVIEW_GCONF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL );
   klass->gconf_client = client;

#if HAVE_DBUS
   klass->connection = dbus_g_bus_get( DBUS_BUS_SESSION, NULL );
#endif

   ucview_signals[ UCVIEW_DISPLAY_IMAGE_SIGNAL ] = 
      g_signal_new( "display_image", 
		    G_TYPE_FROM_CLASS( klass ), 
		    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION, 
		    G_STRUCT_OFFSET( UCViewWindowClass, ucview_window ), 
		    NULL, 
		    NULL, 
		    g_cclosure_marshal_VOID__POINTER, 
		    G_TYPE_NONE, 
		    1, 
		    G_TYPE_POINTER );
   ucview_signals[ UCVIEW_SAVE_IMAGE_SIGNAL ] = 
      g_signal_new( "save_image", 
		    G_TYPE_FROM_CLASS( klass ), 
		    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION, 
		    G_STRUCT_OFFSET( UCViewWindowClass, ucview_window ), 
		    NULL, 
		    NULL, 
		    g_cclosure_user_marshal_VOID__POINTER_STRING, 
		    G_TYPE_NONE, 
		    2, 
		    G_TYPE_POINTER, 
		    G_TYPE_STRING );
   ucview_signals[ UCVIEW_IMAGE_FILE_CREATED_SIGNAL ] = 
      g_signal_new( "image_file_created", 
		    G_TYPE_FROM_CLASS( klass ), 
		    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION, 
		    G_STRUCT_OFFSET( UCViewWindowClass, ucview_window ), 
		    NULL, NULL, 
		    g_cclosure_marshal_VOID__STRING, 
		    G_TYPE_NONE, 
		    1, 
		    G_TYPE_STRING );
   ucview_signals[ UCVIEW_RECORD_START_SIGNAL ] = 
      g_signal_new( "record_start", 
		    G_TYPE_FROM_CLASS( klass ), 
		    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION, 
		    G_STRUCT_OFFSET( UCViewWindowClass, ucview_window ), 
		    NULL, NULL, 
		    g_cclosure_user_marshal_VOID__STRING_STRING, 
		    G_TYPE_NONE, 
		    2, 
		    G_TYPE_STRING, 
		    G_TYPE_STRING );
   ucview_signals[ UCVIEW_TIME_LAPSE_START_SIGNAL ] = 
      g_signal_new( "time_lapse_start", 
		    G_TYPE_FROM_CLASS( klass ), 
		    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION, 
		    G_STRUCT_OFFSET( UCViewWindowClass, ucview_window ), 
		    NULL, NULL, 
		    g_cclosure_marshal_VOID__VOID, 
		    G_TYPE_NONE, 0 );
      
   ucview_signals[ UCVIEW_VIDEO_FILE_CREATED_SIGNAL ] = 
      g_signal_new( "video_file_created", 
		    G_TYPE_FROM_CLASS( klass ), 
		    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION, 
		    G_STRUCT_OFFSET( UCViewWindowClass, ucview_window ), 
		    NULL, NULL, 
		    g_cclosure_marshal_VOID__STRING, 
		    G_TYPE_NONE, 
		    1, 
		    G_TYPE_STRING );

   crash_data.client = client;
   crash_data.plugin = NULL;
   
   sigact.sa_sigaction = crash_handler;
   sigact.sa_flags = SA_RESETHAND | SA_SIGINFO;
   sigaction( SIGSEGV, &sigact, NULL );
}

static void crash_close_cb( GtkWidget *box, gint response, UCViewWindow *window )
{
   gconf_client_set_bool( crash_data.client, UCVIEW_GCONF_DIR "/crash_detected", FALSE, NULL );
   gtk_widget_destroy( box );
}


static gboolean delete_window_cb( GtkWidget *widget, GdkEvent * event, gpointer data )
{
   prefs_store_window_state( UCVIEW_WINDOW( widget ) );
   
   return FALSE;
}


static void ucview_window_init( UCViewWindow *window )
{
   UCViewWindowClass *klass = UCVIEW_WINDOW_GET_CLASS( window );

   window->priv = UCVIEW_WINDOW_GET_PRIVATE( window );

   window->keep_dbus_alive = FALSE;
   window->modules = NULL;
   window->mutex = g_mutex_new();
   window->record = FALSE;
   window->still_image = FALSE;
   window->last_video_file = NULL;
   window->property_dialog = NULL;
   window->client = klass->gconf_client;
   window->glade = glade_xml_new( INSTALL_PREFIX "/share/ucview/ucview.glade", "ucview_main_hbox", NULL );
   window->fps_timeout = 0xffffffff;

   g_signal_connect( window, "realize", (GCallback)ucview_window_realize, NULL );
   g_signal_connect( window, "unrealize", (GCallback)ucview_window_unrealize, NULL );
   
   if( !window->glade )
   {
      window->glade = glade_xml_new( "ucview.glade", "ucview_main_hbox", NULL );
   }
   
   window->video_filename = NULL;
   build_main_window( window );

   window->size_restored = prefs_restore_window_state( window );

   window->fps_timer = g_timer_new();


   g_signal_connect( window, "delete-event", (GCallback)delete_window_cb, NULL );

   window->gconf_cnxn_id = gconf_client_notify_add( window->client, UCVIEW_GCONF_DIR, 
						    (GConfClientNotifyFunc) gconf_notification_cb, window, NULL, NULL );

   gconf_client_notify( window->client, UCVIEW_GCONF_DIR "/show_fps" );
   gconf_client_notify( window->client, UCVIEW_GCONF_DIR "/scale_to_fit" );

   if( gconf_client_get_bool(  window->client, UCVIEW_GCONF_DIR "/crash_detected", NULL ) )
   {
      GtkWidget *info_box;
      GtkWidget *info_text;
      gchar *message;
      gchar *modname;		
   
      modname = gconf_client_get_string( window->client, UCVIEW_GCONF_DIR "/crash_module", NULL );

      if( modname )
      {
	 asprintf( &message, 
		   _("<b>Crash detected!</b>\nUCView detected a malfunction of the \n<i>'%s'</i>\nplugin. The plugin got disabled." ), 
		   modname );
	 g_free( modname );
      }
      else
      {
	 message = g_strdup( _("<b>Crash detected!</b>\nUCView detected a crash but it could not find the cause. \nPlease report a bug." ) );
      }

      info_text = gtk_label_new( message );
      gtk_misc_set_alignment( GTK_MISC( info_text ), 0.0f, 0.0f );
      g_free( message );
      
      g_object_set( info_text, "use-markup", TRUE, NULL );
      info_box = g_object_new( INFO_BOX_TYPE, "image", gtk_image_new_from_stock( GTK_STOCK_DIALOG_WARNING, GTK_ICON_SIZE_DIALOG ),
			       "text", info_text, NULL );
      info_box_add_action_widget( INFO_BOX( info_box ),
				  gtk_button_new_from_stock( GTK_STOCK_CLOSE ),
				  GTK_RESPONSE_CLOSE );
      g_signal_connect( info_box, "response", G_CALLBACK( crash_close_cb ), window );
      
      ucview_set_info_box( window, info_box );
   }
}

static void ucview_window_set_property( GObject *object, 
					guint property_id, 
					const GValue *value, 
					GParamSpec *pspec )
{
   UCViewWindow *window = UCVIEW_WINDOW( object );
   
   switch( property_id )
   {
      case PROP_UNICAP_HANDLE:
      {
	 unicap_handle_t handle = NULL;	 
	 handle = ( unicap_handle_t ) g_value_get_pointer( value );

	 ucview_set_handle( window, handle );
	 ucview_register_object( window, handle );
/* 	 g_object_set( window->settings_dialog, "unicap-handle", handle, NULL ); */
      }
      break;

      case PROP_KEEP_DBUS_ALIVE:
	 window->keep_dbus_alive = g_value_get_boolean( value );
	 break;
      
      default:
	 G_OBJECT_WARN_INVALID_PROPERTY_ID( object, property_id, pspec );
	 break;
   }
}
      
static void ucview_window_get_property( GObject *object, 
					guint property_id, 
					GValue *value, 
					GParamSpec *pspec )
{
   UCViewWindow *window = UCVIEW_WINDOW( object );
   
   switch( property_id )
   {
      case PROP_UNICAP_HANDLE:
      {
	 g_value_set_pointer( value, window->device_handle );
      }
      break;

      case PROP_KEEP_DBUS_ALIVE:
	 g_value_set_boolean( value, window->keep_dbus_alive );
	 break;

      default:
	 G_OBJECT_WARN_INVALID_PROPERTY_ID( object, property_id, pspec );
	 break;
   }
}

static void ucview_register_object( UCViewWindow *window, unicap_handle_t handle )
{
#if HAVE_DBUS
   gchar *devid;
   unicap_device_t device;
   UCViewWindowClass *klass = UCVIEW_WINDOW_GET_CLASS( window );

   if( klass->connection )
   {
      unicap_get_device( handle, &device );
      devid = build_dbus_string( device.identifier );
      window->dbus_path = g_strconcat( "/org/unicapimaging/UCView/", devid, NULL );

      dbus_g_object_type_install_info( UCVIEW_WINDOW_TYPE, &dbus_glib_ucview_window_object_info );
      dbus_g_connection_register_g_object (klass->connection,
					   window->dbus_path,
					   G_OBJECT( window ) );   
   }
#endif
}

static void ucview_window_realize( UCViewWindow *window )
{
}

static void ucview_window_unrealize( UCViewWindow *window )
{
   ucview_unload_plugins( window );
}

static gboolean property_dialog_filter( unicap_property_t *property, UCViewWindow *window )
{
   GConfClient *client;
   GSList *list;
   gboolean filter = FALSE;
   
   client = gconf_client_get_default( );

   list = gconf_client_get_list( client, UCVIEW_GCONF_DIR "/property_filter", GCONF_VALUE_STRING, NULL );
   
   if( g_slist_find( list, property->identifier ) != NULL )
   {
      filter = TRUE;
   }
   
   g_slist_free( list );

   g_object_unref( client );
   
   return filter;
}


static void ucview_set_handle( UCViewWindow *window, unicap_handle_t handle )
{
   GtkWidget *property_dialog;
   GtkWidget *vbox;
   unicap_format_t format;
   unicap_device_t device;
   GString *title;
   GtkWidget *label;
   GList *entry;   
   gint w;

   vbox = glade_xml_get_widget( window->glade, "ucview_main_vbox" );
   window->device_handle = handle;

   unicap_get_device( handle, &device );
   title = g_string_new( "" );
   g_string_printf( title, "%s - UCView", device.identifier );
   gtk_window_set_title( GTK_WINDOW( window ), title->str );
   g_string_free( title, TRUE );
   
   label = gtk_label_new( _("Starting video device...") );
   gtk_widget_show( label );
   gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW( window->display_window ), label );
   window->display = create_display_widget( handle );
   gtk_widget_destroy( label );
   g_object_set( window->display, "scale-to-fit", gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/scale_to_fit", NULL ), NULL );
   if( gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/disable_xv", NULL ) )
   {
      g_object_set( window->display, "backend", "gtk", NULL );
   }
   

   unicap_get_format( handle, &format );
   if( !window->size_restored )
   {
      if( ( format.size.width > 1024 ) || ( format.size.height > 768 ) )
      {
	 gtk_widget_set_size_request( window->display_window, 1024, 768 );
      }
      else
      {
	 gtk_widget_set_size_request( window->display_window, format.size.width + 10, format.size.height + 10 );
      }
   }
   window->display_ebox = gtk_event_box_new();
   gtk_container_add( GTK_CONTAINER( window->display_ebox ), window->display );
   gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW( window->display_window ), window->display_ebox );
   gtk_widget_show_all( GTK_WIDGET( vbox ) );
   g_signal_connect( G_OBJECT (window), "delete-event", (GCallback)gtk_widget_destroy, NULL );
   unicapgtk_video_display_set_handle( UNICAPGTK_VIDEO_DISPLAY( window->display ), handle );
   if( gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/encoder_fastpath", NULL ) )
   {
      g_object_set( window->display, "backend_fourcc", UCIL_FOURCC( 'I', '4', '2', '0' ), NULL );
   }
   // Try to set I420 format for display to save resources for video encoding
   if( SUCCESS( unicapgtk_video_display_set_format( UNICAPGTK_VIDEO_DISPLAY( window->display ), &format ) ) )
   {
      // check whether the change of fourcc was successful
      unsigned int fourcc;
      
      g_object_get( window->display, "backend_fourcc", &fourcc, NULL );
      
      if( fourcc == UCIL_FOURCC( 'I', '4', '2', '0' ) )
      {
	 unicapgtk_video_display_set_new_frame_callback( UNICAPGTK_VIDEO_DISPLAY( window->display ), 
							 UNICAPGTK_CALLBACK_FLAGS_XV_CB | UNICAPGTK_CALLBACK_FLAGS_BEFORE, 
							 (unicap_new_frame_callback_t)new_frame_cb, 
							 window );
	 window->encoder_fastpath = TRUE;
      }
      else if( gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/encoder_fastpath", NULL ) )
      {
	 // failed! Start again with any format
	 g_object_set( window->display, "backend_fourcc", 0, NULL );
	 unicapgtk_video_display_set_format( UNICAPGTK_VIDEO_DISPLAY( window->display ), &format );
	 unicapgtk_video_display_set_new_frame_callback( UNICAPGTK_VIDEO_DISPLAY( window->display ), 
							 UNICAPGTK_CALLBACK_FLAGS_BEFORE, 
							 (unicap_new_frame_callback_t)new_frame_cb, 
							 window );
      }
      else
      {
	 unicapgtk_video_display_set_new_frame_callback( UNICAPGTK_VIDEO_DISPLAY( window->display ), 
							 UNICAPGTK_CALLBACK_FLAGS_BEFORE, 
							 (unicap_new_frame_callback_t)new_frame_cb, 
							 window );
      }
   }
   else
   {
      // failed! Start again with any format
      g_object_set( window->display, "backend_fourcc", 0, NULL );
      unicapgtk_video_display_set_format( UNICAPGTK_VIDEO_DISPLAY( window->display ), &format );
      unicapgtk_video_display_set_new_frame_callback( UNICAPGTK_VIDEO_DISPLAY( window->display ), 
						      UNICAPGTK_CALLBACK_FLAGS_BEFORE, 
						      (unicap_new_frame_callback_t)new_frame_cb, 
						      window );
   }

   window->sidebar = sidebar_new( window );
   gtk_box_pack_start( GTK_BOX( glade_xml_get_widget( window->glade, "ucview_sidebar_box" ) ), 
		       window->sidebar, 
		       TRUE, TRUE, 0 );

   w = gconf_client_get_int( window->client, UCVIEW_GCONF_DIR "/sidebar_width", NULL );
   if( !w )
   {
      w = gconf_client_get_int( window->client, UCVIEW_GCONF_DIR "/sidebar_image_width", NULL );
      if( w )
      {
	 w += 8;
      }
      else
      {
	 w = 136;
      }
   }
   
   gtk_paned_set_position( GTK_PANED( glade_xml_get_widget( window->glade, "ucview_hpaned" ) ), w );
   
   
      

   if( !SUCCESS( unicapgtk_video_display_start( UNICAPGTK_VIDEO_DISPLAY( window->display ) ) ) )
   {
      g_warning( _("Failed to start video display!" ) );
   }
      
   
   


   if( window->property_dialog )
   {
      gtk_widget_destroy( window->property_dialog );
   }
   property_dialog = unicapgtk_property_dialog_new_by_handle( handle );

   unicapgtk_property_dialog_set_filter( UNICAPGTK_PROPERTY_DIALOG( property_dialog ), 
					 (unicap_property_filter_func_t) property_dialog_filter, window );
   g_object_set( property_dialog, "update-interval", 0, NULL );
   g_signal_connect_swapped( G_OBJECT( window ), "destroy", G_CALLBACK( gtk_widget_destroy ), property_dialog );
   g_signal_connect( G_OBJECT( property_dialog ), "delete-event", G_CALLBACK( gtk_widget_hide_on_delete ), NULL );
   gtk_window_set_transient_for( GTK_WINDOW( property_dialog ), GTK_WINDOW( window ) );
   window->property_dialog = property_dialog;
   gtk_widget_hide( window->property_dialog );

   unicap_copy_format( &window->format, &format );
   unicap_copy_format( &window->still_image_buffer.format, &format );
   window->still_image_buffer.format.fourcc = UCIL_FOURCC( 'R', 'G', 'B', '3' );
   window->still_image_buffer.format.bpp = 24;
   window->still_image_buffer.format.buffer_size = window->still_image_buffer.format.size.width * 
      window->still_image_buffer.format.size.height * 3;
   window->still_image_buffer.buffer_size = window->still_image_buffer.format.buffer_size;
   window->still_image_buffer.data = malloc( window->still_image_buffer.format.buffer_size );

   g_signal_connect( G_OBJECT( window->display ), "motion_notify_event", G_CALLBACK( display_motion_notify_cb ), window );
   g_signal_connect( G_OBJECT( window->display ), "button_press_event", G_CALLBACK( display_button_press_event_cb ), window );
   g_signal_connect( G_OBJECT( window->display ), "unicapgtk_video_display_predisplay", G_CALLBACK( predisplay_cb ), window );

   g_idle_add( (GSourceFunc)resize_window , window );

   g_mutex_lock( window->mutex );
   
   ucview_load_plugins( window );
   settings_dialog_update_plugins( SETTINGS_DIALOG( window->settings_dialog ), window );

   for( entry = g_list_first( window->modules ); entry; entry=g_list_next( entry ) )
   {
      UCViewPluginData *plugin;
      UCViewPluginEnableFunc enable_func;

      plugin = entry->data;
      if( plugin->enable )
      {
	 if( g_module_symbol( plugin->module, "ucview_plugin_enable", (void*)&enable_func ) )
	 {
	    enable_func( plugin->plugin_data, &window->format );
	 }
      }
   }

   g_mutex_unlock( window->mutex );

   ucview_window_set_fullscreen( window, gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/fullscreen", NULL ) );
   unicapgtk_property_dialog_reset( UNICAPGTK_PROPERTY_DIALOG( property_dialog ) );
}


static void ucview_window_destroy( GtkObject *obj )
{
   UCViewWindow *ucv = UCVIEW_WINDOW( obj );   

   gdk_threads_leave();
   if( ucv->device_handle )
   {
      if( ucv->display )
      {
	 unicapgtk_video_display_stop( UNICAPGTK_VIDEO_DISPLAY( ucv->display ) );
      }
      user_config_store_device( ucv->device_handle );
      unicap_close( ucv->device_handle );
   }
   gdk_threads_enter();
 
   if( ucv->fps_timeout != 0xffffffff )
   {
      g_source_remove( ucv->fps_timeout );
      ucv->fps_timeout = 0xffffffff;
   }

/*    gtk_container_foreach( GTK_CONTAINER( obj ), (GtkCallback) gtk_widget_destroy, NULL ); */

   if( ucv->dbus_path )
   {
      g_free( ucv->dbus_path );
   } 

   gconf_client_notify_remove( ucv->client, ucv->gconf_cnxn_id );

}

void ucview_set_info_box( UCViewWindow *window, GtkWidget *info_box )
{
   GtkWidget *container;
   
   container = glade_xml_get_widget( window->glade, "info_box_box" );
   g_assert( container );
   gtk_container_foreach( GTK_CONTAINER( container ), (GtkCallback)gtk_widget_destroy, NULL );
   
   gtk_container_add( GTK_CONTAINER( container ), info_box );
}

static void hide_image_info_box_toggled_cb( GtkToggleButton *button, UCViewWindow *window )
{
   gconf_client_set_bool( window->client, 
			  UCVIEW_GCONF_DIR "/hide_image_saved_box", 
			  gtk_toggle_button_get_active( button ), 
			  NULL );
}

static void predisplay_cb( UnicapgtkVideoDisplay *ugtk, unicap_data_buffer_t *buffer, UCViewWindow *window )
{
   if( window->still_image || window->clipboard_copy_image )
   {
      GdkPixbuf *pixbuf;

      ucil_convert_buffer( &window->still_image_buffer, buffer );

	 
      pixbuf = gdk_pixbuf_new_from_data( window->still_image_buffer.data, 
					 GDK_COLORSPACE_RGB, 
					 0, 
					 8, 
					 window->still_image_buffer.format.size.width, 
					 window->still_image_buffer.format.size.height, 
					 window->still_image_buffer.format.size.width * 3, 
					 NULL, 
					 NULL );
					 
					 
      if( pixbuf )
      {
	 if( window->still_image )
	 {
	    g_signal_emit( G_OBJECT( window ), ucview_signals[ UCVIEW_SAVE_IMAGE_SIGNAL ], 0, pixbuf, window->still_image_path );
	    gdk_pixbuf_save( pixbuf, window->still_image_path, window->still_image_file_type, NULL, NULL );
	    g_signal_emit( G_OBJECT( window ), ucview_signals[ UCVIEW_IMAGE_FILE_CREATED_SIGNAL ], 0, window->still_image_path );

	    if( !window->time_lapse && !gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/hide_image_saved_box", NULL ) )
	    {
	       GtkWidget *vbox;
	       GtkWidget *info_text;
	       GtkWidget *info_box;
	       GtkWidget *check_button;
	       gchar *message;
	       
	       vbox = gtk_vbox_new( FALSE, 6 );
	       
	       asprintf( &message, 
			 _("Picture saved to: \n<i>%s</i>"), 
			 window->still_image_path );
	       
	       info_text = gtk_label_new( message );
	       g_object_set( info_text, "use-markup", TRUE, NULL );
	       g_free( message );

	       gtk_box_pack_start( GTK_BOX( vbox ), info_text, FALSE, TRUE, 0 );

	       check_button = gtk_check_button_new_with_label( _("Do not show this message again") );
	       g_signal_connect( check_button, "toggled", (GCallback)hide_image_info_box_toggled_cb, window );
	       gtk_box_pack_start( GTK_BOX( vbox ), check_button, FALSE, TRUE, 0 );
	       
	       info_box = g_object_new( INFO_BOX_TYPE, "image", gtk_image_new_from_stock( GTK_STOCK_DIALOG_INFO, GTK_ICON_SIZE_DIALOG ),
					"text", vbox, NULL );
	       
	       info_box_add_action_widget( INFO_BOX( info_box ),
					   gtk_button_new_from_stock( GTK_STOCK_CLOSE ),
					   GTK_RESPONSE_CLOSE );
	       
	       g_signal_connect( info_box, "response", G_CALLBACK( gtk_widget_destroy ), NULL );
	       gtk_widget_show_all( info_box );
	       
	       ucview_set_info_box( window, info_box );

	    }
	    sidebar_add_image( SIDEBAR( window->sidebar ), buffer, window->still_image_path );
	 }
	 
	 
	 if( window->clipboard_copy_image )
	 {
	    GtkClipboard *clipboard;
	    
	    clipboard = gtk_clipboard_get( GDK_SELECTION_CLIPBOARD );
	    
	    gtk_clipboard_set_image( clipboard, pixbuf );
	 }
	 g_object_unref( pixbuf );
      }
      
      window->still_image = FALSE;
      window->clipboard_copy_image = FALSE;
   }

   g_signal_emit( G_OBJECT( window ), ucview_signals[ UCVIEW_DISPLAY_IMAGE_SIGNAL ], 0, (gpointer)buffer );


}

const gchar *ucview_window_get_dbus_path( UCViewWindow *window )
{
   return window->dbus_path;
}

#if HAVE_DBUS
static gchar *build_dbus_string( gchar *name )
{
   gchar *result;
   gchar *c;
   
   result = g_strdup( name );
   
   for( c = result; *c; c++ )
   {
      if( !g_ascii_isalnum( *c ) )
      {
	 *c = '_';
      }
   }
   
   return result;
}
#endif


static void new_frame_cb( unicap_event_t event, unicap_handle_t handle, unicap_data_buffer_t *buffer, UCViewWindow *window )
{
   gdouble ival;
   static gdouble fps[30] = { 0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0, 0,0,0,0,0,0,0,0,0,0, };
   static int i = 0;
   int j;
   gdouble tmp = 0;
   GList *e;
   

   g_mutex_lock( window->mutex );

   ival = g_timer_elapsed( window->fps_timer, NULL );
   g_timer_start( window->fps_timer );
   
   fps[i++] = 1.0 / ((ival>0.0) ? ival : 1.0 );
   i = i % 30;
   for( j = 0; j < 30; j++ )
   {
      tmp += fps[j];
   }
   tmp /= 30.0;
   window->fps = tmp;


   for( e = g_list_first( window->modules ); e; e = g_list_next( e ) )
   {
      UCViewPluginData *plugin;
      
      plugin = e->data;
      crash_data.plugin = plugin;
      if( plugin->enable && plugin->plugin_data->new_frame_cb )
      {
	 plugin->plugin_data->new_frame_cb( event, handle, buffer, plugin->plugin_data->user_ptr );
      }
      crash_data.plugin = NULL;
   }

   if( window->time_lapse )
   {
      gboolean stop = FALSE;     
      
      if( window->time_lapse_first_frame || ( ( g_timer_elapsed( window->time_lapse_timer, NULL ) * 1000.0 ) >= window->time_lapse_delay ) )
      {
	 window->time_lapse_frames++;
	 if( window->time_lapse_mode == UCVIEW_TIME_LAPSE_VIDEO )
	 {
	    if( window->record )
	    {
	       ucil_encode_frame( window->video_object, buffer );
	    }
	 }
	 else
	 {
	    GtkAction *save_action = gtk_ui_manager_get_action( window->priv->ui, "/MenuBar/FileMenu/SaveImage" );
	    gtk_action_activate( save_action );
	 }
	 g_timer_start( window->time_lapse_timer );
	 window->time_lapse_first_frame = FALSE;
      }

      if( window->time_lapse_total_time && 
	  ( ( g_timer_elapsed( window->time_lapse_total_timer, NULL ) * 1000.0 ) >= window->time_lapse_total_time ) )
      {
	 stop = TRUE;
      }
      
      if( window->time_lapse_total_frames && ( window->time_lapse_frames >= window->time_lapse_total_frames ) )
      {
	 stop = TRUE;
      }
      
      if( stop )
      {
	 GtkAction *time_lapse_action = gtk_ui_manager_get_action( window->priv->ui, "/MenuBar/EditMenu/TimeLapse" );
	 gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( time_lapse_action ), FALSE );
      }
   }
   else if( window->record )
   {
      ucil_encode_frame( window->video_object, buffer );
   }

   for( e = g_list_first( window->modules ); e; e = g_list_next( e ) )
   {
      UCViewPluginData *plugin;
      
      plugin = e->data;
      crash_data.plugin = plugin;
      if( plugin->enable && plugin->plugin_data->display_frame_cb )
      {
	 plugin->plugin_data->display_frame_cb( event, handle, buffer, plugin->plugin_data->user_ptr );
      }
      crash_data.plugin = NULL;
   }

   g_mutex_unlock( window->mutex );
}


void *create_display_worker( unicap_handle_t handle )
{
   unicap_format_t format;

   unicap_get_format( handle, &format );
   format.buffer_type = UNICAP_BUFFER_TYPE_SYSTEM;
   unicap_set_format( handle, &format );
   
   return NULL;
}
   

static GtkWidget *create_display_widget( unicap_handle_t handle )
{
   GtkWidget *ugtk;
   
   run_worker( (void*(*)(void*))create_display_worker, handle );
   
   ugtk = unicapgtk_video_display_new( );  

   return ugtk;
}

static void build_main_window( UCViewWindow *window )
{
   GtkWidget *hbox;
   GtkWidget *box;
   GtkWidget *toolbar;
   GdkPixbuf *pixbuf;
   GtkWidget *settings_dialog;
   GtkWidget *frame;
   GtkWidget *eventbox;

   pixbuf = gdk_pixbuf_new_from_inline( -1, icon_ucview_data, FALSE, NULL );
   gtk_window_set_icon( GTK_WINDOW( window ), pixbuf );
   g_object_unref( G_OBJECT( pixbuf ) );

   hbox = glade_xml_get_widget( window->glade, "ucview_main_hbox" );
   gtk_container_add( GTK_CONTAINER( window ), hbox );
   
   window->priv->ui = gui_create_manager( window );
   box = glade_xml_get_widget( window->glade, "ucview_main_menubar_box" );
   gtk_container_foreach( GTK_CONTAINER( box ), (GtkCallback)gtk_widget_destroy, NULL );
   gtk_box_pack_start( GTK_BOX( box ), gtk_ui_manager_get_widget( window->priv->ui, "/MenuBar" ), TRUE, TRUE, 0 );
   gtk_window_add_accel_group( GTK_WINDOW( window ), gtk_ui_manager_get_accel_group( window->priv->ui ) );

   box = glade_xml_get_widget( window->glade, "ucview_main_toolbar_box" );
   gtk_container_foreach( GTK_CONTAINER( box ), (GtkCallback)gtk_widget_destroy, NULL );
   toolbar = gtk_ui_manager_get_widget( window->priv->ui, "/ToolBar" );
   gtk_box_pack_start( GTK_BOX( box ), toolbar, TRUE, TRUE, 0 );


   window->vtoolbarwindow = gtk_window_new( GTK_WINDOW_POPUP );
   eventbox = gtk_event_box_new();
   frame = gtk_frame_new( NULL );
   gtk_window_set_decorated( GTK_WINDOW( window->vtoolbarwindow ), FALSE );
   gtk_window_set_skip_taskbar_hint( GTK_WINDOW( window->vtoolbarwindow ), TRUE );
   toolbar = gtk_ui_manager_get_widget( window->priv->ui, "/VToolBar" );
   gtk_toolbar_set_orientation( GTK_TOOLBAR( toolbar ), GTK_ORIENTATION_VERTICAL );
   gtk_toolbar_set_show_arrow( GTK_TOOLBAR( toolbar ), FALSE );
   gtk_container_add( GTK_CONTAINER( frame ), toolbar );
   gtk_container_add( GTK_CONTAINER( eventbox ), frame );
   gtk_container_add( GTK_CONTAINER( window->vtoolbarwindow ), eventbox );
   gtk_window_set_keep_above( GTK_WINDOW( window->vtoolbarwindow ), TRUE );
   gtk_window_set_default_size( GTK_WINDOW( window->vtoolbarwindow ), 10, 10 );

   g_signal_connect( eventbox, "enter-notify-event", (GCallback)fs_toolbar_enter_notify_cb, window );


   settings_dialog = settings_dialog_new( window );
   g_signal_connect_swapped( G_OBJECT( window ), "destroy", G_CALLBACK( gtk_widget_destroy ), settings_dialog );
   g_signal_connect( G_OBJECT( settings_dialog ), "delete-event", G_CALLBACK( gtk_widget_hide_on_delete ), NULL );   
   gtk_window_set_transient_for( GTK_WINDOW( settings_dialog ), GTK_WINDOW( window ) );
   window->settings_dialog = settings_dialog;
   
   window->display_window = glade_xml_get_widget( window->glade, "ucview_display_scrolledwindow" );
   window->priv->statusbar = glade_xml_get_widget( window->glade, "ucview_statusbar" );


}

static gboolean resize_window( UCViewWindow *ucv )
{
   // This is a bit of a hack: 
   // Reset the size request of the display window to a small value to
   // alow the user to make the window smaller. 
   // The display request is set to the full resolution of the video
   // format previously so that the window is created with the correct size.

   gtk_widget_set_size_request( ucv->display_window, 64, 64 );
   return FALSE;
}

GtkWidget *ucview_window_new( unicap_handle_t handle )
{
   UCViewWindow *window;
   
   window = g_object_new( UCVIEW_WINDOW_TYPE, NULL );
   if( handle )
   {
      g_object_set( window, "unicap-handle", handle, NULL );
   }
   
   
   return GTK_WIDGET( window );
}


static gboolean ucview_window_enable_record( UCViewWindow *window, gboolean enable, GError **error )
{
   GtkAction *play_video_action;
   GtkAction *record_video_action;
   
   play_video_action = gtk_ui_manager_get_action( window->priv->ui, "/MenuBar/EditMenu/PlayVideo" );
   if( play_video_action )
   {
      gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( play_video_action ), FALSE );
   }
   record_video_action = gtk_ui_manager_get_action( window->priv->ui, "/MenuBar/EditMenu/RecordVideo" );
   if( record_video_action )
   {
      gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( record_video_action ), enable );
   }

   return TRUE;
}

static gboolean ucview_window_record_video_file( UCViewWindow *window, gchar *filename, GError **error )
{
   GtkAction *play_video_action;
   GtkAction *record_video_action;
   
   play_video_action = gtk_ui_manager_get_action( window->priv->ui, "/MenuBar/EditMenu/PlayVideo" );
   gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( play_video_action ), FALSE );
   window->video_filename = g_path_get_basename( filename );
   record_video_action = gtk_ui_manager_get_action( window->priv->ui, "/MenuBar/EditMenu/RecordVideo" );
   gtk_toggle_action_set_active( GTK_TOGGLE_ACTION( record_video_action ), TRUE );
   
   return TRUE;
}

static gboolean ucview_window_save_still_image( UCViewWindow *window, GError **error )
{
   GtkAction *save_image_action;
   
   save_image_action = gtk_ui_manager_get_action( window->priv->ui, "/MenuBar/FileMenu/SaveImage" );
   gtk_action_activate( save_image_action );
   
   return TRUE;
}

void ucview_window_set_fullscreen( UCViewWindow *window, gboolean fullscreen )
{
   GtkWidget *eventbox;
   GtkWidget *htoolbar;
   GtkWidget *vtoolbar;
   GtkWidget *menubar;
   
   htoolbar = gtk_ui_manager_get_widget( window->priv->ui, "/ToolBar" );
   vtoolbar = gtk_ui_manager_get_widget( window->priv->ui, "/VToolBar" );
   menubar = gtk_ui_manager_get_widget( window->priv->ui, "/MenuBar" );
   eventbox = glade_xml_get_widget( window->glade, "ucview_main_eventbox" );
   g_assert( htoolbar && vtoolbar && menubar && eventbox );
   gtk_widget_show( eventbox );
   
   window->is_fullscreen = fullscreen;
	 
   if( !fullscreen )
   {
      gtk_window_unfullscreen( GTK_WINDOW( window ) );
      gconf_client_set_bool( window->client, UCVIEW_GCONF_DIR "/fullscreen", FALSE, NULL );
      gtk_widget_hide( window->vtoolbarwindow );
      if( gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/show_toolbar", NULL ) )
      {
	 gtk_widget_show( htoolbar );
      }

      if( gconf_client_get_bool( window->client, UCVIEW_GCONF_DIR "/show_sidebar", NULL ) )
      {
	 gtk_widget_show( glade_xml_get_widget( window->glade, "ucview_sidebar_box" ) );
      }
      
      gtk_widget_show( menubar );
      gtk_widget_show( window->priv->statusbar );
      gtk_widget_modify_bg( window->display_ebox, GTK_STATE_NORMAL, NULL );
   }
   else
   {
      GdkColormap *colormap;
      GdkColor color;      
      GtkRequisition toolbar_req;
      GdkScreen *screen;

      color.red = color.green = color.blue = 0;
      colormap = gtk_widget_get_colormap( window->display_ebox );
      gdk_rgb_find_color( colormap, &color );
      gtk_widget_modify_bg( window->display_ebox, GTK_STATE_NORMAL, &color );
      gtk_window_present( GTK_WINDOW( window->vtoolbarwindow ) );

      gtk_window_fullscreen( GTK_WINDOW( window ) );
      screen = gtk_widget_get_screen( GTK_WIDGET( window ) );
      gtk_widget_show_all( window->vtoolbarwindow );
      gtk_widget_size_request( window->vtoolbarwindow, &toolbar_req );
      gtk_window_move( GTK_WINDOW( window->vtoolbarwindow ), 0, gdk_screen_get_height( screen ) / 2 - toolbar_req.height / 2 );

      gconf_client_set_bool( window->client, UCVIEW_GCONF_DIR "/fullscreen", TRUE, NULL );
      gtk_widget_hide( menubar );
      gtk_widget_hide( window->priv->statusbar );
      gtk_widget_hide( htoolbar );
      gtk_widget_hide( glade_xml_get_widget( window->glade, "ucview_sidebar_box" ) );
      gtk_widget_show_all( vtoolbar );
   }   
}



#if HAVE_DBUS
static void new_window_async_reply( DBusGProxy *proxy, char *answer, GError *error, gpointer userdata)
{
   return;
}
#endif

static void ucview_destroy_event_cb( GtkWidget *ucv, gpointer data )
{
   gboolean keep_alive;

   g_object_get( ucv, "keep-dbus-alive", &keep_alive, NULL );

   if( !keep_alive )
   {
      gtk_main_quit();
   }
}
   

static gboolean open_window_no_dbus()
{
   GtkWidget *ucv;
   GtkWidget *device_dialog;
   unicap_handle_t handle;
   unicap_device_t device;

   if( !SUCCESS( unicap_enumerate_devices( NULL, &device, 0 ) ) )
   {
      GtkWidget *dlg;
      dlg = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
				    "No supported video capture device found!" );
      gtk_dialog_run( GTK_DIALOG( dlg ) );
      gdk_threads_leave();
      gtk_widget_destroy( dlg );
/*       g_set_error( error, 0, 100, "No supported video capture device found!" ); */
      return TRUE;
   }

   device_dialog = g_object_new( DEVICE_DIALOG_TYPE, "restore-device", TRUE, NULL );
   gtk_widget_show_all( device_dialog );
   handle = device_dialog_run( DEVICE_DIALOG( device_dialog ) );
   gtk_widget_destroy( device_dialog );

   if( !handle )
   {
      GtkWidget *dlg;
      dlg = gtk_message_dialog_new( NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
				    "No device!" );
      gtk_dialog_run( GTK_DIALOG( dlg ) );
      gdk_threads_leave();
      gtk_widget_destroy( dlg );
/*       g_set_error( error, 0, 101, "No device selected" ); */
      return TRUE;
   }

   icons_add_stock_items();
   ucv = ucview_window_new( NULL );
   gtk_widget_show( ucv );
   g_object_set( ucv, "unicap-handle", handle, NULL );
   g_signal_connect( G_OBJECT( ucv ), "destroy", G_CALLBACK( ucview_destroy_event_cb ), NULL );
   
   return FALSE;
}



GtkUIManager *ucview_get_ui_manager( UCViewWindow *ucv )
{
   return ucv->priv->ui;
}

GtkWidget *ucview_get_statusbar( UCViewWindow *ucv )
{
   return ucv->priv->statusbar;
}

/* GtkWidget *ucview_get_display( UCViewWindow *ucv ) */
/* { */
/*    return ucv->priv->display; */
/* } */



static gboolean register_or_call_service( )
{
#if HAVE_DBUS
   DBusGConnection *connection;
   DBusGProxy *proxy;
   GError *error = NULL;
   gboolean quit = TRUE;

   connection = dbus_g_bus_get( DBUS_BUS_SESSION, &error );
   
   if( !connection )
   {
      g_warning( _("Could not initialize DBUS connection :-(\n") );
      return open_window_no_dbus();
   }
   else
   {
      proxy = dbus_g_proxy_new_for_name_owner( connection,
					       "org.unicapimaging.UCView",
					       "/org/unicapimaging/UCView",
					       "org.unicapimaging.UCView", 
					       &error );
      if( !error )
      {
	 if( !org_unicapimaging_UCView_new_window( proxy, &error ) )
	 {
	    g_error( _("Failed to open device dialog!\n") );
	 }
      }
      else
      {
	 ServerObject *dbus_server;
	 
	 g_error_free( error );
	 
	 icons_add_stock_items();
	 dbus_server = g_object_new( SERVER_OBJECT_TYPE, NULL );
	 proxy = dbus_g_proxy_new_for_name_owner( connection,
						  "org.unicapimaging.UCView",
						  "/org/unicapimaging/UCView",
						  "org.unicapimaging.UCView", 
						  &error );
	 
	 org_unicapimaging_UCView_new_window_async( proxy, (org_unicapimaging_UCView_new_window_reply)new_window_async_reply, NULL );
	 quit = FALSE;
      }
   
      g_object_unref (proxy);
   }

   return quit;
#else
   return open_window_no_dbus();
#endif
}

int main( int argc, char**argv )
{

   g_thread_init(NULL);
   gdk_threads_init();
   gdk_threads_enter();
   gtk_init (&argc, &argv);

   setlocale( LC_ALL, "" );

   bindtextdomain( GETTEXT_PACKAGE, UCVIEW_LOCALEDIR );
   bind_textdomain_codeset( GETTEXT_PACKAGE, "UTF-8" );
   textdomain( GETTEXT_PACKAGE );
   
   if( !register_or_call_service() )
   {
      gtk_main();
   }

   gdk_threads_leave();
   return 0;
}

