# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright_cap (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All right_caps reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import pgm

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.graph.image import Image


class SlicedImageHorizontal(Widget):

    def __init__(self):
        super(SlicedImageHorizontal, self).__init__()

        self.body = Image()
        self.add(self.body)
        self.body.bg_a = 0
        self.body.layout = pgm.IMAGE_FILLED
        self.body.interp = pgm.IMAGE_NEAREST
        self.body.connect("pixbuf-loaded", lambda x: self._layout())

        self.right_cap = Image()
        self.add(self.right_cap)
        self.right_cap.bg_a = 0
        self.right_cap.layout = pgm.IMAGE_FILLED
        self.right_cap.interp = pgm.IMAGE_NEAREST
        self.right_cap.connect("pixbuf-loaded", lambda x: self._layout())

        self.left_cap = Image()
        self.add(self.left_cap)
        self.left_cap.bg_a = 0
        self.left_cap.layout = pgm.IMAGE_FILLED
        self.left_cap.interp = pgm.IMAGE_NEAREST
        self.left_cap.connect("pixbuf-loaded", lambda x: self._layout())

    def width__get(self):
        return super(SlicedImageHorizontal, self).width__get()

    def width__set(self, width):
        super(SlicedImageHorizontal, self).width__set(width)
        self._layout()

    width = property(width__get, width__set)

    def height__get(self):
        return super(SlicedImageHorizontal, self).height__get()

    def height__set(self, height):
        super(SlicedImageHorizontal, self).height__set(height)
        self._layout()

    height = property(height__get, height__set)

    def _layout(self):
        if not self.left_cap.storage_type == pgm.IMAGE_EMPTY and \
           not self.right_cap.storage_type == pgm.IMAGE_EMPTY and \
           not self.body.storage_type == pgm.IMAGE_EMPTY:
            self._left_cap_layout()
            self._right_cap_layout()
            self._body_layout()
            self.left_cap.visible = True
            self.right_cap.visible = True
            self.body.visible = True

    def _left_cap_layout(self):
        self.left_cap.alignment = pgm.IMAGE_LEFT
        self.left_cap.x = 0.0
        self.left_cap.y = 0.0
        self.left_cap.height = 1.0
        self._respect_aspect_ratio(self.left_cap)

    def _body_layout(self):
        self.body.x = self.left_cap.x + self.left_cap.width
        self.body.y = 0.0
        self.body.height = 1.0
        self.body.width = 1.0-self.left_cap.width-self.right_cap.width

    def _right_cap_layout(self):
        self.right_cap.alignment = pgm.IMAGE_RIGHT
        self.right_cap.height = 1.0
        self._respect_aspect_ratio(self.right_cap)
        self.right_cap.x = 1.0-self.right_cap.width
        self.right_cap.y = 0.0

    def _respect_aspect_ratio(self, image):
        real_aspect_ratio = image.aspect_ratio[0]/float(image.aspect_ratio[1])
        current_aspect_ratio = image.absolute_width/image.absolute_height
        deformation = real_aspect_ratio/current_aspect_ratio

        # current_aspect_ratio is made equal to real_aspect_ratio
        image.width *= deformation

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        widget = cls()
        widget.left_cap.set_from_file("left_cap.png")
        widget.right_cap.set_from_file("right_cap.png")
        widget.body.set_from_file("body.png")
        widget.visible = True
        return widget


if __name__ == '__main__':
    import logging
    logger = logging.getLogger()
    logger.setLevel(logging.DEBUG)

    i = SlicedImageHorizontal.demo()

    try:
        __IPYTHON__
    except NameError:
        pgm.main()
