/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment X11 pixmaps as image system buffers example
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <cairo.h>
#include <cairo-xlib.h>
#include <pgm/pgm.h>

static PgmDrawable *img1 = NULL;
static PgmDrawable *img2 = NULL;
static Pixmap *raw_pixmap = NULL;
static Pixmap *cairo_pixmap = NULL;
static Display *dpy = NULL;

static void
keypress_event_cb (PgmViewport *viewport,
                   PgmEventKey *event,
                   gpointer data)
{
  if (event->type == PGM_KEY_PRESS)
    {
      switch (event->keyval)
        {
          /* Quit */
        case PGM_q:
        case PGM_Escape:
          pgm_main_quit ();
          break;

        default:
          break;
        }
    }
}

static gboolean
render_raw_xlib_pixmap (gpointer data)
{
  static const gchar *color_string[6] =
    { "#FF0000", "#00FF00", "#0000FF", "#FFFF00", "#FF00FF", "#00FFFF" };
  XColor color[6];
  Colormap colormap;
  unsigned int i;
  GC gc;

  colormap = DefaultColormap (dpy, 0);

  for (i = 0; i < 4; i++)
    {
      XParseColor (dpy, colormap, color_string[rand () % 6], &color[i]);
      XAllocColor (dpy, colormap, &color[i]);
    }

  pgm_image_system_buffer_lock (PGM_IMAGE (img1));

  gc = XCreateGC (dpy, *raw_pixmap, 0, NULL);

  /* Render */
  XSetForeground (dpy, gc, color[0].pixel);
  XFillRectangle (dpy, *raw_pixmap, gc, 0, 0, 175, 175);
  XSetForeground (dpy, gc, color[1].pixel);
  XFillRectangle (dpy, *raw_pixmap, gc, 175, 0, 350, 175);
  XSetForeground (dpy, gc, color[2].pixel);
  XFillRectangle (dpy, *raw_pixmap, gc, 0, 175, 175, 350);
  XSetForeground (dpy, gc, color[3].pixel);
  XFillRectangle (dpy, *raw_pixmap, gc, 175, 175, 350, 350);

  XFreeGC (dpy, gc);
  XSync (dpy, False);

  pgm_image_system_buffer_unlock (PGM_IMAGE (img1));

  return TRUE;
}

static gboolean
render_cairo_xlib_pixmap (int screen)
{
  double x1 = 140.0, y1 = 315.0, x2 = 210.0, y2 = 35.0, x3 = 315.0, y3 = 175.0;
  double x = 35.0, y = 175.0;
  cairo_surface_t *surface;
  cairo_t *cr;

  pgm_image_system_buffer_lock (PGM_IMAGE (img2));

  /* Create surface and context */
  surface = cairo_xlib_surface_create (dpy, *cairo_pixmap,
                                       DefaultVisual (dpy, screen),
                                       350, 350);
  if (!surface)
    return FALSE;
  cr = cairo_create (surface);
  if (!cr)
    {
      cairo_surface_destroy (surface);
      return FALSE;
    }

  /* Render */
  cairo_set_source_rgba (cr, 1, 1, 1, 1);
  cairo_paint (cr);
  cairo_set_source_rgba (cr, 0, 0, 0, 1);
  cairo_move_to (cr, x, y);
  cairo_curve_to (cr, x1, y1, x2, y2, x3, y3);
  cairo_set_line_width (cr, 10.0);
  cairo_stroke (cr);
  cairo_set_source_rgba (cr, 1, 0.2, 0.2, 0.6);
  cairo_set_line_width (cr, 6.0);
  cairo_move_to (cr,x,y);
  cairo_line_to (cr,x1,y1);
  cairo_move_to (cr,x2,y2);
  cairo_line_to (cr,x3,y3);
  cairo_stroke (cr);

  /* Then destroy the surface and context */
  cairo_destroy (cr);
  cairo_surface_destroy (surface);

  XSync (dpy, False);

  pgm_image_system_buffer_unlock (PGM_IMAGE (img2));

  return TRUE;
}

static gboolean
update_images_opacity (gpointer data)
{
  static gfloat angle = 3.1415f / 2.0f;
  gfloat opacity;

  opacity = (((sin (angle) + 1.0f) / 2.0f) * 200.0f) + 55.0f;
  angle += 0.1f;

  pgm_drawable_set_opacity (img1, opacity);
  pgm_drawable_set_opacity (img2, opacity);

  return TRUE;
}

int
main (int argc,
      char *argv[])
{
  PgmDrawable *text1 = NULL, *text2 = NULL;
  PgmViewport *viewport = NULL;
  PgmCanvas *canvas = NULL;
  gulong caps_mask = 0;
  int root, screen;

  /* Init */
  pgm_init (&argc, &argv);

/* OpenGL viewport creation */
  pgm_viewport_factory_make ("opengl", &viewport);
  if (!viewport)
    {
      g_print ("Cannot create the 'opengl' viewport\n");
      return -1;
    }
  pgm_viewport_set_size (viewport, 775, 415);
  pgm_viewport_set_title (viewport, "Xlib pixmaps as image system buffers");
  g_signal_connect (G_OBJECT (viewport), "key-press-event",
                    G_CALLBACK (keypress_event_cb), NULL);
  g_signal_connect (G_OBJECT (viewport), "delete-event",
                    G_CALLBACK (pgm_main_quit), NULL);

  /* Exit if system buffers are not supported */
  pgm_viewport_get_caps_mask (viewport, &caps_mask);
  if (!(caps_mask & PGM_VIEWPORT_X11_SYSTEM_BUFFER))
    {
      g_print ("X11 system buffer not supported by the OpenGL plugin, your "
               "graphical card has to support non-power-of-two textures and "
               "your GLX implementation has to support the texture_from_pixmap "
               "extension. Try to set the LIBGL_ALWAYS_INDIRECT environment "
               "variable with Mesa DRI drivers\n");
      gst_object_unref (viewport);
      pgm_deinit ();
      return -1;
    }

  dpy = XOpenDisplay (g_getenv ("DISPLAY"));
  if (!dpy)
    {
      g_print ("couldn't open default display\n");
      gst_object_unref (viewport);
      pgm_deinit ();
      return -1;
    }
  screen = DefaultScreen (dpy);
  root = RootWindow (dpy, screen);

  /* Create pixmaps */
  raw_pixmap = g_slice_new (Pixmap);
  *raw_pixmap = XCreatePixmap (dpy, root, 350, 350, 24);
  cairo_pixmap = g_slice_new (Pixmap);
  *cairo_pixmap = XCreatePixmap (dpy, root, 350, 350, 24);

  /* 1st text for X11 Pixmap */
  text1 = pgm_text_new ("<span weight='bold'>Xlib rendered pixmap:</span>");
  pgm_drawable_set_size (text1, 350.0f, 20.0f);
  pgm_drawable_set_position (text1, 25.0f, 15.0f, 0.0f);
  pgm_drawable_set_bg_color (text1, 255, 255, 255, 0);
  pgm_drawable_set_fg_color (text1, 255, 255, 255, 255);
  pgm_drawable_show (text1);

  /* 1st image for X11 Pixmap */
  img1 = pgm_image_new ();
  pgm_image_set_from_system_buffer (PGM_IMAGE (img1), PGM_IMAGE_RGB, 350, 350,
                                    raw_pixmap);
  pgm_drawable_set_size (img1, 350.0f, 350.0f);
  pgm_drawable_set_position (img1, 25.0f, 40.0f, 0.0f);
  pgm_drawable_set_fg_color (img1, 255, 255, 255, 255);
  pgm_drawable_set_bg_color (img1, 255, 255, 255, 0);
  pgm_drawable_show (img1);

  /* 2nd text for Cairo X11 Pixmap */
  text2 = pgm_text_new ("<span weight='bold'>Cairo Xlib rendered pixmap:</span>");
  pgm_drawable_set_size (text2, 350.0f, 20.0f);
  pgm_drawable_set_position (text2, 400.0f, 15.0f, 0.0f);
  pgm_drawable_set_bg_color (text2, 255, 255, 255, 0);
  pgm_drawable_set_fg_color (text2, 255, 255, 255, 255);
  pgm_drawable_show (text2);

  /* 2nd image for Cairo X11 Pixmap */
  img2 = pgm_image_new ();
  pgm_image_set_from_system_buffer (PGM_IMAGE (img2), PGM_IMAGE_RGB, 350, 350,
                                    cairo_pixmap);
  pgm_drawable_set_size (img2, 350.0f, 350.0f);
  pgm_drawable_set_position (img2, 400.0f, 40.0f, 0.0f);
  pgm_drawable_set_fg_color (img2, 255, 255, 255, 255);
  pgm_drawable_set_bg_color (img2, 255, 255, 255, 0);
  pgm_drawable_show (img2);

  /* Render pixmaps */
  if (!render_raw_xlib_pixmap (NULL))
    {
      g_print ("Cannot render the raw xlib pixmap\n");
      return -1;
    }
  if (!render_cairo_xlib_pixmap (screen))
    {
      g_print ("Cannot render the cairo xlib pixmap\n");
      return -1;
    }

  /* Canvas handling */
  canvas = pgm_canvas_new ();
  pgm_canvas_set_size (canvas, 775.0f, 415.0f);
  pgm_viewport_set_canvas (viewport, canvas);
  pgm_canvas_add_many (canvas, PGM_DRAWABLE_MIDDLE,
                       text1, img1, text2, img2, NULL);

  /* Add a source to modify the X11 pixmap each 500 ms */
  g_timeout_add (500, render_raw_xlib_pixmap, NULL);
  g_timeout_add (40, update_images_opacity, NULL);

  /* Main loop */
  pgm_viewport_show (viewport);
  pgm_main ();

  /* Deinit */
  gst_object_unref (canvas);
  gst_object_unref (viewport);
  XFreePixmap (dpy, *raw_pixmap);
  g_slice_free (Pixmap, raw_pixmap);
  XFreePixmap (dpy, *cairo_pixmap);
  g_slice_free (Pixmap, cairo_pixmap);
  XCloseDisplay (dpy);
  pgm_deinit ();

  return 0;
}
