/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * hgfsUtil.h must be included before vm_basic_asm.h, as hgfsUtil.h
 * includes kernel headers on Linux.  That is, vmware.h must come after
 * hgfsUtil.h.
 */
#include "hgfsUtil.h"
#include "vmware.h"
#include "vm_basic_asm.h"


/*
 * hgfsUtil.c --
 *
 *    Conversion routines between Unix time and Windows NT time.
 *    The former is in units of seconds since midnight 1/1/1970, while the
 *    latter is in units of 100 nanoseconds since midnight 1/1/1601.
 */


/*
 * NT time of the Unix epoch:
 * midnight January 1, 1970 UTC
 */
#define UNIX_EPOCH ((((uint64)369 * 365) + 89) * 24 * 3600 * 10000000)

/*
 * NT time of the Unix 32 bit signed time_t wraparound:
 * 03:14:07 January 19, 2038 UTC
 */
#define UNIX_S32_MAX (UNIX_EPOCH + (uint64)0x80000000 * 10000000)


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsConvertToNtTime --
 *
 *    Convert from Unix time to Windows NT time.
 *
 * Results:
 *    The time in Windows NT format.
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

uint64
HgfsConvertToNtTime(time_t unixTime, // IN: Time in Unix format (seconds)
		    long   nsec)     // IN: nanoseconds
{
   return (uint64)unixTime * 10000000 + nsec / 100 + UNIX_EPOCH;
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsConvertFromNtTimeNsec --
 *
 *    Convert from Windows NT time to Unix time. If NT time is outside of
 *    UNIX time range (1970-2038), returned time is nearest time valid in
 *    UNIX.
 *
 * Results:
 *    0        on success
 *    non-zero if NT time is outside of valid range for UNIX
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

int
HgfsConvertFromNtTimeNsec(struct timespec *unixTime, // OUT: Time in UNIX format
			  uint64 ntTime) // IN: Time in Windows NT format
{
#ifndef VM_X86_64
   uint32 sec;
   uint32 nsec;

   ASSERT(unixTime);
   /* We assume that time_t is 32bit */
   ASSERT(sizeof (unixTime->tv_sec) == 4);

   /* Cap NT time values that are outside of Unix time's range */

   if (ntTime >= UNIX_S32_MAX) {
      unixTime->tv_sec = 0x7FFFFFFF;
      unixTime->tv_nsec = 0;
      return 1;
   }
#else
   ASSERT(unixTime);
#endif

   if (ntTime < UNIX_EPOCH) {
      unixTime->tv_sec = 0;
      unixTime->tv_nsec = 0;
      return -1;
   }

#ifndef VM_X86_64
   Div643232(ntTime - UNIX_EPOCH, 10000000, &sec, &nsec);
   unixTime->tv_sec = sec;
   unixTime->tv_nsec = nsec * 100;
#else
   unixTime->tv_sec = (ntTime - UNIX_EPOCH) / 10000000;
   unixTime->tv_nsec = ((ntTime - UNIX_EPOCH) % 10000000) * 100;
#endif

   return 0;
}


/*
 *-----------------------------------------------------------------------------
 *
 * HgfsConvertFromNtTime --
 *
 *    Convert from Windows NT time to Unix time.
 *
 * Results:
 *    0       on success
 *    nonzero if time is not representable on UNIX 
 *
 * Side effects:
 *    None
 *
 *-----------------------------------------------------------------------------
 */

int
HgfsConvertFromNtTime(time_t *unixTime, // OUT: Time in UNIX format
		      uint64 ntTime) // IN: Time in Windows NT format
{
   struct timespec tm;
   int ret;
   
   ret = HgfsConvertFromNtTimeNsec(&tm, ntTime);
   *unixTime = tm.tv_sec;
   return ret;
}


#undef UNIX_EPOCH
#undef UNIX_S32_MAX
