/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib/gi18n.h>

#include <gmenu-tree.h>

#include "launcher-application-chooser.h"
#include "launcher-app.h"
#include "launcher-defines.h"

#include "launcher-util.h"

G_DEFINE_TYPE (LauncherApplicationChooser, launcher_application_chooser, GTK_TYPE_DIALOG);

#define LAUNCHER_APPLICATION_CHOOSER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_APPLICATION_CHOOSER, \
  LauncherApplicationChooserPrivate))

struct _LauncherApplicationChooserPrivate
{
  GMenuTree *apps;
  GMenuTree *prefs;
  
  GtkTreeStore *store;
  GtkWidget    *view;
};

enum 
{
  COL_ICON = 0,
  COL_NAME,
  COL_DATA,

  N_COLS
};

#define ICON_SIZE 32

gboolean 
launcher_application_chooser_get_selected (LauncherApplicationChooser *chooser,
                                           gchar                      **name,
                                           gchar                      **exec,
                                           gchar                      **icon)
{
  LauncherApplicationChooserPrivate *priv;
  GtkTreeSelection *select;
  GtkTreeIter iter = { 0, };
  gpointer data;
  GtkTreeModel *model;

  g_return_val_if_fail (LAUNCHER_IS_APPLICATION_CHOOSER (chooser), FALSE);
  priv = chooser->priv;

  model = GTK_TREE_MODEL (priv->store);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (priv->view));
  
  if (!gtk_tree_selection_get_selected (select, &model, &iter))
  {
    return FALSE;
  }

  gtk_tree_model_get (GTK_TREE_MODEL (priv->store),
                      &iter,
                      COL_DATA, &data,
                      -1);

  if (!data)
    return FALSE;

  *name = g_strdup (gmenu_tree_entry_get_name (GMENU_TREE_ENTRY (data)));
  *exec = g_strdup (gmenu_tree_entry_get_exec (GMENU_TREE_ENTRY (data)));
  *icon = g_strdup (gmenu_tree_entry_get_icon (GMENU_TREE_ENTRY (data)));

  return TRUE;
}

static void
add_item (GtkTreeStore *store, 
          const gchar  *icon_name,
          const gchar  *name,
          const gchar  *comment,
          GtkTreeIter  *child, 
          GtkTreeIter  *parent,
          gpointer      data)
{
  GdkPixbuf *icon;
  gchar *ename;
  gchar *ecomment;
  gchar *markup;

  icon = get_icon (icon_name, ICON_SIZE);
  ename = g_markup_escape_text (name, -1);
  ecomment = g_markup_escape_text (comment ?  comment : " ", -1);
  markup = g_strdup_printf ("<b>%s</b>\n%s", ename, ecomment);

  gtk_tree_store_append (store, child, parent);
  gtk_tree_store_set (store, child, 
                      COL_ICON, icon, COL_NAME, markup, COL_DATA, data,
                      -1);

  g_free (ename);
  g_free (ecomment);
  g_free (markup);
  if (icon) g_object_unref (icon);
}


static void
load_dir (GtkTreeStore *store, GMenuTreeDirectory *dir, GtkTreeIter *parent)
{
  GSList      *entries, *e;
  GtkTreeIter  cat = { 0, };
    
  entries = e = gmenu_tree_directory_get_contents (dir);
  
  add_item (store,
            gmenu_tree_directory_get_icon (dir),
            gmenu_tree_directory_get_name (dir),
            gmenu_tree_directory_get_comment (dir),
            &cat, parent,
            NULL);

  for (e = entries; e; e = e->next)
  {
    GMenuTreeItem *item = e->data;
    GtkTreeIter iter = { 0, };
        
    switch (gmenu_tree_item_get_type (item))
    {
      case GMENU_TREE_ITEM_DIRECTORY:
        load_dir (store, GMENU_TREE_DIRECTORY (item), &cat);
        break;
      case GMENU_TREE_ITEM_ENTRY:
        add_item (store,
                  gmenu_tree_entry_get_icon (GMENU_TREE_ENTRY (item)),
                  gmenu_tree_entry_get_name (GMENU_TREE_ENTRY (item)),
                  gmenu_tree_entry_get_comment (GMENU_TREE_ENTRY (item)),
                  &iter, &cat,
                  item);
        break;
      default:
        break;
    }
  }
  g_slist_free (entries);
}

static void
populate_view (LauncherApplicationChooser *self)
{
  LauncherApplicationChooserPrivate *priv = self->priv;
  GtkTreeStore *store = priv->store;
  GMenuTree *tree;
  GMenuTreeDirectory *dir;
  GSList *entries, *e;
  
  tree=priv->apps=gmenu_tree_lookup("applications.menu", GMENU_TREE_FLAGS_NONE);
  dir = gmenu_tree_get_root_directory (tree);
  entries = gmenu_tree_directory_get_contents (dir);

  for (e = entries; e; e = e->next)
  {
    GMenuTreeItem *item = e->data;
    GtkTreeIter iter = { 0, };
   
    switch (gmenu_tree_item_get_type (item))
    {
      case GMENU_TREE_ITEM_DIRECTORY:
        load_dir (store, GMENU_TREE_DIRECTORY (item), NULL);
        break;
      case GMENU_TREE_ITEM_ENTRY:
        add_item (store,
                  gmenu_tree_entry_get_icon (GMENU_TREE_ENTRY (item)),
                  gmenu_tree_entry_get_name (GMENU_TREE_ENTRY (item)),
                  gmenu_tree_entry_get_comment (GMENU_TREE_ENTRY (item)),
                  &iter, NULL,
                  item);
        break;
      default:
        break;
    }
  }
  g_slist_free (entries);

  tree = priv->prefs=gmenu_tree_lookup ("settings.menu", GMENU_TREE_FLAGS_NONE);
  dir = gmenu_tree_get_root_directory (tree);
  entries = gmenu_tree_directory_get_contents (dir);

  for (e = entries; e; e = e->next)
  {
    GMenuTreeItem *item = e->data;
    GtkTreeIter iter = { 0, };
   
    switch (gmenu_tree_item_get_type (item))
    {
      case GMENU_TREE_ITEM_DIRECTORY:
        load_dir (store, GMENU_TREE_DIRECTORY (item), NULL);
        break;
      case GMENU_TREE_ITEM_ENTRY:
        add_item (store,
                  gmenu_tree_entry_get_icon (GMENU_TREE_ENTRY (item)),
                  gmenu_tree_entry_get_name (GMENU_TREE_ENTRY (item)),
                  gmenu_tree_entry_get_comment (GMENU_TREE_ENTRY (item)),
                  &iter, NULL,
                  item);
        break;
      default:
        break;
    }
  }
  g_slist_free (entries);
}

static void
on_tree_view_cursor_changed (GtkTreeView *view, LauncherApplicationChooser *app)
{
  GtkTreePath *path;

  gtk_tree_view_get_cursor (view, &path, NULL);
  gtk_tree_view_expand_to_path (view, path);
}

/* GObject stuff */
static void
launcher_application_chooser_finalize (GObject *application_chooser)
{
  LauncherApplicationChooserPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_APPLICATION_CHOOSER (application_chooser));
  priv = LAUNCHER_APPLICATION_CHOOSER_GET_PRIVATE (application_chooser);

  gmenu_tree_unref (priv->apps);
  gmenu_tree_unref (priv->prefs);

  G_OBJECT_CLASS (launcher_application_chooser_parent_class)->finalize (application_chooser);
}

static void
launcher_application_chooser_class_init (LauncherApplicationChooserClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  
  obj_class->finalize = launcher_application_chooser_finalize;

  g_type_class_add_private (obj_class, 
                            sizeof (LauncherApplicationChooserPrivate));
}
      
static void
launcher_application_chooser_init (LauncherApplicationChooser *self)
{
  LauncherApplicationChooserPrivate *priv;
  GtkWidget *window = GTK_WIDGET (self);
  GtkWidget *align, *scroll, *view;
  GtkTreeViewColumn *col;
  GtkCellRenderer *renderer;
  
  priv = self->priv = LAUNCHER_APPLICATION_CHOOSER_GET_PRIVATE (self);

  gtk_widget_set_size_request (GTK_WIDGET (self), 400, 500);

  align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (align), 12, 0, 12, 12);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (window)->vbox), align);
  g_object_set (GTK_DIALOG(window)->vbox,  "spacing", 8, NULL);  
    
  gtk_widget_show_all (align);

  priv->store = gtk_tree_store_new (N_COLS,
                                    GDK_TYPE_PIXBUF,
                                    G_TYPE_STRING,
                                    G_TYPE_POINTER);

  scroll = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll),
                                  GTK_POLICY_AUTOMATIC,
                                  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (align), scroll);
  gtk_widget_show (scroll);

  view = priv->view = gtk_tree_view_new_with_model 
                                                (GTK_TREE_MODEL (priv->store));
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (view), FALSE);
  gtk_container_add (GTK_CONTAINER (scroll), view);

  renderer = gtk_cell_renderer_pixbuf_new ();
  col = gtk_tree_view_column_new_with_attributes ("",
                                                  renderer,
                                                  "pixbuf", COL_ICON,
                                                  NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (view), col);

  renderer = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new_with_attributes ("",
                                                  renderer,
                                                  "markup", COL_NAME,
                                                  NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (view), col);
  
  gtk_widget_show_all (view);

  g_signal_connect (GTK_TREE_VIEW (view), "cursor-changed", 
                    G_CALLBACK (on_tree_view_cursor_changed), self);

  populate_view (self);
}

GtkWidget *
launcher_application_chooser_new (void)

{
  GtkWidget *application_chooser = NULL;

  application_chooser = g_object_new (LAUNCHER_TYPE_APPLICATION_CHOOSER, 
                                  "window-position", GTK_WIN_POS_CENTER_ALWAYS,
                                  "title", _("Choose an application to add to the category"),
                                  NULL);

  return application_chooser;
}
