/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib/gi18n.h>
#include <libgnomeui/libgnomeui.h>

#include "launcher-category-editor.h"
#include "launcher-app.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-menu.h"
#include "launcher-util.h"

G_DEFINE_TYPE (LauncherCategoryEditor, launcher_category_editor, GTK_TYPE_DIALOG);

#define LAUNCHER_CATEGORY_EDITOR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_CATEGORY_EDITOR, \
  LauncherCategoryEditorPrivate))

struct _LauncherCategoryEditorPrivate
{
  LauncherMenuCategory *cat;

  GtkWidget *name_entry;
  GtkWidget *icon_button;
  GtkWidget *icon;
  gchar *icon_filename;
};

enum
{
	PROP_0,
	PROP_NAME,
  PROP_ICON
};

static void
on_entry_activate (GtkEntry *entry, LauncherCategoryEditor *self)
{
  const gchar *text;

  text = gtk_entry_get_text (entry);

  if (text)
  {
    gtk_dialog_response (GTK_DIALOG (self), GTK_RESPONSE_YES);
  }
}

static void
on_update_preview (GtkFileChooser *chooser, GtkImage *preview)
{
  char *filename;
  GdkPixbuf *pixbuf = NULL;
  gboolean have_preview;
  gint cat_height = (launcher_config_get_default ())->cat_height;
  gint icon_size;

  filename = gtk_file_chooser_get_preview_filename (chooser);

  icon_size = CAT_ICON_SIZE();
  if (filename)
    pixbuf = gdk_pixbuf_new_from_file_at_size (filename, icon_size, icon_size, NULL);
  have_preview = (pixbuf != NULL);
  g_free (filename);

  gtk_image_set_from_pixbuf (preview, pixbuf);
  if (pixbuf)
    g_object_unref (pixbuf);

  gtk_file_chooser_set_preview_widget_active (chooser, have_preview);
}

static void
on_icon_clicked (GtkButton *button, LauncherCategoryEditor *self)
{
  GtkWidget *chooser;
  GtkWidget *preview;
  GtkFileFilter *filter;

  filter = gtk_file_filter_new ();
  gtk_file_filter_add_pixbuf_formats (filter);

  preview = gtk_image_new_from_pixbuf (gtk_image_get_pixbuf (GTK_IMAGE (self->priv->icon)));

  chooser = gtk_file_chooser_dialog_new (_("Choose category icon"),
                                         GTK_WINDOW (self),
                                         GTK_FILE_CHOOSER_ACTION_OPEN,
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                         NULL);
  gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (chooser), self->priv->icon_filename);
  gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (chooser), filter);
  gtk_file_chooser_set_preview_widget (GTK_FILE_CHOOSER (chooser), preview);
  g_signal_connect (GTK_FILE_CHOOSER (chooser), "update-preview",
                    G_CALLBACK (on_update_preview), preview);

  if (gtk_dialog_run (GTK_DIALOG (chooser)) == GTK_RESPONSE_ACCEPT)
  {
    gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
    g_object_set (self, "category-icon", filename, NULL);
    g_free (filename);
  }

  gtk_widget_destroy (chooser);
}

static void
set_icon_path (LauncherCategoryEditor *self, const gchar *icon_name)
{
  GtkIconTheme *theme = gtk_icon_theme_get_default ();
  GtkIconInfo *info = NULL;
  const gchar *filename = NULL;
  gint icon_size;
  gint cat_height = (launcher_config_get_default ())->cat_height;
  GdkPixbuf *pixbuf;

  icon_size = CAT_ICON_SIZE();
  info = gtk_icon_theme_lookup_icon (theme, icon_name, 
                                     icon_size, 
                                     GTK_ICON_LOOKUP_FORCE_SVG);

  if (info)
    filename = gtk_icon_info_get_filename (info);
  else
    filename = icon_name;

  pixbuf = gdk_pixbuf_new_from_file_at_size (filename, icon_size, icon_size, NULL);

  if (pixbuf)
  {
    g_free (self->priv->icon_filename);
    self->priv->icon_filename = g_strdup (filename);
    gtk_image_set_from_pixbuf (GTK_IMAGE (self->priv->icon), pixbuf);
    g_object_unref (pixbuf);
  }

  if (info)
    gtk_icon_info_free (info);
}

void
launcher_category_editor_set_name (LauncherCategoryEditor *self, 
                                   const gchar            *name)
{
  g_return_if_fail (LAUNCHER_IS_CATEGORY_EDITOR (self));

  gtk_entry_set_text (GTK_ENTRY (self->priv->name_entry), name);
}

void
launcher_category_editor_set_icon (LauncherCategoryEditor *self, 
                                   const gchar            *icon_name)
{
  g_return_if_fail (LAUNCHER_IS_CATEGORY_EDITOR (self));

  set_icon_path (self, icon_name);
}

/* GObject stuff */
static void
launcher_category_editor_finalize (GObject *category_editor)
{
  LauncherCategoryEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_CATEGORY_EDITOR (category_editor));
  priv = LAUNCHER_CATEGORY_EDITOR_GET_PRIVATE (category_editor);

  G_OBJECT_CLASS (launcher_category_editor_parent_class)->finalize (category_editor);
}

static void
set_property (GObject      *object, 
              guint         prop_id,
              const GValue *value, 
              GParamSpec   *pspec)
{
  LauncherCategoryEditor *self = LAUNCHER_CATEGORY_EDITOR (object);
  LauncherCategoryEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_CATEGORY_EDITOR (object));
  priv = LAUNCHER_CATEGORY_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_NAME:
      launcher_category_editor_set_name (self, g_value_get_string (value));
      break;
    case PROP_ICON:
      launcher_category_editor_set_icon (self, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
get_property (GObject      *object, 
              guint         prop_id,
              GValue       *value, 
              GParamSpec   *pspec)
{
  LauncherCategoryEditorPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_CATEGORY_EDITOR (object));
  priv = LAUNCHER_CATEGORY_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_NAME:
      g_value_set_string (value, 
                          gtk_entry_get_text (GTK_ENTRY (priv->name_entry)));
      break;
    case PROP_ICON:
      g_value_set_string (value, priv->icon_filename);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
launcher_category_editor_class_init (LauncherCategoryEditorClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  
  obj_class->finalize = launcher_category_editor_finalize;
  obj_class->get_property = get_property;
  obj_class->set_property = set_property;


  /* Class properties */
  g_object_class_install_property (obj_class,
                                   PROP_NAME,
                                   g_param_spec_string ("category-name",
                                   "Category Name",
                                   "Name of category",
                                   NULL,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_ICON,
                                   g_param_spec_string ("category-icon",
                                   "Category Icon",
                                   "Icon of category",
                                   NULL,
                                   G_PARAM_READWRITE));


  g_type_class_add_private (obj_class, sizeof (LauncherCategoryEditorPrivate));
}

      
static void
launcher_category_editor_init (LauncherCategoryEditor *self)
{
  LauncherCategoryEditorPrivate *priv;
  GtkWidget *window = GTK_WIDGET (self);
  GtkWidget *align, *vbox, *hbox;
  GtkWidget *label, *entry, *icon;
  GtkSizeGroup *group;
    
  priv = self->priv = LAUNCHER_CATEGORY_EDITOR_GET_PRIVATE (self);

  launcher_app_lock (launcher_app_get_default (), TRUE);

  group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (align), 12, 0, 12, 12);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (window)->vbox), align);
  g_object_set (GTK_DIALOG(window)->vbox,  "spacing", 8, NULL);  
 
  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_add (GTK_CONTAINER (align), vbox);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, FALSE, FALSE);
  
  label = gtk_label_new (_("Name:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  entry = priv->name_entry = gtk_entry_new ();
  gtk_widget_set_size_request (entry, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox), entry, 0, TRUE, TRUE);
  gtk_widget_grab_focus (entry);
  g_signal_connect (entry, "activate", G_CALLBACK (on_entry_activate), self);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, FALSE, FALSE);
  
  label = gtk_label_new (_("Icon:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  icon = priv->icon_button = gtk_button_new ();
  priv->icon = gtk_image_new ();
  priv->icon_filename = NULL;
  gtk_button_set_image (GTK_BUTTON (icon), priv->icon);
  set_icon_path (self, "applications-internet");
  gtk_box_pack_start (GTK_BOX (hbox), icon, 0, TRUE, TRUE);
  g_signal_connect (icon, "clicked", G_CALLBACK (on_icon_clicked), self);
  
  gtk_widget_show_all (align);
}

GtkWidget *
launcher_category_editor_new (void)

{
  GtkWidget *category_editor = NULL;

  category_editor = g_object_new (LAUNCHER_TYPE_CATEGORY_EDITOR, 
                                  "window-position", GTK_WIN_POS_CENTER_ALWAYS,
                                  "default-width", 300,
                                  NULL);

  return category_editor;
}
