/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib/gi18n.h>

#include <tidy/tidy.h>
#include <string.h>

#include "launcher-icon.h"

#include "clutter-drag-server.h"

#include "launcher-app.h"
#include "launcher-behave.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-iconview.h"
#include "launcher-menu.h"
#include "launcher-shortcut-editor.h"
#include "launcher-startup.h"
#include "launcher-util.h"

G_DEFINE_TYPE (LauncherIcon, launcher_icon, CLUTTER_TYPE_GROUP);

#define LAUNCHER_ICON_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_ICON, \
  LauncherIconPrivate))

#define BG_PADDING 5

static ClutterActor *bg_texture = NULL;

struct _LauncherIconPrivate
{
  LauncherMenuApplication *app;

  ClutterTimeline       *slow_time;
  ClutterBehaviour      *behave;
  
  ClutterTimeline       *time;
  ClutterEffectTemplate *temp;

  ClutterTimeline       *fast_time;
  ClutterEffectTemplate *fast_temp;

  ClutterTimeline       *current;

  ClutterActor *bg;
  ClutterActor *label;
  ClutterActor *texture;
  ClutterActor *clone;

  ClutterActor *toggle;

  ClutterActor *options;
  gboolean      flipped;

  gint x;
  gint y;

  gboolean show_tooltip;
  ClutterActor *tooltip;
};

enum
{
  FLIPPED,

  LAST_SIGNAL
};
static guint _icon_signals[LAST_SIGNAL] = { 0 };

/*
 * Forwards
 */
static void     rotate            (LauncherIcon *icon);
static gboolean on_toggle_clicked (ClutterActor       *close,
                                   ClutterButtonEvent *event,
                                   LauncherIcon       *icon);

static ClutterActor *
clone_icon (LauncherIcon *icon)
{
  LauncherIconPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterActor *clone, *bg, *tex, *label;
  ClutterColor color = { 0xff, 0xff, 0xff, 0xff };
  
  priv = icon->priv;

  clone = clutter_group_new ();
  clutter_actor_set_size (clone, cfg->icon_width, cfg->icon_height);
  clutter_actor_show (clone);

  bg = clutter_rectangle_new_with_color (&color);
  clutter_container_add_actor (CLUTTER_CONTAINER (clone), bg);
  clutter_actor_set_size (bg, cfg->icon_width, cfg->icon_height);
  clutter_actor_set_position (bg, 0, 0);
  clutter_actor_set_opacity (bg, 50);
  clutter_actor_show (bg);
 
  tex = clutter_clone_texture_new (CLUTTER_TEXTURE (priv->texture));
  clutter_container_add_actor (CLUTTER_CONTAINER (clone), tex);
  clutter_actor_set_size (tex, CAW (priv->texture), CAH (priv->texture));
  clutter_actor_set_anchor_point_from_gravity (tex, CLUTTER_GRAVITY_NORTH);
  clutter_actor_set_position (tex,
                              clutter_actor_get_x (priv->texture),
                              clutter_actor_get_y (priv->texture));

  label = clutter_label_new_full (cfg->font_name,
                          clutter_label_get_text (CLUTTER_LABEL (priv->label)),
                                  &cfg->icon_font_color);
  clutter_label_set_line_wrap (CLUTTER_LABEL (label), FALSE);
  clutter_label_set_line_wrap_mode (CLUTTER_LABEL (label), 
                                    PANGO_WRAP_WORD_CHAR);
  clutter_label_set_alignment (CLUTTER_LABEL (label), PANGO_ALIGN_CENTER);
  clutter_container_add_actor (CLUTTER_CONTAINER (clone), label);
  clutter_actor_set_size (label, CAW (priv->label), CAH (priv->label));
  clutter_actor_set_position (label,
                              clutter_actor_get_x (priv->label),
                              clutter_actor_get_y (priv->label));

  clutter_actor_show_all (clone);
  return clone;
}

static gboolean
on_motion (ClutterActor       *actor,
           ClutterMotionEvent *event,
           LauncherIcon       *icon)
{
  LauncherIconPrivate *priv;
  ClutterActor *clone;
  ClutterActor *stage = clutter_stage_get_default ();
  gint x = 0, y = 9;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (event->modifier_state & CLUTTER_BUTTON1_MASK)
  {
    if (priv->clone)
    {
      clutter_actor_destroy (priv->clone);
    }

    clutter_actor_get_abs_position (actor, &x, &y);

    priv->clone = clone = clone_icon (icon);
    clutter_container_add_actor (CLUTTER_CONTAINER (stage),priv->clone);  
    clutter_actor_set_anchor_point (clone,
                                    event->x - x,
                                    event->y - y);
    clutter_actor_set_position (clone, event->x, event->y);
    clutter_actor_show (clone);

    g_object_set_data (G_OBJECT (clone), "LauncherIcon", icon);

    clutter_drag_server_begin_drag (clutter_drag_server_get_default(),
                                    clone,
                                    priv->app);
   return TRUE;
  }

  return FALSE;
}

static gboolean
on_app_clicked (ClutterActor       *actor,
                ClutterButtonEvent *event,
                LauncherIcon       *icon)
{
  LauncherIconPrivate *priv;
  gint icon_width = (launcher_config_get_default ())->icon_width;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (event->button == 1)
    ;
  else if (event->button == 3)
  {
    on_toggle_clicked (priv->toggle, event, icon);
    return TRUE;
  }
  else
    return FALSE;

  if (priv->flipped)
  {
    rotate (icon);
    return TRUE;
  }

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    return FALSE;
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }
 
  launcher_startup_launch_app (launcher_startup_get_default (),
                               priv->app);

  clutter_effect_rotate (priv->temp,
                         CLUTTER_ACTOR (icon),
                         CLUTTER_Y_AXIS,
                         360.0,
                         icon_width/2, 0, 0, 
                         CLUTTER_ROTATE_CW,
                         NULL, NULL);
  return TRUE;
}

static gboolean
tooltip_timeout (LauncherIcon *icon)
{
  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);

  if (clutter_actor_get_opacity (icon->priv->toggle) !=0
       && icon->priv->show_tooltip)
    clutter_effect_fade (icon->priv->temp, icon->priv->tooltip, 255, NULL, NULL);  

  return FALSE;
}

static gboolean
on_enter (ClutterActor          *actor,
          ClutterCrossingEvent  *event,
          LauncherIcon          *icon)
{
  LauncherIconPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }

  clutter_effect_fade (priv->temp, priv->toggle, 255, NULL, NULL);
  clutter_effect_fade (priv->temp, priv->bg, 130, NULL, NULL);
  
  if (priv->tooltip)
  {
    clutter_actor_set_position (priv->tooltip, event->x, event->y);
    priv->show_tooltip = TRUE;
    g_timeout_add (400, (GSourceFunc)tooltip_timeout, icon);
  }
  return TRUE;
}


static gboolean
on_leave (ClutterActor          *actor,
          ClutterCrossingEvent  *event,
          LauncherIcon          *icon)
{
  LauncherIconPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    clutter_timeline_stop (priv->current);
    g_object_unref (priv->current);
  }
  clutter_effect_fade (priv->temp, priv->toggle,  0, NULL, NULL);
  clutter_effect_fade (priv->temp, priv->bg, 50, NULL, NULL);

  if (priv->tooltip)
    clutter_effect_fade (priv->temp, priv->tooltip, 0, NULL, NULL);    

  priv->show_tooltip = FALSE;
  
  return TRUE;
}

void 
launcher_icon_set_show_target (LauncherIcon         *icon, 
                               ClutterCrossingEvent *event,
                               gboolean              show)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  if (show)
    on_enter (NULL, event, icon);
  else
    on_leave (NULL, event, icon);
}

static void
rotate_func (ClutterBehaviour *behave,
            guint32           alpha_value,
            LauncherIcon     *icon)
{
  LauncherIconPrivate *priv;
  gfloat factor;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  factor = (gfloat)alpha_value / CLUTTER_ALPHA_MAX_ALPHA;

  if (priv->flipped)
  {
    /* We're restoring the tile to it's original state */
    clutter_actor_set_rotation (priv->bg, CLUTTER_Y_AXIS,
                                180.0 + (180.0*factor),
                                CAW (priv->bg)/2, 0, 0);
    clutter_actor_set_rotation (priv->texture, CLUTTER_Y_AXIS,
                                180.0 + (180.0*factor),
                                0, 0, 0);
    clutter_actor_set_rotation (priv->label, CLUTTER_Y_AXIS,
                                180.0 + (180.0*factor),
                                CAW (priv->label)/2, 0, 0);
    
    if (factor * 180 > 90 && CLUTTER_IS_ACTOR (priv->options))
    {
      clutter_actor_set_opacity (priv->texture, 255);
      clutter_actor_set_opacity (priv->label, 255);
      clutter_actor_destroy (priv->options);
      priv->options = NULL;
    }

    if (CLUTTER_IS_ACTOR (priv->options))
    {
      clutter_actor_set_rotation (priv->options, CLUTTER_Y_AXIS,
                                  180.0*factor,
                                  CAW (priv->options)/2, 0, 0);
    }

  }
  else
  {
    /* We're flipping the tile */
    clutter_actor_set_rotation (priv->bg, CLUTTER_Y_AXIS,
                                180.0*factor,
                                CAW (priv->bg)/2, 0, 0);
    clutter_actor_set_rotation (priv->texture, CLUTTER_Y_AXIS,
                                180.0*factor,
                                0, 0, 0);
    clutter_actor_set_rotation (priv->label, CLUTTER_Y_AXIS,
                                180.0*factor,
                                CAW (priv->label)/2, 0, 0);
    if (factor * 180 > 90)
    {
      clutter_actor_set_opacity (priv->texture, 0);
      clutter_actor_set_opacity (priv->label, 0);
      clutter_actor_set_opacity (priv->options, 255);
    }

    clutter_actor_set_rotation (priv->options, CLUTTER_Y_AXIS,
                                180 + (180.0*factor),
                                CAW (priv->options)/2, 0, 0);
  }

  clutter_actor_queue_redraw (CLUTTER_ACTOR (icon));
}

static void
on_rotation_completed (ClutterTimeline *timeline, LauncherIcon *icon)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  priv->flipped = !priv->flipped;
}

static void
rotate (LauncherIcon *icon)
{
  LauncherIconPrivate *priv = icon->priv;

  if (!clutter_timeline_is_playing (priv->slow_time))
    clutter_timeline_start (priv->slow_time);
} 

static gboolean
on_launch_clicked (ClutterActor       *actor, 
                   ClutterButtonEvent *event, 
                   LauncherIcon       *icon)
{
  LauncherIconPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (CLUTTER_IS_TIMELINE (priv->current) 
        && clutter_timeline_is_playing (priv->current))
  {
    return FALSE;
  }
 
  launcher_startup_launch_app (launcher_startup_get_default (),
                               priv->app);

  rotate (icon);

  return TRUE;
}

static void
on_edit_reponse (GtkWidget *dialog, gint res, LauncherIcon *icon)
{
  if (!LAUNCHER_IS_ICON (icon))
  {
    gtk_widget_destroy (dialog);
    return;
  }

  if (res == GTK_RESPONSE_YES)
  {

     gchar *name = NULL, *exec = NULL, *icon_name = NULL;
     g_object_get (dialog, 
                   "shortcut-name", &name,
                   "shortcut-exec", &exec,
                   "shortcut-icon", &icon_name,
                   NULL);
     launcher_menu_edit_application (launcher_menu_get_default (),
                                    icon->priv->app, 
                                    NULL, name, exec, icon_name);
  }

  gtk_widget_destroy (dialog);
}

static void
popup_edit_dialog (LauncherIcon *icon)
{
  LauncherIconPrivate *priv;
  LauncherMenuApplication *app;
  GtkWidget *dialog;
  
  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  app = priv->app;

  launcher_app_lock (launcher_app_get_default (), TRUE);

  dialog = launcher_shortcut_editor_new ();
  g_object_set (dialog, 
                "title", _("Edit Shortcut"),
                "shortcut-name", launcher_menu_application_get_name (app),
                "shortcut-exec", launcher_menu_application_get_exec (app),
                "shortcut-icon", launcher_menu_application_get_icon_name (app),
                NULL);
  gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
                          "gtk-cancel", GTK_RESPONSE_CANCEL,
                          GTK_STOCK_APPLY, GTK_RESPONSE_YES,
                          NULL);
  g_signal_connect (dialog, "response",
                    G_CALLBACK (on_edit_reponse), icon);
                                   
  launcher_util_present_window (GTK_WINDOW (dialog));
}

static gboolean
on_edit_clicked (ClutterActor       *actor, 
                 ClutterButtonEvent *event, 
                 LauncherIcon       *icon)
{
  LauncherIconPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  popup_edit_dialog (icon);

  rotate (icon);

  return TRUE;
}

static void
on_delete_reponse (GtkWidget *dialog, gint res, LauncherIcon *icon)
{
  if (!LAUNCHER_IS_ICON (icon))
  {
    gtk_widget_destroy (dialog);
    return;
  }

  if (res == GTK_RESPONSE_YES)
  {
    launcher_util_remove_favorite (icon->priv->app);
  }

  gtk_widget_destroy (dialog);
}

static void
popup_delete_dialog (LauncherIcon *icon)
{
  LauncherIconPrivate *priv = icon->priv;
  GtkWidget *dialog;

  launcher_app_lock (launcher_app_get_default (), TRUE);

  dialog = gtk_message_dialog_new (NULL, 0,
                                   GTK_MESSAGE_QUESTION,
                                   GTK_BUTTONS_NONE,
                                   "Are you sure you want to remove the "
                                   "%s shortcut?",
                                   launcher_menu_application_get_name (priv->app)
                                   );
  gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
                          "gtk-cancel", GTK_RESPONSE_CANCEL,
                          "gtk-delete", GTK_RESPONSE_YES,
                          NULL);
  g_object_set (dialog, 
                "title", _("Remove Shortcut"),
                "accept-focus", TRUE,
                "focus-on-map", TRUE,
                "window-position", GTK_WIN_POS_CENTER_ALWAYS,
                NULL);
  g_signal_connect (dialog, "response",
                    G_CALLBACK (on_delete_reponse), icon);
                                   
  launcher_util_present_window (GTK_WINDOW (dialog));

}

static gboolean
on_delete_clicked (ClutterActor       *actor, 
                   ClutterButtonEvent *event, 
                   LauncherIcon       *icon)
{
  LauncherIconPrivate *priv;

  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  popup_delete_dialog (icon);

  rotate (icon);

  return TRUE;
}

static ClutterActor *
make_item (const gchar *name, const gchar *icon_name, gint width, gint height)
{
  LauncherConfig *cfg = launcher_config_get_default ();
  ClutterActor *bg, *icon, *group, *label;

  bg = clutter_group_new ();
  clutter_actor_set_size (bg, width, height);
  clutter_actor_show (bg);
  
  icon = launcher_util_texture_new_from_named_icon (icon_name);
  clutter_container_add_actor (CLUTTER_CONTAINER (bg), icon);
  clutter_actor_set_size (icon, height * 0.80, height * 0.80);
  clutter_actor_set_anchor_point_from_gravity (icon, CLUTTER_GRAVITY_WEST);
  clutter_actor_set_position (icon, height *0.15, height/2);
  clutter_actor_show (icon);

  group = clutter_group_new ();
  clutter_container_add_actor (CLUTTER_CONTAINER (bg), group);
  clutter_actor_set_size (group, height, width - CAW (icon) - height);
  clutter_actor_set_anchor_point_from_gravity (group, CLUTTER_GRAVITY_WEST);
  clutter_actor_set_position (group, CAW (icon) + height * 0.30, height/2);
  clutter_actor_show (group);
  
  label = clutter_label_new_full (cfg->font_name, name, &cfg->icon_font_color);
  clutter_container_add_actor (CLUTTER_CONTAINER (group), label);
  clutter_actor_set_width (label, width - CAW (icon));
  clutter_label_set_ellipsize (CLUTTER_LABEL (label), PANGO_ELLIPSIZE_END);
  clutter_actor_set_position (label, 0, - (CAH (label)/2));
  clutter_actor_show (label);


  return bg;
}

static gboolean
on_toggle_clicked (ClutterActor *close,
                   ClutterButtonEvent *event,
                   LauncherIcon *icon)
{
  LauncherIconPrivate *priv;
  //LauncherConfig *cfg = launcher_config_get_default ();
  
  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), FALSE);
  priv = icon->priv;

  if (clutter_timeline_is_playing (priv->slow_time))
    return TRUE;

  if (!priv->flipped)
  {
    ClutterActor *bg, *item;

    priv->options = clutter_group_new ();
    clutter_container_add_actor (CLUTTER_CONTAINER (icon), priv->options);
    clutter_actor_set_size (priv->options, CAW (icon), CAH (icon));
    clutter_actor_set_position (priv->options, 0, 0);
    clutter_actor_set_rotation (priv->options, CLUTTER_Y_AXIS, 
                                180, CAW (priv->options)/2, 0, 0);
    clutter_actor_set_opacity (priv->options, 0);
    clutter_actor_show (priv->options);

    bg = item = launcher_util_texture_new_from_file (PKGDATADIR"/options.svg");
    clutter_actor_set_size (bg, CAW (icon)*0.98, CAH (icon) * 0.70);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->options), bg);
    clutter_actor_set_anchor_point_from_gravity (bg, CLUTTER_GRAVITY_NORTH);
    clutter_actor_set_position (bg, CAW (icon)/2, CAH (icon) *0.25);
    clutter_actor_show (bg);

    item = clutter_clone_texture_new (CLUTTER_TEXTURE (priv->texture));
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->options), item);
    clutter_actor_set_size (item, CAH (bg)/3.4, CAH (bg)/3.4);
    clutter_actor_set_position (item, CAW (item) * 0.1, CAW (item) *0.1);
    clutter_actor_show (item);

    item = make_item (_("Launch"), "applications-system", 
                      CAW (bg), CAH (bg)/3);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->options), item);
    clutter_actor_set_position (item, 
                                CAW (icon) * 0.01,
                                clutter_actor_get_y (bg));
    clutter_actor_show (item);
    clutter_actor_set_reactive (item, TRUE);
    g_signal_connect (item, "button-release-event",
                      G_CALLBACK (on_launch_clicked), icon);
    g_signal_connect (item, "enter-event",
                      G_CALLBACK (on_enter), icon);
  
    item = make_item (_("Edit"), "gtk-properties",
                      CAW (bg), CAH (bg)/3);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->options), item);
    clutter_actor_set_position (item, 
                                CAW (icon) * 0.01,
                                clutter_actor_get_y (bg) + CAH (bg)/3);
    clutter_actor_show (item);
    clutter_actor_set_reactive (item, TRUE);
    g_signal_connect (item, "button-release-event",
                      G_CALLBACK (on_edit_clicked), icon);
    g_signal_connect (item, "enter-event",
                      G_CALLBACK (on_enter), icon);

    item = make_item (_("Delete"), "gtk-delete",
                      CAW (bg), CAH (bg)/3);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->options), item);
    clutter_actor_set_position (item, 
                                CAW (icon) * 0.01,
                                clutter_actor_get_y (bg) + ((CAH (bg)/3)*2));
    clutter_actor_show (item);
    clutter_actor_set_reactive (item, TRUE);
    g_signal_connect (item, "button-release-event",
                      G_CALLBACK (on_delete_clicked), icon);
    g_signal_connect (item, "enter-event",
                      G_CALLBACK (on_enter), icon);
   }

  rotate (icon);

  g_signal_emit (icon, _icon_signals[FLIPPED], 0);

  return TRUE;
}

/*
 * Make sure the label doesn't span more the two lines...the nasty way!
 * Blame http://bugzilla.openedhand.com/show_bug.cgi?id=918
 */
static gboolean
clip_label (ClutterLabel *label, const gchar *text, gint len, gint real_width)
{
  PangoLayout *layout = clutter_label_get_layout (label);
  gint lines;
  ClutterUnit awidth, lwidth;
  gint max_lines = 2;
  gboolean ellipsize = FALSE;
  gboolean touched = FALSE; 

  gchar buf[len];

  memcpy (buf, text, len+1);

  lines = pango_layout_get_line_count (layout);  
  while (lines > max_lines)
  {
    *g_utf8_prev_char (&buf[len]) = '\0';
    clutter_label_set_text (label, buf);
    len -= 5;
    layout = clutter_label_get_layout (label);
    lines = pango_layout_get_line_count (layout);
    touched = TRUE;
  }

  awidth = CLUTTER_UNITS_FROM_DEVICE (real_width);
  lwidth = CLUTTER_UNITS_FROM_PANGO_UNIT (pango_layout_get_width (layout));
  while (lwidth > awidth && len > 12)
  {
    *g_utf8_prev_char (&buf[len]) = '\0';
    clutter_label_set_text (label, buf);
    len -= 1;
    layout = clutter_label_get_layout (label);
    lwidth =  CLUTTER_UNITS_FROM_PANGO_UNIT (pango_layout_get_width (layout));
    ellipsize = TRUE;
  }

  if (ellipsize)
  {
    buf[len] = '.';
    buf[len-1] = '.';
    buf[len-2] = '.';
    clutter_label_set_text (label, buf);
  }

  if (ellipsize || touched)
    return TRUE;
  else
    return FALSE;
}

void 
launcher_icon_set_application (LauncherIcon            *icon, 
                               LauncherMenuApplication *application)
{
  LauncherIconPrivate *priv;
  LauncherConfig *cfg = launcher_config_get_default ();
  GdkPixbuf *pixbuf;
  ClutterActor *bg, *texture;
  ClutterActor *label, *new;;
  ClutterColor color = { 0xff, 0xff, 0xff, 0xff };
  ClutterColor orange = { 0x00, 0x00, 0x00, 0xff };
  gint width = cfg->icon_width;
  gint height = cfg->icon_height;
  gint view_padding = cfg->iconview_padding;
  gint icon_size = cfg->is_poulsbo ? height * 0.4 : height * 0.6;
 
  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  priv->app = application;

  /* bg */
  bg = priv->bg = clutter_rectangle_new_with_color (&color);
  clutter_container_add_actor (CLUTTER_CONTAINER (icon), bg);
  clutter_actor_set_size (bg, width - 1,height -1);
  clutter_actor_set_position (bg, 0, 0);
  clutter_actor_set_reactive (bg, TRUE);
  clutter_actor_set_opacity (bg, 50);
  clutter_actor_show (bg);
      
  /* Icon */
  pixbuf = launcher_menu_application_get_icon (application);
  texture = priv->texture = clutter_texture_new_from_pixbuf (pixbuf);
  clutter_actor_set_size (texture, icon_size, icon_size);
  clutter_actor_set_anchor_point_from_gravity (texture, CLUTTER_GRAVITY_NORTH);
  clutter_actor_set_position (texture, 
                              width/2,
                              (cfg->is_poulsbo) ? height *0.3 :height *0.1);
  clutter_container_add_actor (CLUTTER_CONTAINER (icon), texture);
  clutter_actor_show (texture);

  /* Text label */
  label = priv->label = clutter_label_new ();
  clutter_label_set_font_name (CLUTTER_LABEL (label), 
                               cfg->font_name);
  clutter_label_set_color (CLUTTER_LABEL (label), &cfg->icon_font_color);
  clutter_label_set_line_wrap (CLUTTER_LABEL (label), FALSE);
  clutter_label_set_line_wrap_mode (CLUTTER_LABEL (label), 
                                    PANGO_WRAP_WORD_CHAR);
  clutter_label_set_alignment (CLUTTER_LABEL (label), PANGO_ALIGN_CENTER);
  clutter_label_set_text (CLUTTER_LABEL (label),
                          launcher_menu_application_get_name (application));
  clutter_actor_set_width (label, width);
  clutter_container_add_actor (CLUTTER_CONTAINER (icon), label);
  clutter_actor_set_position (label, 
                              (width-CAW(label))/2,
                              clutter_actor_get_y (texture) + CAH(texture)
          + (cfg->is_poulsbo ? (height *0.10) : (height * 0.05)));
  clutter_actor_show (label);
  priv->show_tooltip = clip_label (CLUTTER_LABEL (label),
                                   application->name,
                                   strlen (application->name),
                                   width + view_padding);
  if (priv->show_tooltip)
  {
    ClutterActor *label, *rect;
    
    priv->tooltip = clutter_group_new ();
    clutter_actor_set_opacity (priv->tooltip,  0);
    clutter_container_add_actor (CLUTTER_CONTAINER (clutter_stage_get_default()), 
                                 priv->tooltip);
    clutter_actor_show (priv->tooltip);

    rect = clutter_rectangle_new_with_color (&orange);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->tooltip), rect);
    clutter_actor_set_opacity (rect, 150);
    clutter_actor_show (rect);

    label = clutter_label_new_full (cfg->font_name, application->name, &color);
    clutter_label_set_line_wrap (CLUTTER_LABEL (label), TRUE);
    clutter_label_set_line_wrap_mode (CLUTTER_LABEL (label), 
                                      PANGO_WRAP_WORD);
    clutter_label_set_alignment (CLUTTER_LABEL (label), PANGO_ALIGN_CENTER);
    clutter_label_set_text (CLUTTER_LABEL (label), application->name);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->tooltip), label);
    clutter_actor_show (label);

    clutter_actor_set_size (rect, CAW(label), CAH (label));
    clutter_actor_set_anchor_point_from_gravity (priv->tooltip,   
                                                 CLUTTER_GRAVITY_NORTH);
    clutter_actor_set_position (priv->tooltip, width/2, height);
  } 

   /* 'New' indicator */
  if (launcher_menu_application_is_new (application))
  {
    new = launcher_util_texture_new_from_named_icon ("emblem-new");
    clutter_container_add_actor (CLUTTER_CONTAINER (icon), new);
    clutter_actor_set_size (new, 
                            2*view_padding,
                            2*view_padding);
    clutter_actor_set_position (new, 2*view_padding, 0);
    clutter_actor_show (new);
  }

  priv->toggle = launcher_util_texture_new_from_file (PKGDATADIR"/info.svg");
  clutter_container_add_actor (CLUTTER_CONTAINER (icon), priv->toggle);
  clutter_actor_set_size (priv->toggle, height*0.18, height*0.18);
  clutter_actor_set_position (priv->toggle, 
                              width - (CAW (priv->toggle))- 3, 3);
  clutter_actor_set_opacity (priv->toggle, 0);
  clutter_actor_show (priv->toggle);

  clutter_actor_set_reactive (priv->toggle, TRUE);
  g_signal_connect (priv->toggle, "button-release-event",
                    G_CALLBACK (on_toggle_clicked), icon);
  g_signal_connect (priv->toggle, "enter-event",
                    G_CALLBACK (on_enter), icon);
  g_signal_connect (priv->toggle, "leave-event",
                    G_CALLBACK (on_leave), icon);
  
  g_signal_connect (bg, "button-release-event",
                    G_CALLBACK (on_app_clicked), icon);
  g_signal_connect (bg, "enter-event",
                    G_CALLBACK (on_enter), icon);
  g_signal_connect (bg, "leave-event",
                    G_CALLBACK (on_leave), icon);
  g_signal_connect (bg, "motion-event", 
                    G_CALLBACK (on_motion), icon);
}

static void
on_show (ClutterActor *icon)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = LAUNCHER_ICON (icon)->priv;

  CLUTTER_ACTOR_CLASS (launcher_icon_parent_class)->show (icon);
 
  //clutter_actor_set_opacity (icon, 0);
  //clutter_effect_fade (priv->temp, icon, 255, NULL, NULL);
}

LauncherMenuApplication * 
launcher_icon_get_application (LauncherIcon *icon)
{
  g_return_val_if_fail (LAUNCHER_IS_ICON (icon), NULL);

  return icon->priv->app;
}

void       
launcher_icon_set_flipped (LauncherIcon *icon, gboolean flipped)
{
  g_return_if_fail (LAUNCHER_IS_ICON (icon));

  if (icon->priv->flipped != FLIPPED)
    rotate (icon);
}

void 
launcher_icon_set_focus (LauncherIcon *icon, gboolean has_focus)
{
  LauncherIconPrivate *priv;
    
  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = icon->priv;

  if (has_focus)
  {
    clutter_effect_fade (priv->temp, priv->bg, 130, NULL, NULL);
  }
  else
  {
    clutter_effect_fade (priv->temp, priv->bg, 50, NULL, NULL);
  }
}

void 
launcher_icon_launch (LauncherIcon *icon)
{
  LauncherConfig *cfg = launcher_config_get_default ();

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  
  launcher_startup_launch_app (launcher_startup_get_default (),
                               icon->priv->app);

  clutter_effect_rotate (icon->priv->temp,
                         CLUTTER_ACTOR (icon),
                         CLUTTER_Y_AXIS,
                         360.0,
                         cfg->icon_width/2, 0, 0, 
                         CLUTTER_ROTATE_CW,
                         NULL, NULL);
}



/* GObject stuff */
static void
launcher_icon_finalize (GObject *icon)
{
  LauncherIconPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_ICON (icon));
  priv = LAUNCHER_ICON_GET_PRIVATE (icon);

  if (priv->tooltip)
    clutter_actor_destroy (priv->tooltip);

  G_OBJECT_CLASS (launcher_icon_parent_class)->finalize (icon);
}

static void
launcher_icon_class_init (LauncherIconClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass   *act_class = CLUTTER_ACTOR_CLASS (klass);

  obj_class->finalize = launcher_icon_finalize;
  act_class->show = on_show;

	_icon_signals[FLIPPED] =
		g_signal_new ("flipped",
			      G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherIconClass, flipped),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);

  g_type_class_add_private (obj_class, sizeof (LauncherIconPrivate));
}

      
static void
launcher_icon_init (LauncherIcon *icon)
{
  LauncherIconPrivate *priv;
  ClutterAlpha *alpha;
  
  priv = icon->priv = LAUNCHER_ICON_GET_PRIVATE (icon);

  priv->show_tooltip = FALSE;
  priv->flipped = FALSE;

  if (!CLUTTER_IS_ACTOR (bg_texture))
  {
    GdkPixbuf *temp;

    temp = gdk_pixbuf_new_from_file (PKGDATADIR"/icon.svg", NULL);
    bg_texture = clutter_texture_new_from_pixbuf (temp);
    clutter_container_add_actor (CLUTTER_CONTAINER (clutter_stage_get_default()), 
                                 bg_texture);
    clutter_actor_realize (bg_texture);
    g_object_unref (temp);
  }

  priv->slow_time = launcher_util_timeline_new_slow ();
  alpha = clutter_alpha_new_full (priv->slow_time,
                                  clutter_sine_inc_func,
                                  NULL, NULL);
  priv->behave = launcher_behave_new (alpha, 
                                      (LauncherBehaveAlphaFunc)rotate_func,
                                      (gpointer)icon);

  g_signal_connect (priv->slow_time, "completed",
                    G_CALLBACK (on_rotation_completed), icon);


  priv->time = launcher_util_timeline_new_medium ();
  priv->temp = clutter_effect_template_new (priv->time, 
                                            clutter_sine_inc_func);

  priv->fast_time = launcher_util_timeline_new_fast ();
  priv->fast_temp = clutter_effect_template_new (priv->fast_time,
                                                 clutter_sine_inc_func);
}

ClutterActor *
launcher_icon_new (void)

{
  ClutterActor *icon = NULL;

  icon = g_object_new (LAUNCHER_TYPE_ICON, 
                           NULL);
  return icon;
}
