/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib/gi18n.h>
#include <libgnomeui/libgnomeui.h>

#include "launcher-link-editor.h"
#include "launcher-app.h"
#include "launcher-defines.h"
#include "launcher-menu.h"
#include "launcher-util.h"

G_DEFINE_TYPE (LauncherLinkEditor, launcher_link_editor, GTK_TYPE_DIALOG);

#define LAUNCHER_LINK_EDITOR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  LAUNCHER_TYPE_LINK_EDITOR, \
  LauncherLinkEditorPrivate))

#define WEB_ICON_NAME "applications-internet"
#define FOLDER_ICON_NAME "gnome-fs-directory"
#define APP_ICON_NAME "gnome-panel-launcher"

#define GLOBEDIR PKGDATADIR"/globes"
#define GLOBE_SIZE 64
#define FAV_SIZE 16
#define FAV_X 4
#define FAV_Y 4


struct _LauncherLinkEditorPrivate
{
  LauncherMenuCategory *cat;

  GdkPixbuf *favicon;
  gboolean   composite;

  GtkWidget *combo;
  GtkWidget *name_entry;
  GtkWidget *loc_entry;
  GtkWidget *icon;
  GtkWidget *image;
  GtkWidget *browse;

  gboolean icon_change_lock;
};

enum
{
	PROP_0,
  PROP_TYPE,
	PROP_NAME,
  PROP_EXEC,
  PROP_ICON
};

static void
on_entry_activate (GtkEntry *entry, GtkDialog *self)
{
    gtk_dialog_response (GTK_DIALOG (self), GTK_RESPONSE_YES);
}

static void
set_composite (LauncherLinkEditor *self, const gchar *globe_file)
{
  LauncherLinkEditorPrivate *priv = self->priv;
  GdkPixbuf *globe;
 
  globe = gdk_pixbuf_new_from_file (globe_file, NULL);

  if (GDK_IS_PIXBUF (globe))
  {
    gdk_pixbuf_composite (priv->favicon,
                          globe, 
                          4, 4,
                          FAV_SIZE, FAV_SIZE, 
                          FAV_X, FAV_Y, 1.0, 1.0, 
                          GDK_INTERP_BILINEAR,
                          255);

    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->image), globe);
  }
  else
    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->image), priv->favicon);

  priv->composite = TRUE;

  g_object_unref (globe);
}

static void
set_pixbuf (LauncherLinkEditor *self, const gchar *globe_file)
{
  LauncherLinkEditorPrivate *priv = self->priv;
  GdkPixbuf *globe;
 
  globe = gdk_pixbuf_new_from_file (globe_file, NULL);

  if (GDK_IS_PIXBUF (globe))
  {
    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->image), globe);
  }
  else
    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->image), priv->favicon);

  priv->composite = FALSE;

  g_object_unref (globe);
}
/*
 * Set the initial globe icon
 */
static void
set_icon_path (LauncherLinkEditor *self, const gchar *icon_name)
{
  LauncherLinkEditorPrivate *priv = self->priv;
  GdkPixbuf *globe, *favicon;

  priv->composite = FALSE;
 
  globe = gdk_pixbuf_new_from_file (GLOBEDIR"/globe_aqua.png", NULL);
  favicon = priv->favicon = gdk_pixbuf_new_from_file_at_size (icon_name,  
                                              FAV_SIZE, FAV_SIZE, NULL);

  if (GDK_IS_PIXBUF (globe) && GDK_IS_PIXBUF (favicon))
  {
    gdk_pixbuf_composite (favicon,
                          globe, 
                          4, 4,
                          FAV_SIZE, FAV_SIZE, 
                          FAV_X, FAV_Y, 1.0, 1.0, 
                          GDK_INTERP_BILINEAR,
                          255);

    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->image), globe);

    priv->composite = TRUE;
  }
  else
    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->image), favicon);

  g_object_unref (globe);
}

static void
on_browse_response (GtkWidget *dialog, gint res, LauncherLinkEditor *self)
{
  if (res == GTK_RESPONSE_YES)
  { 
    gchar *filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));

    if (filename)
    {
      if (strstr (filename, GLOBEDIR))
      {
        set_composite (self, filename);
      }
      else
      {
        set_pixbuf (self, filename);
      }
    }
  }
  gtk_widget_destroy (dialog);    
}

static void
update_preview (GtkFileChooser *chooser, GtkImage *preview)
{
  gchar *filename = NULL;
  GdkPixbuf *pixbuf = NULL;

  filename = gtk_file_chooser_get_filename (chooser);
  
  if (filename)
    pixbuf = gdk_pixbuf_new_from_file_at_size (filename, 
                                             GLOBE_SIZE, GLOBE_SIZE, NULL);

  if (pixbuf)
  {
    gtk_image_set_from_pixbuf (preview, pixbuf);
    g_object_unref (pixbuf);
  }
  
  g_free (filename);
}

static void
on_icon_clicked (GtkWidget *widget, LauncherLinkEditor *self)
{
  GtkWidget *dialog;
  GtkFileFilter *filter;
  GtkWidget *preview;

  filter = gtk_file_filter_new ();
  gtk_file_filter_set_name (filter, _("Images"));
  gtk_file_filter_add_pixbuf_formats (filter);

  preview = gtk_image_new ();
  gtk_widget_set_size_request (preview, GLOBE_SIZE, GLOBE_SIZE);
  
  dialog = gtk_file_chooser_dialog_new (_("Choose link image"),
                                        GTK_WINDOW (self),
                                        GTK_FILE_CHOOSER_ACTION_OPEN,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_OPEN, GTK_RESPONSE_YES,
                                        NULL);
  gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (dialog), FALSE);
  gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog), GLOBEDIR);
  gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog), filter);
  gtk_file_chooser_set_preview_widget (GTK_FILE_CHOOSER (dialog), preview);
  
  g_signal_connect (dialog, "response", 
                    G_CALLBACK (on_browse_response), self);
  g_signal_connect (dialog, "update-preview", 
                    G_CALLBACK (update_preview), preview);

  gtk_widget_show (dialog);
}

void
launcher_link_editor_set_name (LauncherLinkEditor *self, 
                                   const gchar            *name)
{
  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (self));

  gtk_entry_set_text (GTK_ENTRY (self->priv->name_entry), name);
}

void
launcher_link_editor_set_exec (LauncherLinkEditor *self, 
                                   const gchar            *exec)
{
  LauncherLinkEditorPrivate *priv;
  static gint open_len = 0;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (self));
  priv = self->priv;

  if (!open_len)
    open_len = strlen (XDG_OPEN) + 1;

  gtk_entry_set_text (GTK_ENTRY (priv->loc_entry), exec);
}

gchar *
launcher_link_editor_get_exec (LauncherLinkEditor *self)
{
  LauncherLinkEditorPrivate *priv = self->priv;
  gchar *exec = NULL;

  exec = g_strdup_printf ("%s \"%s\"", XDG_OPEN,
                          gtk_entry_get_text (GTK_ENTRY (priv->loc_entry)));


  return exec;
}

void
launcher_link_editor_set_icon (LauncherLinkEditor *self, 
                               const gchar        *icon_name)
{
  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (self));

  set_icon_path (self, icon_name);
  self->priv->icon_change_lock = TRUE;
}

gchar *
launcher_link_editor_get_icon (LauncherLinkEditor *self)
{
#define ICON_DIR ".config/dell-launcher/icons"
 LauncherLinkEditorPrivate *priv = self->priv;
 gchar *filename, *leaf;

 leaf = g_strdup_printf ("favicon-%ld.png", time (NULL));
 filename = g_build_filename (g_get_home_dir (), ICON_DIR, leaf, NULL);

 gdk_pixbuf_save (gtk_image_get_pixbuf (GTK_IMAGE (priv->image)), 
                  filename, "png", NULL, NULL);
 g_print (filename);

 return filename;
}

/* GObject stuff */
static void
launcher_link_editor_finalize (GObject *link_editor)
{
  LauncherLinkEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (link_editor));
  priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (link_editor);

  G_OBJECT_CLASS (launcher_link_editor_parent_class)->finalize (link_editor);
}

static void
set_property (GObject      *object, 
              guint         prop_id,
              const GValue *value, 
              GParamSpec   *pspec)
{
  LauncherLinkEditor *self = LAUNCHER_LINK_EDITOR (object);
  LauncherLinkEditorPrivate *priv;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (object));
  priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_TYPE:
      gtk_combo_box_set_active (GTK_COMBO_BOX (priv->combo),  
                                g_value_get_int (value));
      break;
    case PROP_NAME:
      launcher_link_editor_set_name (self, g_value_get_string (value));
      break;
    case PROP_EXEC:
      launcher_link_editor_set_exec (self, g_value_get_string (value));
      break;
    case PROP_ICON:
      launcher_link_editor_set_icon (self, g_value_get_string (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
get_property (GObject      *object, 
              guint         prop_id,
              GValue       *value, 
              GParamSpec   *pspec)
{
  LauncherLinkEditor *self = LAUNCHER_LINK_EDITOR (object);
  LauncherLinkEditorPrivate *priv;
  gchar *exec, *icon_name;

  g_return_if_fail (LAUNCHER_IS_LINK_EDITOR (object));
  priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (object);

  switch (prop_id) 
  {
    case PROP_TYPE:
      g_value_set_int (value,
                       gtk_combo_box_get_active (GTK_COMBO_BOX (priv->combo)));

      break;
    case PROP_NAME:
      g_value_set_string (value, 
                          gtk_entry_get_text (GTK_ENTRY (priv->name_entry)));
      break;
    case PROP_EXEC:
      exec = launcher_link_editor_get_exec (self);
      g_value_set_string (value, exec);
      g_free (exec);
      break;
    case PROP_ICON:
      icon_name = launcher_link_editor_get_icon (self);
      g_value_set_string (value, icon_name);
      g_free (icon_name);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
launcher_link_editor_class_init (LauncherLinkEditorClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  
  obj_class->finalize = launcher_link_editor_finalize;
  obj_class->get_property = get_property;
  obj_class->set_property = set_property;


  /* Class properties */
  g_object_class_install_property (obj_class,
                                   PROP_TYPE,
                                   g_param_spec_int ("link-category",
                                   "Link Type",
                                   "Type of link",
                                   0, 100, 0, 
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_NAME,
                                   g_param_spec_string ("link-name",
                                   "Link Name",
                                   "Name of link",
                                   NULL,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_EXEC,
                                   g_param_spec_string ("link-exec",
                                   "Link Exec",
                                   "Exec of link",
                                   NULL,
                                   G_PARAM_READWRITE));

  g_object_class_install_property (obj_class,
                                   PROP_ICON,
                                   g_param_spec_string ("link-icon",
                                   "Link Icon",
                                   "Icon of link",
                                   NULL,
                                   G_PARAM_READWRITE));


  g_type_class_add_private (obj_class, sizeof (LauncherLinkEditorPrivate));
}

      
static void
launcher_link_editor_init (LauncherLinkEditor *self)
{
  LauncherLinkEditorPrivate *priv;
  GtkWidget *window = GTK_WIDGET (self);
  GtkWidget *align, *vbox, *hbox;
  GtkWidget *label, *combo, *entry, *icon;
  GtkSizeGroup *group;
  LauncherMenu *menu = launcher_menu_get_default ();
  GList *cats = launcher_menu_get_categories (menu), *c;
  gint active_cat = 0, i = 0;
  
  priv = self->priv = LAUNCHER_LINK_EDITOR_GET_PRIVATE (self);

  launcher_app_lock (launcher_app_get_default (), TRUE);

  priv->icon_change_lock = FALSE;

  group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  align = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (align), 12, 0, 12, 12);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), align, 
                      0, TRUE, TRUE);
  
  g_object_set (GTK_DIALOG(window)->vbox,  "spacing", 8, NULL);  
 
  vbox = gtk_vbox_new (FALSE, 8);
  gtk_container_add (GTK_CONTAINER (align), vbox); 

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, TRUE, TRUE);
  
  label = gtk_label_new (_("Category:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  combo = priv->combo = gtk_combo_box_new_text ();
  for (c = cats; c; c= c->next)
  {
    LauncherMenuCategory *cat = c->data;
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo),
                               cat->name);
    if (cat->name && strstr (cat->name, _("Favorite")))
      active_cat = i;
    i++;
  } 
  gtk_combo_box_set_active (GTK_COMBO_BOX (combo), active_cat);
  gtk_widget_set_size_request (combo, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox), combo, 0, TRUE, TRUE);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, TRUE, TRUE);
  
  label = gtk_label_new (_("Name:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  entry = priv->name_entry = gtk_entry_new ();
  gtk_widget_set_size_request (entry, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox), entry, 0, TRUE, TRUE);
  gtk_widget_grab_focus (entry);
  g_signal_connect (entry, "activate", G_CALLBACK (on_entry_activate), self);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, TRUE, TRUE);
  
  label = gtk_label_new (_("Location:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  entry = priv->loc_entry = gtk_entry_new ();
  gtk_widget_set_size_request (entry, 300, -1);
  gtk_box_pack_start (GTK_BOX (hbox), entry, 0, TRUE, TRUE);
  g_signal_connect (entry, "activate", G_CALLBACK (on_entry_activate), self);

  hbox = gtk_hbox_new (FALSE, 8);
  gtk_box_pack_start (GTK_BOX (vbox), hbox, 0, TRUE, TRUE);
  
  label = gtk_label_new (_("Icon:"));
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (hbox), label, 0, FALSE, FALSE);
  gtk_size_group_add_widget (group, label);

  icon = priv->icon = gtk_button_new ();
  priv->image = gtk_image_new ();
  gtk_container_add (GTK_CONTAINER (icon), priv->image);
  gtk_box_pack_start (GTK_BOX (hbox), icon, 0, TRUE, TRUE);
  g_signal_connect (icon, "clicked", G_CALLBACK (on_icon_clicked), self);
  
  gtk_widget_show_all (align);
}

GtkWidget *
launcher_link_editor_new (void)

{
  GtkWidget *link_editor = NULL;

  link_editor = g_object_new (LAUNCHER_TYPE_LINK_EDITOR, 
                              "window-position", GTK_WIN_POS_CENTER_ALWAYS,
                              "type", GTK_WINDOW_TOPLEVEL,
                              "decorated", TRUE,
                              NULL);

  return link_editor;
}
