/*
 * Copyright (C) 2007 Intel
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <njp@o-hand.com>
 *             Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <glib.h>
#include <glib/gi18n.h>

#include <stdio.h>
#include <string.h>

#define GMENU_I_KNOW_THIS_IS_UNSTABLE 1
#include <gmenu-tree.h>

#include <libwnck/window.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include <clutter/clutter.h>
#include <libgnome/gnome-desktop-item.h>

#include "launcher-defines.h"
#include "launcher-menu.h"
#include "launcher-util.h"

G_DEFINE_TYPE (LauncherMenu, launcher_menu, G_TYPE_OBJECT)

#define LAUNCHER_MENU_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        LAUNCHER_TYPE_MENU, LauncherMenuPrivate))

#define LAUNCHER_PATH "/apps/dell-launcher"
#define LAUNCHER_CAT_ORDER LAUNCHER_PATH "/category_order"
#define LAUNCHER_APP_ORDER LAUNCHER_PATH "/application_order"

struct _LauncherMenuPrivate
{
  GList      *categories;
  GList      *menu_apps;
  GMenuTree *app_tree;
  GMenuTree *sys_tree;

  guint      tag;

  gboolean   refresh;
  GList     *old_cats;
  GList     *old_apps;
};


enum 
{
  MENU_CHANGED,

  LAST_SIGNAL
};

static guint _menu_signals[LAST_SIGNAL] = { 0 };

/* Forwards */
static void    tree_changed      (GMenuTree            *tree, 
                                  LauncherMenu         *menu);
static void    save_categories   (LauncherMenu         *menu);
static void    save_applications (LauncherMenu         *menu, 
                                  LauncherMenuCategory *cat);

static gchar * _get_category_id  (const gchar          *path);

/* Public functions */
void
launcher_menu_add_application  (LauncherMenu         *menu,
                                LauncherMenuCategory *cat,
                                const gchar          *name,
                                const gchar          *exec,
                                const gchar          *icon)
{
  LauncherMenuPrivate *priv;
  gchar *filename, *path, *uri;

  g_return_if_fail (menu);
  g_return_if_fail (cat);
  priv = menu->priv;

  filename = g_strdup_printf ("%s-favorite-%ld.desktop", name, time (NULL));
  _normalize (filename);
  path = g_build_filename (g_get_home_dir (),
                           DESKTOP_FILE_PATH,
                           filename,
                           NULL);
  uri = g_filename_to_uri (path, NULL, NULL);

  /* Save the favourite */
  launcher_util_new_shortcut (uri, name, cat->id, exec, icon);

  g_free (filename);
  g_free (path);
  g_free (uri);
}


void
launcher_menu_edit_application  (LauncherMenu            *menu,
                                 LauncherMenuApplication *app,
                                 LauncherMenuCategory    *cat,
                                 const gchar             *name,
                                 const gchar             *exec,
                                 const gchar             *icon)
{
  LauncherMenuPrivate *priv;
  gchar *uri;  

  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  g_return_if_fail (app);
  priv = menu->priv;

  uri = g_filename_to_uri (app->path, NULL, NULL);

  if (cat)
  {
    LauncherMenuCategory *old_cat = app->category;
    old_cat->applications = g_list_remove (old_cat->applications, app);
    app->category = cat;
    cat->applications = g_list_append (cat->applications, app);
  }
  if (name)
  {
    g_free (app->name);
    app->name = g_strdup (name);
  }
  if (exec)
  {
    g_free (app->exec);
    app->exec = g_strdup (exec);
  }
  if (icon)
  {
    g_free (app->icon);
    app->icon = g_strdup (icon);
    g_object_unref (app->pixbuf);
    app->pixbuf = NULL;
  }

  launcher_util_edit_shortcut (uri, 
                               app->category->id, 
                               app->name, 
                               app->exec, 
                               app->icon);

  g_free (uri);
}

void
launcher_menu_add_category  (LauncherMenu         *menu,
                             const gchar          *name,
                             const gchar          *icon)
{
  LauncherMenuPrivate *priv;
  GList *c;
  gchar *filename, *path, *uri, *id;

  g_return_if_fail (menu);
  g_return_if_fail (name);
  g_return_if_fail (icon);
  priv = menu->priv;

  /* Check if a category with the same name already exists */
  for (c = priv->categories; c; c = c->next)
  {
    LauncherMenuCategory *cat = c->data;

    if (cat && cat->name && g_strcmp0 (cat->name, name) == 0)
    {
      GtkWidget *dialog;

      dialog = gtk_message_dialog_new (NULL, 0, 
                                       GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE,
                                       _("The launcher cannot create two "
                                         "categories with the same name"));
      gtk_window_set_title (GTK_WINDOW (dialog), 
                            _("Unable to create new category"));
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return;
    }
  }

  filename = g_strdup_printf ("%s-category-%ld.directory", name, time (NULL));
  _normalize (filename);
  path = g_build_filename (g_get_home_dir (),
                           DIRECTORY_FILE_PATH,
                           filename,
                           NULL);
  uri = g_filename_to_uri (path, NULL, NULL);
  id = _get_category_id (path);

  /* Save the category */
  launcher_util_new_category (uri, id, name, icon);

  g_free (filename);
  g_free (path);
  g_free (uri);
  g_free (id);
}


void
launcher_menu_edit_category  (LauncherMenu            *menu,
                              LauncherMenuCategory    *cat,
                              const gchar             *name,
                              const gchar             *icon)
{
  LauncherMenuPrivate *priv;
  gchar *uri;  

  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  g_return_if_fail (cat);
  priv = menu->priv;

  uri = g_filename_to_uri (cat->path, NULL, NULL);

  if (name)
  {
    g_free (cat->name);
    cat->name = g_strdup (name);
  }

  if (icon)
  {
    g_free (cat->icon);
    cat->icon = g_strdup (icon);
    g_object_unref (cat->pixbuf);
    cat->pixbuf = NULL;
  }

  launcher_util_edit_category (uri, cat->name, cat->icon);

  g_free (uri);
}



GList*
launcher_menu_get_categories (LauncherMenu *menu)
{
  g_return_val_if_fail (LAUNCHER_IS_MENU (menu), NULL);
  return menu->priv->categories;
}

void
launcher_menu_set_categories (LauncherMenu *menu, GList *categories)
{
  g_return_if_fail (LAUNCHER_IS_MENU (menu));

  g_list_free (menu->priv->categories);
  menu->priv->categories = categories;

  save_categories (menu);
}

gint 
launcher_menu_get_n_categories (LauncherMenu         *menu)
{
  g_return_val_if_fail (LAUNCHER_IS_MENU (menu), 0);

  return g_list_length (menu->priv->categories);
}

void
launcher_menu_set_applications (LauncherMenu         *menu, 
                                LauncherMenuCategory *category,
                                GList                *applications)
{
  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  g_return_if_fail (category);

  g_list_free (category->applications);
  category->applications = applications;

  save_applications (menu, category);
}

GList*
launcher_menu_get_applications (LauncherMenu *menu)
{
  g_return_val_if_fail (LAUNCHER_IS_MENU (menu), NULL);
  return menu->priv->menu_apps;
}

/* Private */
static gchar *
_get_category_id (const gchar *path)
{
  gchar *basename;
  gchar *id;

  if (!path)
    return NULL;

  basename = g_filename_display_basename (path);
  id = g_strndup (basename, strlen (basename) - 10); 

  g_free (basename);
  return id;
}


static LauncherMenuCategory*
make_category (LauncherMenu *menu, GMenuTreeDirectory *dir)
{
  LauncherMenuPrivate *priv = menu->priv;
  LauncherMenuCategory *category = NULL;

  /* if we are refreshing, try searching for the category */
  if (priv->refresh)
  {
    GList *l;
    const gchar *path = gmenu_tree_directory_get_desktop_file_path (dir);
    for (l = priv->old_cats; l; l = l->next)
    {
      LauncherMenuCategory *cat = l->data;
      if (path && cat->path && strcmp (path, cat->path) == 0)
      {
        category = cat;
        break;
      }
    }
    if (category)
    {
      g_list_free (category->applications);
      category->applications = NULL;
      priv->categories = g_list_append (priv->categories, (gpointer)category);

      priv->old_cats = g_list_remove (priv->old_cats, category);
      
      g_debug ("Existing Category: %s", category->name);
      return category;
    }
  }

  category = g_slice_new0 (LauncherMenuCategory);
  category->name = g_strdup (gmenu_tree_directory_get_name (dir));
  category->comment = g_strdup (gmenu_tree_directory_get_comment (dir));
  category->icon = g_strdup (gmenu_tree_directory_get_icon (dir));
  category->applications = NULL;
  category->path = g_strdup (gmenu_tree_directory_get_desktop_file_path (dir));
  category->id = _get_category_id (category->path);
  category->is_new = priv->refresh;
  if (!category->id)
    g_debug ("Unable to create id:%s %s", category->name, category->path);
  priv->categories = g_list_append (priv->categories, (gpointer)category);

  g_debug ("New Category: %s", category->name);

  return category;
} 

/*
 * Each application has a pointer to its category, and gets added to the 
 * categories application list and the main applicaton list 
 */
static void
make_application (LauncherMenu *menu, 
                  GMenuTreeEntry *entry, 
                  LauncherMenuCategory *category)
{
  LauncherMenuPrivate *priv = menu->priv;
  LauncherMenuApplication *app = NULL;

  /* if we are refreshing, try searching for the category */
  if (priv->refresh)
  {
    GList *a;
    const gchar *path = gmenu_tree_entry_get_desktop_file_path (entry);

    LauncherMenuApplication *application;
    for (a = priv->old_apps; a; a = a->next)
    {
      application = a->data;
      if (strcmp (path, application->path) == 0)
      {
        app = application;
        break;
      }
    }
    if (app)
    {
      app->category = category;
      category->applications = g_list_append (category->applications, app);
      priv->menu_apps = g_list_append (priv->menu_apps, app);

      priv->old_apps = g_list_remove (priv->old_apps, app);
      return;
    }    
  }

  app = g_slice_new0 (LauncherMenuApplication);
  app->name = g_strdup (gmenu_tree_entry_get_name (entry));
  app->comment = g_strdup (gmenu_tree_entry_get_comment (entry));
  app->icon = g_strdup (gmenu_tree_entry_get_icon (entry));
  app->exec = g_strdup (gmenu_tree_entry_get_exec (entry));
  app->path = g_strdup (gmenu_tree_entry_get_desktop_file_path (entry));
  app->category = category;
  app->pid = -1;
  app->window = NULL;
  app->isnew = priv->refresh;

  category->applications = g_list_append (category->applications,
                                          (gpointer)app);
  priv->menu_apps = g_list_append (priv->menu_apps, (gpointer)app);
} 

/* 
 * Traverse through the root tree, treating each directory as a category, and
 * each entry as an application. We only want 1st tier categories, so we pass
 * a 'category' variable to the function, which, if present, blocks the
 * 2nd tier directory from becoming a new category, and uses it's parent as the
 * category.
 */
static void
load_menu_from_directory (LauncherMenu *menu, 
                          GMenuTreeDirectory *dir,
                          LauncherMenuCategory *category)
{
  GSList *list, *l;
  LauncherMenuCategory *root = NULL;
  
  if (dir == NULL)
    return;

  list = gmenu_tree_directory_get_contents (dir);
  for (l = list; l; l = l->next)
  {
    GMenuTreeItem *item = (GMenuTreeItem*)l->data;

    switch (gmenu_tree_item_get_type (item))
    {
      case GMENU_TREE_ITEM_DIRECTORY:
        
        if (!category)
        {
          load_menu_from_directory (menu, 
                                   GMENU_TREE_DIRECTORY (item),
                                   make_category (menu,
                                                 GMENU_TREE_DIRECTORY (item)));
        }
        else
          load_menu_from_directory (menu, 
                                    GMENU_TREE_DIRECTORY (item), category);
        break;
      case GMENU_TREE_ITEM_ENTRY:
        if (category)
          make_application (menu, GMENU_TREE_ENTRY (item), category);
        else
        {
          if (root == NULL)
            root = make_category (menu, dir);
          make_application (menu, GMENU_TREE_ENTRY (item), root);        
        }

        break;
        
      default:
        break;
    }
    gmenu_tree_item_unref (item);
  }  
  g_slist_free (list);
}

static GMenuTree *
load_menu_from_tree (LauncherMenu *menu, const gchar *name)
{
  LauncherMenuPrivate *priv;
  GMenuTreeDirectory *root = NULL;
  GMenuTree *tree = NULL;

  g_return_val_if_fail (LAUNCHER_IS_MENU (menu), NULL);
  priv = menu->priv;
 
  tree = gmenu_tree_lookup (name, GMENU_TREE_FLAGS_SHOW_EMPTY);
  if (!tree)
  {
    g_warning ("Unable to find %s", name);
    return NULL;
  }
  root = gmenu_tree_get_root_directory (tree);
  load_menu_from_directory (menu, root, NULL); 
  gmenu_tree_item_unref (root);

  gmenu_tree_add_monitor (tree, (GMenuTreeChangedFunc)tree_changed, menu);

  return tree;
}

static gboolean
_poor_mans_garbage_collection (GList *pixbufs)
{
  GList *p;

  for (p = pixbufs; p; p = p->next)
  {
    while (G_IS_OBJECT (p->data))
    {
      g_debug ("PMGC: Swept one up");
      g_object_unref (p->data);
    }
  }
  g_list_free (pixbufs);

  return FALSE;
}

static void
free_menu (LauncherMenu *menu)
{
  LauncherMenuPrivate *priv;
  GList *l;
  GList *pixbufs = NULL;
  static gint i = 0;

  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  priv = menu->priv;

  for (l = priv->old_apps; l; l = l->next)
  {
    LauncherMenuApplication *app = l->data;
    
    g_free (app->name);
    g_free (app->comment);
    g_free (app->icon);
    g_free (app->exec);
    g_free (app->path);
    
    if (G_IS_OBJECT (app->pixbuf))
    {
      if (G_OBJECT (app->pixbuf)->ref_count >1)
        g_object_unref (app->pixbuf);
      else
        pixbufs = g_list_append (pixbufs, app->pixbuf);
    }
    g_slice_free (LauncherMenuApplication, app);
    app = NULL;
  }
  g_list_free (priv->old_apps);
  priv->old_apps = NULL;

  for (l = priv->old_cats; l; l = l->next)
  {
    LauncherMenuCategory *cat = l->data;
    
    g_free (cat->name);
    g_free (cat->comment);
    g_free (cat->icon);
    g_free (cat->path);
    g_free (cat->id);
    g_list_free (cat->applications);
    cat->applications = NULL;

    if (G_IS_OBJECT (cat->pixbuf))
    {
      if (G_OBJECT (cat->pixbuf)->ref_count >1)
        g_object_unref (cat->pixbuf);
      else
        pixbufs = g_list_append (pixbufs, cat->pixbuf);
    }
    g_slice_free (LauncherMenuCategory, cat);
    l->data = NULL;
  }
  g_list_free (priv->old_cats);
  priv->old_cats = NULL;

  g_timeout_add (1000, (GSourceFunc)_poor_mans_garbage_collection, pixbufs);

  i++;
}

static void
save_categories (LauncherMenu *menu)
{
  LauncherMenuPrivate *priv = menu->priv;
  GConfClient *client = gconf_client_get_default ();
  GSList *order = NULL;
  GList *c;

  for (c = priv->categories; c; c = c->next)
  {
    LauncherMenuCategory *cat = c->data;
    if (cat->path)
      order = g_slist_append (order, cat->path);
  }

  gconf_client_set_list (client, LAUNCHER_CAT_ORDER, GCONF_VALUE_STRING, 
                         order, NULL);

  g_slist_free (order);
  g_object_unref (client);
}

static void
sort_categories (LauncherMenu *menu)
{
  LauncherMenuPrivate *priv;
  GConfClient *client = gconf_client_get_default ();
  GSList *order, *p;
  GList *c;
  priv = menu->priv;
  gint i = 0;

  gconf_client_add_dir (client, LAUNCHER_CAT_ORDER, 
                        GCONF_CLIENT_PRELOAD_NONE, NULL);

  order = gconf_client_get_list (client, LAUNCHER_CAT_ORDER, 
                                 GCONF_VALUE_STRING, NULL);

  for (p = order; p; p = p->next)
  {
    const gchar *path = p->data;
    for (c = priv->categories; c; c = c->next)
    {
      LauncherMenuCategory *cat = c->data;

      if (path && cat->path && strcmp (path, cat->path) == 0)
      {
        priv->categories = g_list_remove (priv->categories, cat);
        priv->categories = g_list_insert (priv->categories, cat, i);
        i++;
        break;
      }
    }
  }
  g_slist_foreach (order, (GFunc)g_free, NULL);
  g_slist_free (order);
  order = NULL;

  save_categories (menu);

  g_object_unref (client);
}

static gchar *
get_gconf_key_for_category (LauncherMenuCategory *cat)
{
  gchar *basename;
  gchar *key;

  basename = g_filename_display_basename (cat->path);
  _normalize (basename);
  
  key = g_strdup_printf ("%s/%s", LAUNCHER_APP_ORDER, basename);

  g_free (basename);

  return key;
}

static void
save_applications (LauncherMenu *menu, LauncherMenuCategory *cat)
{
  GConfClient *client = gconf_client_get_default ();
  GSList *order = NULL;
  GList *a;
  gchar *key;

  key = get_gconf_key_for_category (cat);

  for (a = cat->applications; a; a = a->next)
  {
    LauncherMenuApplication *app= a->data;
    if (app->path)
      order = g_slist_append (order, app->path);
  }

  gconf_client_set_list (client, key, GCONF_VALUE_STRING, 
                         order, NULL);

  g_free (key);
  g_slist_free (order);
  g_object_unref (client);
}

static void
sort_applications (LauncherMenu *menu, LauncherMenuCategory *cat)
{
  GConfClient *client = gconf_client_get_default ();
  GSList *order, *p;
  GList *a;
  gchar *key;
  gint i = 0;

  key = get_gconf_key_for_category (cat);
  gconf_client_add_dir (client, LAUNCHER_APP_ORDER, 
                        GCONF_CLIENT_PRELOAD_NONE, NULL);

  order = gconf_client_get_list (client, key,
                                 GCONF_VALUE_STRING, NULL);

  for (p = order; p; p = p->next)
  {
    const gchar *path = p->data;
    for (a = cat->applications; a; a = a->next)
    {
      LauncherMenuApplication *app = a->data;

      if (path && app->path && strcmp (path, app->path) == 0)
      {
        cat->applications = g_list_remove (cat->applications, app);
        cat->applications= g_list_insert (cat->applications, app, i);
        i++;
        break;
      }
    }
  }
  g_slist_foreach (order, (GFunc)g_free, NULL);
  g_slist_free (order);
  order = NULL;

  save_applications (menu, cat);

  g_object_unref (client);
  g_free (key);
}

static gboolean
list_changed (LauncherMenu *menu)
{
  LauncherMenuPrivate *priv;
  GMenuTreeDirectory *root;
  GList *c;
  
  g_return_val_if_fail (LAUNCHER_IS_MENU (menu), FALSE);
  priv = menu->priv;

  g_debug ("Menu changed\n");

  priv->old_cats = priv->categories;
  priv->old_apps = priv->menu_apps;
  priv->categories = priv->menu_apps = NULL;

  priv->refresh = TRUE;

  root = gmenu_tree_get_root_directory (priv->app_tree);
  load_menu_from_directory (menu, root, NULL);
  gmenu_tree_item_unref (root);

  priv->refresh = FALSE;

  free_menu (menu);

  sort_categories (menu);
  for (c = priv->categories; c; c = c->next)
    sort_applications (menu, c->data);

  g_signal_emit (menu, _menu_signals[MENU_CHANGED], 0);
  menu->priv->tag = 0;  
  
  return FALSE;
}

/*
 * FIXME: The tree changing should not be destructive. Find a way to make it
 * work without re-creating the entire menu again.
 */
static void
tree_changed (GMenuTree *tree, LauncherMenu *menu)
{
  LauncherMenuPrivate *priv;
  GMenuTreeDirectory *root;
  
  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  priv = menu->priv;

  root = gmenu_tree_get_root_directory (tree);
  gmenu_tree_item_unref (root);  
  
  if (priv->tag)
  {
    return;
  }
  priv->tag = g_timeout_add (500, (GSourceFunc)list_changed, menu);
}

static void
on_theme_changed (GtkIconTheme *theme, LauncherMenu *menu)
{
  LauncherMenuPrivate *priv;
  GMenuTreeDirectory *root;
  
  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  priv = menu->priv;

  root = gmenu_tree_get_root_directory (priv->app_tree);
  gmenu_tree_item_unref (root);  
  
  if (priv->tag)
  {
    return;
  }
  priv->tag = g_timeout_add (500, (GSourceFunc)list_changed, menu);
}

static gboolean
load_menu_icons (LauncherMenu *menu)
{
  LauncherMenuPrivate *priv;
  GList *a;

  g_return_val_if_fail (LAUNCHER_IS_MENU (menu), FALSE);
  priv = menu->priv;

  for (a = priv->menu_apps; a; a = a->next)
  {
    LauncherMenuApplication *app = a->data;
    
    if (!GDK_IS_PIXBUF (app->pixbuf))
    {
      launcher_menu_application_get_icon (app);
      return TRUE;
    }
  }

  return FALSE;
}

/* GObject functions */
static void
launcher_menu_dispose (GObject *object)
{
  free_menu (LAUNCHER_MENU (object));

  G_OBJECT_CLASS (launcher_menu_parent_class)->dispose (object);
}

static void
launcher_menu_finalize (GObject *menu)
{
  LauncherMenuPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_MENU (menu));
  priv = LAUNCHER_MENU (menu)->priv;

  G_OBJECT_CLASS (launcher_menu_parent_class)->finalize (menu);
}


static void
launcher_menu_class_init (LauncherMenuClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->finalize = launcher_menu_finalize;
  obj_class->dispose = launcher_menu_dispose;

   _menu_signals[MENU_CHANGED] = 
      g_signal_new ("menu-changed",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (LauncherMenuClass, menu_changed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

 
  g_type_class_add_private (obj_class, sizeof (LauncherMenuPrivate)); 
}

static void
launcher_menu_init (LauncherMenu *menu)
{
  LauncherMenuPrivate *priv;
  GtkIconTheme *theme;
  gchar *filename;
  GList *c;
    
  priv = menu->priv = LAUNCHER_MENU_GET_PRIVATE (menu);

  priv->categories = NULL;
  priv->menu_apps = NULL;
  priv->tag = 0;
  priv->refresh = FALSE;

  filename = g_build_filename (g_get_home_dir (), ".config", 
                               "dell-launcher","applications.menu", NULL);

  priv->app_tree = load_menu_from_tree (menu, filename);

  sort_categories (menu);
  for (c = priv->categories; c; c = c->next)
    sort_applications (menu, c->data);

  /* React to theme-changed signals */
  theme = gtk_icon_theme_get_default ();
  g_signal_connect (theme, "changed",
                    G_CALLBACK (on_theme_changed), menu);

  g_free (filename);

  /* Try loading icons in the background when the launcher isn't being used */
  g_idle_add ((GSourceFunc)load_menu_icons, menu);
}

LauncherMenu*
launcher_menu_get_default (void)
{
  static LauncherMenu *menu = NULL;
  
  if (menu == NULL)
    menu = g_object_new (LAUNCHER_TYPE_MENU, 
                         NULL);

  return menu;
}
