/*
 * Copyright (C) 2008 Dell Inc.
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include <config.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <gio/gio.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <libgnome/libgnome.h>
#include <libgnomevfs/gnome-vfs.h>

#include <clutter/clutter.h>
#include <clutter/clutter-x11.h>

#include "launcher-app.h"
#include "launcher-config.h"
#include "launcher-defines.h"
#include "launcher-menu.h"

#define SET_DESKTOP_BACKGROUND_KEY "gconftool-2 --set /desktop/gnome/background/picture_options --type=string stretched"


/* Forwards */
static gboolean redraw_me        (void);
static void     set_window_hints (ClutterStage *stage);
static void     setup_launcher   (void);
static gboolean captured_event   (ClutterActor *stage, ClutterEvent *event);
static void     size_changed     (GdkScreen *screen, ClutterActor *app);
static void     on_system_resume (LauncherConfig *config);

static gboolean windowed = FALSE;
static gboolean debug    = FALSE;
static gint     width    = 1024;
static gint     height   = 600;
static gboolean norestart= FALSE;

static GOptionEntry entries[] =
{
  {
    "windowed",
    'w', 0,
    G_OPTION_ARG_NONE,
    &windowed,
    "Launch in windowed mode (for testing, 800x640)",
    NULL
  },
  {
    "width",
    's', 0,
    G_OPTION_ARG_INT,
    &width,
    "Width of window",
    NULL
  },  
  {
    "height",
    'h', 0,
    G_OPTION_ARG_INT,
    &height,
    "Height of window",
    NULL
  },  
  {
    "debug",
    'd', 0,
    G_OPTION_ARG_NONE,
    &debug,
    "Debug mode",
    NULL
  },  
  {
    "no-restart",
    'r', 0,
    G_OPTION_ARG_NONE,
    &norestart,
    "Do not restart on VT changes",
    NULL
  },  
  {
    NULL
  }
};

gint
main (gint argc, gchar *argv[])
{
	GdkScreen *screen;
  LauncherConfig *cfg;
  ClutterActor *stage;
  ClutterColor black = { 0x00, 0x00, 0x00, 0xff };
  LauncherApp *app;
  GError *error = NULL;

  bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  g_thread_init (NULL);
  g_type_init ();
  g_set_application_name ("Dell Launcher");

  gtk_init (&argc, &argv);
  clutter_init_with_args (&argc, &argv,
                          " - Dell Launcher", entries,
                          NULL,
                          &error);
  if (error)
  {
      g_print ("Unable to run Dell Launcher: %s", error->message);
      g_error_free (error);
      return EXIT_FAILURE;
  }
  gnome_vfs_init ();
  gnome_program_init ("dell-launcher", "1.0", LIBGNOME_MODULE,
                      argc, argv, 
                      NULL);

  setup_launcher ();

  /* Create actual application */
  stage = clutter_stage_get_default ();
  screen = gdk_screen_get_default ();
  if (windowed)
  {
    clutter_actor_set_size (stage, width, height);
    clutter_stage_set_user_resizable (CLUTTER_STAGE (stage), TRUE);
    g_setenv ("LAUNCHER_WINDOWED", "1", TRUE);
  }
  else
  {
    GdkRectangle rect;
    gdk_screen_get_monitor_geometry (screen, 0, &rect);
    clutter_actor_set_size (stage,
                          rect.width,
                          rect.height);
    set_window_hints (CLUTTER_STAGE (stage));
  }
  
  clutter_stage_set_color (CLUTTER_STAGE (stage), &black);
  clutter_stage_set_use_fog (CLUTTER_STAGE (stage), FALSE);

  /* Debugging */
  if (debug)
  {
    g_signal_connect (stage, "captured-event",
                      G_CALLBACK (captured_event), NULL);
  }

  /* No restart */
  if (norestart)
    g_setenv ("LAUNCHER_NORESTART", "1", TRUE);
  else
    g_setenv ("LAUNCHER_NORESTART", "0", TRUE);

  /* Init() the main application */
  cfg = launcher_config_get_default ();
  app = launcher_app_get_default ();
 
  clutter_actor_show (stage);

  g_signal_connect (screen, "size-changed", 
                    G_CALLBACK (size_changed), app);
  g_signal_connect (cfg, "resume-event",
                    G_CALLBACK (on_system_resume), NULL);
  
  if (cfg->is_poulsbo)
  {
    g_timeout_add_seconds (1, (GSourceFunc)redraw_me, NULL);
    g_timeout_add_seconds (2, (GSourceFunc)redraw_me, NULL);
    g_timeout_add_seconds (4, (GSourceFunc)redraw_me, NULL);
  }
  
  clutter_main ();

  if (windowed)
    g_setenv ("LAUNCHER_WINDOWED", "0", TRUE);
  
  if (norestart)
    g_setenv ("LAUNCHER_NORESTART", "0", TRUE);
  
  return EXIT_SUCCESS;
}

static gboolean redraw_me (void)
{
  clutter_actor_queue_redraw (clutter_stage_get_default ());
  return FALSE;
}

static void
restart_launcher ()
{
  gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                    "dell-launcher --no-restart", NULL);
  clutter_main_quit ();
}

static void
on_system_resume (LauncherConfig *config)
{
  if (!norestart)
    restart_launcher ();
}

/*
 * What happens when the stage changes size
 * NOTE: We shouldn't launch with --no-restart if a VT switch has yet to occur
 */
static void
restart_launcher_for_resize ()
{
  if (norestart)
    gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                      "dell-launcher --no-restart", NULL);
  else
    gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                      "dell-launcher", NULL);
  clutter_main_quit ();
}
  
static void    
size_changed (GdkScreen    *screen,
              ClutterActor *app)
{
  static gboolean already_called = FALSE;

  if (CSW () == gdk_screen_get_width (screen) 
      && CSH () == gdk_screen_get_height (screen))
    return;

  if (already_called)
    return;

  already_called = TRUE;
  restart_launcher_for_resize ();
}

/*
 * Apply the 'desktop' window type to the clutter-stage window. Also, set the
 * 'panel' hints, to stop the windows from maximising the entire way.
 */
static void
set_window_hints (ClutterStage *stage)
{
  GdkDisplay *display = gdk_display_get_default (); 
  Display *xdisplay;
  Window stage_win;
  Atom atom;
  GList *list = NULL;
  GdkPixbuf *pixbuf;
  GdkWindow *window;

  xdisplay = GDK_DISPLAY_XDISPLAY (display);
  stage_win = clutter_x11_get_stage_window (stage);

  /* 
   * Make the clutter window a 'desktop' window, i.e maximised, but below
   * everything else
   */
  atom = gdk_x11_get_xatom_by_name_for_display (display,
                                                "_NET_WM_WINDOW_TYPE_DESKTOP");
  XChangeProperty (xdisplay, stage_win,
                   gdk_x11_get_xatom_by_name_for_display (display,
                                                "_NET_WM_WINDOW_TYPE"),
                   XA_ATOM , 32, PropModeReplace, 
                   (guchar *)&atom, 1);

  /*
   * The window icon 
   */
  pixbuf = gdk_pixbuf_new_from_file (PKGDATADIR"/netbook-mode.png", NULL);
  list = g_list_append (list, pixbuf);
  window = gdk_window_foreign_new (stage_win);
  gdk_window_set_icon_list (window, list);
}

static gchar *
get_menu (void)
{
  gchar menu_buffer[] = {""
  "<!DOCTYPE Menu PUBLIC \"-//freedesktop//DTD Menu 1.0//EN\" "
  "\"http://www.freedesktop.org/standards/menu-spec/1.0/menu.dtd\">"
  "<Menu>"
  "<Name>Applications</Name>"
  "<Directory>Applications.directory</Directory>"
  "<AppDir>%s/.config/dell-launcher/applications</AppDir>"
  "<DirectoryDir>%s/.config/dell-launcher/desktop-directories</DirectoryDir>"
  "<MergeDir>%s/.config/dell-launcher/menus</MergeDir>"
  "</Menu> <!-- End Applications -->"};

  return g_strdup_printf (menu_buffer, 
                          g_get_home_dir(),
                          g_get_home_dir (),
                          g_get_home_dir ());
}

static gboolean
create_menu (const gchar *filename)
{
  GFile *file;
  GFileOutputStream *stream;
  GError *error = NULL;
  gchar *buffer = NULL;
  gsize written = 0;

  file = g_file_new_for_path (filename);

  stream = g_file_create (file, G_FILE_CREATE_NONE, NULL, &error);
  if (error)
  {
    g_warning ("Error creating .menu file: %s", error->message);
    g_error_free (error);
    g_object_unref (file);
    return FALSE;
  }

  buffer = get_menu ();

  g_output_stream_write_all (G_OUTPUT_STREAM (stream),
                             buffer,
                             strlen (buffer),
                             &written,
                             NULL, 
                             &error);
  if (error)
  {
    g_warning ("Error writing data: %s", error->message);
    g_error_free (error);
    g_object_unref (file);
    g_object_unref (stream);
    return FALSE;
  }
                           
  g_assert (written == strlen (buffer));

  g_output_stream_close (G_OUTPUT_STREAM (stream), NULL, &error);
  if (error)
  {
    g_warning ("Error saving file file: %s", error->message);
    g_error_free (error);
    g_object_unref (file);
    g_object_unref (stream);
    return FALSE;
  }
  g_object_unref (stream);
  g_object_unref (file);
  g_free (buffer);


  return TRUE;
}

static gboolean
create_dir (const gchar *dir)
{
  if (!g_file_test (dir, G_FILE_TEST_EXISTS))
  {
    g_mkdir_with_parents (dir, 0700);
    return TRUE;
  }
  return FALSE;
}

static void
copy_dir (const gchar *from, const gchar *to)
{
  GDir *dir;
  GError *error = NULL;
  const gchar *name;

  dir = g_dir_open (from, 0, &error);
  if (error)
  {
    g_warning ("Unable to open %s", from);
    g_error_free (error);
    return;
  }
  
  while ((name = g_dir_read_name (dir)))
  {
    GFile *to_file;
    GFile *from_file;
    gchar *from_filename = g_build_filename (from, name, NULL);
    gchar *to_filename = g_build_filename (to, name, NULL);

    from_file =g_file_new_for_path (from_filename);
    to_file = g_file_new_for_path (to_filename);

    error = NULL;
    g_file_copy (from_file, to_file, G_FILE_COPY_OVERWRITE,
                 NULL, NULL, NULL, &error);
    if (error)
    {
      g_warning ("Unable to copy %s to %s: %s", 
                 from_filename, to_filename, error->message);
      g_error_free (error);
    }
    g_object_unref (from_file);
    g_object_unref (to_file);
    g_free (from_filename);
    g_free (to_filename);
  }

  g_dir_close (dir);
}

/*
 * This is all the initial set-up magic
 */
static void
setup_launcher (void)
{
#define SWITCHERDIR    DATADIR"/desktop-switcher"
#define MERGEDIR       ".config/menus/applications-merged"
#define DELL_APPS_MENU "/dell-applications.menu"
#define DELL_SETS_MENU  "/dell-settings.menu"
  gchar *dir, *file;

  /* Some tests that shouldn't be in here */
  dir = g_build_filename (g_get_home_dir (), ".config", "dell-launcher", NULL);
  if (!(g_file_test (dir, G_FILE_TEST_EXISTS)))
  {
    /* Copy over the smaller menu */
    GFile *from;
    GFile *to;
    gchar *temp;

    temp = g_build_filename (g_get_home_dir (), MERGEDIR, NULL);
    g_mkdir_with_parents (temp, 0777);
    g_free (temp);

    /* The applications.menu file */
    temp = g_build_filename (g_get_home_dir (), MERGEDIR, DELL_APPS_MENU, NULL);
    from = g_file_new_for_path (SWITCHERDIR DELL_APPS_MENU);
    to = g_file_new_for_path (temp);
    g_file_copy (from, to, 0, NULL, NULL, NULL, NULL);
    g_object_unref (from);
    g_object_unref (to);
    g_free (temp);

    /* The settings.menu file */
    temp = g_build_filename (g_get_home_dir (), MERGEDIR, DELL_SETS_MENU, NULL);
    from = g_file_new_for_path (SWITCHERDIR DELL_SETS_MENU);
    to = g_file_new_for_path (temp);
    g_file_copy (from, to, 0, NULL, NULL, NULL, NULL);
    g_object_unref (from);
    g_object_unref (to);
    g_free (temp);   

    g_debug ("Copying menu suppresion files over");
  }
  g_free (dir);

  dir = g_build_filename (g_get_home_dir (), ".local", 
                          "share", "applications", NULL);
  create_dir (dir);
  g_free (dir);

  dir = g_build_filename (g_get_home_dir (), ".config", 
                          "menus", "applications-dell",  NULL);
  create_dir (dir);
  g_free (dir);

  dir = g_build_filename (g_get_home_dir (), 
                          ".config", "dell-launcher", NULL);
  create_dir (dir);
  g_free (dir);

  dir = g_build_filename (g_get_home_dir (), 
                          ".config", "dell-launcher", "applications", NULL);
  if (create_dir (dir))
    copy_dir (PKGDATADIR"/applications", dir);
  g_free (dir);

  dir = g_build_filename (g_get_home_dir (), 
                      ".config", "dell-launcher", "desktop-directories", NULL);
  if (create_dir (dir))
    copy_dir (PKGDATADIR"/desktop-directories", dir);  
  g_free (dir);

  dir = g_build_filename (g_get_home_dir (), 
                          ".config", "dell-launcher", "menus", NULL);
  if (create_dir (dir))
    copy_dir (PKGDATADIR"/menus", dir);  
  g_free (dir);

  dir = g_build_filename (g_get_home_dir (), 
                          ".config", "dell-launcher", "icons", NULL);
  create_dir (dir);
  g_free (dir);

  file = g_build_filename (g_get_home_dir(), ".config", "dell-launcher",
                           "applications.menu", NULL);
  if (!g_file_test (file, G_FILE_TEST_EXISTS))
    create_menu (file);
  g_free (file);
}



/*
 * Print all captured events for debugging 
 */
static gboolean
captured_event (ClutterActor *stage, ClutterEvent *event)
{
  switch (event->type)
  {
    case CLUTTER_MOTION:
      g_print ("Motion event:\t %d %d\n", event->motion.x, event->motion.y);
      break;
    case CLUTTER_BUTTON_PRESS:
      g_print ("Button %d press:\t %d %d\n", event->button.button, 
                                           event->button.x, event->button.y);
      break;
    case CLUTTER_BUTTON_RELEASE:
      g_print ("Button %d release:\t %d %d\n", event->button.button, 
                                           event->button.x, event->button.y);
      break;
    case CLUTTER_STAGE_STATE:
      g_print ("Stage %s\n", 
               (event->stage_state.new_state & CLUTTER_STAGE_STATE_ACTIVATED) ? "Deactivate": "Activated");
      break;
    default:
      break;
   }
  return FALSE;
}
