# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
Scan the resource providers, add some metadata and store it in the database
"""

from elisa.core import common
from elisa.core.utils.locale_helper import system_encoding
from elisa.core.media_uri import MediaUri

from twisted.internet import task, reactor, threads
from twisted.python.failure import Failure

from elisa.core.utils.cancellable_defer import cancellable_coiterate, \
        CancelledError

from elisa.core.utils import defer

import time, os

from elisa.core.components.resource_provider import ResourceProvider

from elisa.plugins.database.scanner_models import Statistic, ScanResource
from elisa.plugins.database.database_parser import DatabaseParser
from elisa.plugins.database.models import \
         PICTURES_SECTION, MUSIC_SECTION, VIDEO_SECTION

from elisa.plugins.base.local_resource import LocalResource

from elisa.plugins.base.models.file import DirectoryModel

try:
    import dbus
    from elisa.plugins.database.dbus_iface import DBusInterface
except ImportError:
    dbus = None

# FIXME: we put here also schema stuff not strictly related with the media
# scanner (the "favorites" table), because we don't have a general
# mechanism to do database stuff yet
SCHEMA = [
        "CREATE TABLE IF NOT EXISTS music_albums" \
                " (name VARCHAR PRIMARY KEY, cover_uri VARCHAR," \
                " release_date VARCHAR);",
        "CREATE TABLE IF NOT EXISTS artists"
                " (name VARCHAR PRIMARY KEY, image_uri VARCHAR);",
        "CREATE TABLE IF NOT EXISTS track_artists" \
                " (artist_name VARCHAR, track_path VARCHAR," \
                " PRIMARY KEY (artist_name, track_path));",
        "CREATE TABLE IF NOT EXISTS music_tracks" \
                " (file_path VARCHAR PRIMARY KEY, title VARCHAR,"\
                " track_number INTEGER, album_name VARCHAR, "\
                " duration INTEGER, genre VARCHAR);",
        "CREATE TABLE IF NOT EXISTS tags"\
                " (name VARCHAR PRIMARY KEY);",
        "CREATE TABLE IF NOT EXISTS files" \
                " (path VARCHAR PRIMARY KEY, source VARCHAR," \
                " modification_time INTEGER DEFAULT 0, mime_type VARCHAR," \
                " hidden INTEGER DEFAULT 0," \
                " deleted INTEGER, playcount INTEGER DEFAULT 0, last_played" \
                " INTEGER DEFAULT 0);",
        "CREATE TABLE IF NOT EXISTS file_tags" \
                " (file_path VARCHAR, tag_name VARCHAR,"\
                " PRIMARY KEY(file_path, tag_name));",
        "CREATE TABLE IF NOT EXISTS images"\
                " (file_path VARCHAR PRIMARY KEY, size VARCHAR, " \
                " shot_time INTEGER, with_flash INTEGER DEFAULT 0," \
                " orientation INTEGER, gps_altitude INTEGER, " \
                " gps_latitude VARCHAR, gps_longitude VARCHAR," \
                " album_name VARCHAR, section INTEGER DEFAULT 0);",
        "CREATE TABLE IF NOT EXISTS photo_albums"\
                " (name VARCHAR PRIMARY KEY, preview_uri VARCHAR);",
        "CREATE TABLE IF NOT EXISTS videos"\
                " (file_path VARCHAR PRIMARY KEY, name VARCHAR,"\
                " creation_time INTEGER, size VARCHAR, duration VARCHAR,"\
                " codec VARCHAR, thumbnail_uri VARCHAR);",

        "CREATE TABLE IF NOT EXISTS favorites"\
                " (id INTEGER PRIMARY KEY, section TEXT,"\
                " added_time TEXT, title TEXT, subtitle,"\
                " foreign_id TEXT, foreign_class TEXT, uri TEXT, type TEXT);",
        # delete triggers
        "CREATE TRIGGER IF NOT EXISTS \"file del\" "\
            "BEFORE DELETE ON files BEGIN "\
                "DELETE FROM music_tracks WHERE file_path = old.path; "\
                "DELETE FROM images WHERE file_path = old.path; "\
                "DELETE FROM videos WHERE file_path = old.path; "\
            "END", 
        "CREATE TRIGGER IF NOT EXISTS \"music track album del\" "\
            "AFTER DELETE ON music_tracks "\
            "WHEN (SELECT count(*) from music_tracks WHERE "\
                    "album_name = old.album_name) = 0  BEGIN "\
                "DELETE FROM music_albums WHERE name = old.album_name; "\
            "END",
        "CREATE TRIGGER IF NOT EXISTS \"del artist\" "\
            "AFTER DELETE ON track_artists "\
            "WHEN (SELECT count(*) from track_artists WHERE "\
                   "artist_name = old.artist_name) = 0  BEGIN "\
                "DELETE FROM artists WHERE name = old.artist_name; "\
            "END",
        "CREATE TRIGGER IF NOT EXISTS \"music track artists del\" "\
            "AFTER DELETE ON music_tracks BEGIN "\
                "DELETE FROM track_artists WHERE track_path = old.file_path; "\
            "END",
        "CREATE TRIGGER IF NOT EXISTS \"image album del\" "\
            "AFTER DELETE ON images "\
            "WHEN (SELECT count(*) from images WHERE album_name = "\
                    "old.album_name) = 0  BEGIN " \
                "DELETE FROM photo_albums WHERE name = old.album_name; "\
            "END",
        ]

class MediaScanner(ResourceProvider):
    """
    Asynchronous working media scanner.
    
    @ivar store: the store for the data
    @type store: L{storm.twisted.storm.DeferredStore}
    """

    default_config = {"delay" : 0.1, "scan_every" : 24}
    config_doc = {
        "delay": "the delay (in seconds) between processing two files",
        "scan_every" : "the delay (in hours) between two automatic scans"
        }

    supported_uri = '^media_scanner://localhost/'

    def __init__(self):
        super(MediaScanner, self).__init__()
        self.store = None
        self._current_delay = None
        self._start_delay = None
        self._current_scan_deferred = None

        self.scan_stat = Statistic()
        self.scan_stat.uri = MediaUri('media_scanner://localhost/statistic')
        self._pending_auto_rescan = None

        self.running = False

    def _scan_directories(self):
        # ugly hack to start the scanning process for the directories the user
        # has set up in the configuration file
        directories_section = common.application.config.get_section('directories')
        if not directories_section:
            self.debug("no default directories")
            return

        for section in ('music', 'pictures', 'video'):
            paths = directories_section.get(section, [])
            for path in paths:
                if path != '*default*': # skip the default configuration thingy
                    uri = MediaUri('file://%s' % path)
                    self.debug("adding %s" % uri)
                    self.put(uri, None, section=section)

    def initialize(self):

        def set_parser(parser):
            self.parser = parser
            self.parser.store = self.store
            # start the scanner with a delay of 5 seconds
            self._start_delay = reactor.callLater(5, self._scan_directories)
            return self
 
        def set_local_resource(local_resource):
            self.local_resource = local_resource
            return self

        dfr = super(MediaScanner, self).initialize()
        store = common.application.store
        self.store = store

        dfr.addCallback(lambda x: self.create_schema())
        # FIXME: add configuration system. Passing through my configuration is
        # not exactly nice.
        dfr.addCallback(lambda x: DatabaseParser.create(self.config))
        dfr.addCallback(set_parser)
        dfr.addCallback(lambda x: LocalResource.create({}))
        dfr.addCallback(set_local_resource)
        # start dbus at the end
        dfr.addCallback(self._initialize_dbus)
        return dfr

    def _initialize_dbus(self, result=None):
        if dbus is None:
            # no dbus support
            return self

        bus = dbus.SessionBus()

        self.dbus_scanner = DBusInterface(self, bus,
                '/com/fluendo/Elisa/Plugins/Database/MediaScanner')
        return self

    def _clean_dbus(self):
        if dbus is None:
            # no dbus support
            return

        bus = dbus.SessionBus()
        self.dbus_scanner.remove_from_connection(bus,
                '/com/fluendo/Elisa/Plugins/Database/MediaScanner')

        # remove the reference cycle
        self.dbus_scanner = None


    def clean(self):
        self._clean_dbus()
        if self._current_delay and not self._current_delay.called:
            self._current_delay.cancel()

        if self._start_delay and not self._start_delay.called:
            self._start_delay.cancel()

        return super(MediaScanner, self).clean()
        
    def _update_layout(self, result):
        def get_columns(result):
            return [ x[1] for x in result ]

        def get_all(result):
            return result.get_all()

        def sqlexec(string):
            return self.store.execute(string)
            
        def get_table_info(table_name):
            dfr = sqlexec("PRAGMA table_info('%s')" % table_name)
            dfr.addCallback(get_all)
            dfr.addCallback(get_columns)
            return dfr

        def got_files(result):
            commit = False
            if not u'playcount' in result:
                sqlexec("ALTER TABLE files ADD playcount INTEGER DEFAULT 0;")
                commit = True
            if not u'last_played' in result:
                sqlexec("ALTER TABLE files ADD last_played INTEGER DEFAULT 0;")
                commit = True
            if u'hidden' not in result:
                sqlexec("ALTER TABLE files ADD hidden INTEGER DEFAULT 0;")
                commit = True

            if commit:
                return self.store.commit()

        def got_images(result):

            def build_update_query(section, path_list):
                path = path_list[0]
                query = "UPDATE images SET section = %d" \
                        " WHERE file_path LIKE \"%s%%\"" % \
                                             (section, os.path.join(path, ''))
                for path in path_list[1:]:
                    query += " OR file_path LIKE \"%s%%" % \
                                              os.path.join(path, '')
                return query

            def update_image_albums(result):
                # we remove the photo albums that are not related to images we
                # got from the pictures section. That fixes databases affected
                # by #259427.
                sqlexec("DELETE FROM photo_albums WHERE name NOT IN " \
                        "(SELECT DISTINCT album_name FROM images "\
                        "WHERE section=%d)" % PICTURES_SECTION)
                return self.store.commit()

            def update_image_section(result):
                # FIXME: blocks
                # FIXME: doesn't handle images that are in more than one
                # section
                directories_section = common.application.config.get_section \
                                                              ('directories')
                for section_name, section in (('music', MUSIC_SECTION),
                                              ('video', VIDEO_SECTION),
                                              ('pictures', PICTURES_SECTION)):
                    path_list = directories_section.get(section_name, [])
                    if path_list != []:
                        query = build_update_query(section, path_list)
                        sqlexec(query)

                dfr =  self.store.commit()
                dfr.addCallback(update_image_albums)
                return dfr

            if not 'shot_time' in result:
                # shot_time is part of the first update we did for the image
                # table. as it is not there, we have to update the table
                # FIXME: this way to identify an Database Update is not nice
                sqlexec("ALTER TABLE images ADD shot_time INTEGER;")
                sqlexec("ALTER TABLE images ADD with_flash INTEGER DEFAULT 0;")
                sqlexec("ALTER TABLE images ADD orientation INTEGER;")
                sqlexec("ALTER TABLE images ADD gps_altitude INTEGER;")
                sqlexec("ALTER TABLE images ADD gps_latitude VARCHAR;")
                sqlexec("ALTER TABLE images ADD gps_longitude VARCHAR;")
                sqlexec("ALTER TABLE images ADD album_name VARCHAR;")
                sqlexec("ALTER TABLE images ADD section INTEGER DEFAULT 0;")
                dfr = self.store.commit()
                dfr.addCallback(update_image_section)
                return dfr

            elif not 'section' in result:
                sqlexec("ALTER TABLE images ADD section INTEGER DEFAULT 0;")
                dfr = self.store.commit()
                dfr.addCallback(update_image_section)
                return dfr

        def got_favorites(result):
            if not 'type' in result:
                sqlexec("ALTER TABLE favorites ADD type TEXT;")
                return self.store.commit()

        def got_albums(result):
            if not u'release_date' in result:
                sqlexec("ALTER TABLE music_albums ADD release_date VARCHAR;")
                return self.store.commit()
        
        def got_videos(result):
            if not u'name' in result:
                sqlexec("ALTER TABLE videos ADD name VARCHAR;")
                return self.store.commit()
            if not u'creation_time' in result:
                sqlexec("ALTER TABLE videos ADD creation_time INTEGER;")
                return self.store.commit()

        def do_files_update():
            dfr = get_table_info('files')
            dfr.addCallback(got_files)
            return dfr

        def do_albums_update(old_res=None):
            dfr = get_table_info('music_albums')
            dfr.addCallback(got_albums)
            return dfr

        def do_images_update(old_res=None):
            dfr = get_table_info("images")
            dfr.addCallback(got_images)
            return dfr

        def do_favorites_update(old_res=None):
            dfr = get_table_info("favorites")
            dfr.addCallback(got_favorites)
            return dfr
        
        def do_videos_update(old_res=None):
            dfr = get_table_info("videos")
            dfr.addCallback(got_videos)
            return dfr

        dfr = do_files_update()
        dfr.addCallback(do_albums_update)
        dfr.addCallback(do_images_update)
        dfr.addCallback(do_videos_update)
        dfr.addCallback(do_favorites_update)
        return dfr

    def create_schema(self):
        def go_through(queries):
            for query in queries:
                dfr = self.store.execute(query)
                yield dfr

        dfr = task.coiterate(go_through(SCHEMA))
        dfr.addCallback(lambda result: self.store.commit())
        dfr.addCallback(self._update_layout)
        return dfr

    def running_get(self):
        return self._running

    def running_set(self, value):
        self._running = value
        self.scan_stat.running = value
        if not value:
            self._current_scan_deferred = None

    running = property(fget=running_get, fset=running_set)

    # general Resource Provider API
    def get(self, uri, context_model=None):
        """
        If the filename is C{statistic} you receive the
        L{elisa.plugins.database.scanner_models.Statistic} for this scanner
        """
        if uri.filename == 'statistic':
            return (self.scan_stat, defer.succeed(self.scan_stat))

        return (None, defer.fail(NotImplementedError()))

    def put(self, source_uri, container_uri,
            context_model=None, section=None):
        """
        put another uri into the queue and start the scanning process if it is
        not yet running

        @ivar section:  the name of the section relevant to the content that
                        is put.
        @type section:  L{str}
        """
        # FIXME check the container uri
        # FIXME no check if this uri might already in the queue...

        if source_uri.filename != '':
            # we want to have the content, so we have to add a trailing slash
            source_uri = source_uri.join('')

        def start_scanning(result, source_uri, section):
            name = os.path.basename(os.path.dirname(source_uri.path))
            s = ScanResource(source_uri, name, section=section)
            self.scan_stat.queue.put(s)

            if not self.running:
                self._scan()

            return s.defer

        return self.parser.mark_deleted(source_uri.path).addCallback(start_scanning,
                source_uri, section)

    # internally used
    def _update_stat(self, result, model, stat):
        stat.files_scanned += 1

        # deactivated as we don't use it atm and it leaks memory
        #if isinstance(result, Failure):
        #    stat.files_failed.append( (model.uri, result) )

    def _count_files(self, path):
        path = os.path.abspath(path)

        deferred = defer.Deferred()

        def iterate(path):
            counter = 0
            try:
                for root, dirs, files in os.walk(path.encode(system_encoding())):

                    if deferred.called:
                        # we are cancelled. Stop the processing without calling
                        # the deferred again
                        return

                    # count only the files
                    counter += len(files)

                reactor.callFromThread(deferred.callback, counter)

            except Exception, exc:
                reactor.callFromThread(deferred.errback, exc)

        # push it into background
        threads.deferToThread(iterate, path)

        return deferred

    def _file_found(self, model, stat):
        dfr = self.parser.query_model(model, stat)
        dfr.addBoth(self._update_stat, model, stat)
        return dfr

    def _cleanup_deleted_files(self, result, scan_resource):
        return self.parser.delete_files(scan_resource.root_uri.path)

    def _scan(self, result=None):

        def run_next(result, scan_resource):
            scan_resource.state = ScanResource.SCANNING_DONE

            self.scan_stat.currently_scanning = None
            self.scan_stat.scanned.append(scan_resource)

            if self.scan_stat.queue.empty():
                # stop processing
                self.running = False
                self._reset_auto_rescan(True)
            else:
                reactor.callLater(0.1, self._scan)

        self.running = True

        scan_resource = self.scan_stat.queue.get_nowait()
        self.scan_stat.currently_scanning = scan_resource

        # clear stat
        scan_resource.last_scan = time.time()
        scan_resource.files_scanned = 0
        scan_resource.files_total = 0
        scan_resource.state = ScanResource.SCANNING_FS

        def set_total_files(result, scan_resource):
            scan_resource.files_total = result
            return self._scan_recursive(scan_resource)

        dfr = self._count_files(scan_resource.root_uri.path)
        dfr.addCallback(set_total_files, scan_resource)
        dfr.addCallback(self._cleanup_deleted_files, scan_resource)
        dfr.addCallback(run_next, scan_resource)

        self._current_scan_deferred = dfr

    def _rescan(self):
        self.debug("Starting to scan everything again")
        dfr = self._reschedule_scanned()
        dfr.addCallback(self._scan)
        return dfr

    def _reset_auto_rescan(self, restart=False):
        if self._pending_auto_rescan and self._pending_auto_rescan.active():
            self._pending_auto_rescan.cancel()

        rescan_every = self.config.get('scan_every', 0) * 3600
        if restart and rescan_every != 0:
            pending = reactor.callLater(rescan_every, self._rescan)
            self._pending_auto_rescan = pending

    def _reschedule_scanned(self):
        scan_stat = self.scan_stat

        scanned, scan_stat.scanned = scan_stat.scanned, [] 

        deleted_dfrs = []
        for stat in scanned:
            scan_stat.queue.put(stat)
            deleted_dfrs.append(self.parser.mark_deleted(stat.root_uri.path))

        if len(deleted_dfrs) == 0:
            return defer.succeed(None)

        return defer.DeferredList(deleted_dfrs)

    def _scan_recursive(self, scan_resource):

        def stopper(data, file):
            self.warning("%s failed: %s" % (file, data))
            return None

        def scan_children(model, dirs):
            def iterator(model, dirs):
                while len(model.files):
                    item = model.files.pop(0)
                    if isinstance(item, DirectoryModel):
                        # delay searching the subfolders for later
                        dirs.append(item.uri)
                    else:
                        dfr = self._file_found(item, scan_resource)
                        yield dfr

            return cancellable_coiterate(iterator, model, dirs)

        def scan_dirs(dirs):
            def iterator(dirs):
                while len(dirs):
                    uri = dirs.pop(0)
                    if uri.filename != '':
                        # we have a filename append the slash to get the
                        # content
                        uri = uri.join('')
                    model, dfr = self.local_resource.get(uri)
                    dfr.addCallback(scan_children, dirs)
                    dfr.addErrback(stopper, uri.path)
                    yield dfr

            return cancellable_coiterate(iterator, dirs)

        return scan_dirs([scan_resource.root_uri])
