# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.widgets.button import Button
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.theme import Theme
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text

from elisa.plugins.poblesec.browser_controller import BrowserController
from elisa.core.input_event import *

from pgm.timing import implicit
import pgm
import gobject


class TopBarButton(Widget):
    
    unselected_opacity = 60
    
    PLAY = 0
    STOP = 1
    LIST = 2
    COVERF = 3
    GRID = 4
    
    def __init__(self):
        super(TopBarButton, self).__init__()
        
        self.index = -1
        self._buttons = []
        
        self.play_button = self._create_image()
        self.stop_button = self._create_image()
        self.list_button = self._create_image()
        self.coverflow_button = self._create_image()
        self.grid_button = self._create_image()
        
        self._update_style_properties(self._style.get_items())
        
    def _create_image(self):
        image =  Image()
        self._buttons.append(image)
        self.add(image)
        image.layout = pgm.IMAGE_SCALED
        image.bg_a = 0
        image.fg_a = self.unselected_opacity
        image.visible = True
        return image
    
    def set_index(self, value):
        if value > len(self._buttons) -1:
            return

        if self.index != -1:
            self._buttons[self.index].fg_a = self.unselected_opacity

        if value >= -1:
            self.index = value
        else:
            self.index = -1

        if self.index > -1:
            self._buttons[self.index].fg_a = 255

    def _update_style_properties(self, props=None):
        super(TopBarButton, self)._update_style_properties(props)

        if props is None:
            return

        theme = Theme.get_default()

        # FIXME: bad prefix 'top-bar-', has just to be dropped
        for key, value in props.iteritems():
            if key == 'button-height':
                height = props['button-height']
                width = props.get('button-width', self.play_button.width)
                space = props.get('space', 0)

                x_offset = 0

                self.play_button.width, self.play_button.height = width, height
                self.play_button.x = x_offset
                self.play_button.y = (1.0 - height) / 2.0
                x_offset += width

                self.stop_button.width, self.stop_button.height = width, height
                self.stop_button.x = x_offset
                self.stop_button.y = (1.0 - height) / 2.0
                x_offset += width + space

                self.list_button.width, self.list_button.height = width, height
                self.list_button.x = x_offset
                self.list_button.y = (1.0 - height) / 2.0
                x_offset += width

                self.coverflow_button.width, self.coverflow_button.height = width, height
                self.coverflow_button.x = x_offset
                self.coverflow_button.y = (1.0 - height) / 2.0
                x_offset += width

                self.grid_button.width, self.grid_button.height = width, height
                self.grid_button.x = x_offset
                self.grid_button.y = (1.0 - height) / 2.0
            elif key == 'top-bar-play-button':
                self.play_button.set_from_file(theme.get_resource(value))
            elif key == 'top-bar-stop-button':
                self.stop_button.set_from_file(theme.get_resource(value))
            elif key == 'top-bar-list-button':
                self.list_button.set_from_file(theme.get_resource(value))
            elif key == 'top-bar-coverflow-button':
                self.coverflow_button.set_from_file(theme.get_resource(value))
            elif key == 'top-bar-grid-button':
                self.grid_button.set_from_file(theme.get_resource(value))


class TopBar(Widget):
    """
    Poblesec top bar
    """
        
    def __init__(self):
        super(TopBar, self).__init__()
        
        self.background = Image()
        self.add(self.background)       
        self.background.bg_a = 0
        self.background.layout = pgm.IMAGE_FILLED
        self.background.visible = True
        
        self.button_bar = TopBarButton()
        self.add(self.button_bar)
        self.button_bar.visible = True
        
        self.image = Image()
        self.add(self.image)
        self.image.bg_a = 0
        self.image.visible = True
        
        self.text = Text()
        self.add(self.text)
        self.text.weight = pgm.TEXT_WEIGHT_BOLD
        self.text.bg_a = 0
        self.text.visible = True
        
        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(TopBar, self)._update_style_properties(props)

        if props is None:
            return

        theme = Theme.get_default()

        # FIXME: bad prefix 'top-bar-', has just to be dropped
        for key, value in props.iteritems():
            if key == 'top-bar-buttons-x':
                self.button_bar.x = value
            elif key == 'top-bar-buttons-width':
                self.button_bar.width = value
            elif key == 'top-bar-image-x':
                self.image.x = value
            elif key == 'top-bar-image-height':
                self.image.height = value
                self.image.y = ( 1.0 - value ) / 2.0
            elif key == 'top-bar-image-width':
                self.image.width = value
            elif key == 'top-bar-text-x':
                self.text.x = value
            elif key == 'top-bar-text-height':
                self.text.height = value
                self.text.y = ( 1.0 - value ) / 2.0
            elif key == 'top-bar-text-width':
                self.text.width = value
            elif key == 'top-bar-font-family':
                self.text.font_family = value
            elif key == 'top-bar-background':
                self.background.set_from_file(theme.get_resource(value))


class BottomBar(Widget):
    """
    Poblesec bottom bar
    """
        
    def __init__(self, history):
        super(BottomBar, self).__init__()

        self.background = Image()
        self.add(self.background)
        self.background.bg_color = (0, 0, 0, 0)
        self.background.layout = pgm.IMAGE_FILLED
        self.background.visible = True

        self.back_button = Image()
        self.add(self.back_button)
        #frontend.load_from_theme('elisa.plugins.poblesec.back_button',
        #                         self.back_button)
        self.back_button.bg_a = 0
        self.back_button.alignment = pgm.IMAGE_BOTTOM_LEFT
        self.back_button.x, self.back_button.y = (0.0, 0.0)
        self.back_button.width, self.back_button.height = (0.1, 1.0)
        self.back_button.visible = True

        def go_back(*args):
            history.go_back()

        self.back_button.connect("clicked", go_back)

        #create animation stuff
        self.animated = implicit.AnimatedObject(self)
        settings = {'duration': 500,
                    'transformation': implicit.DECELERATE,
                    'end_callback': None}
        self.animated.setup_next_animations(**settings)
        self.animated.mode = implicit.REPLACE
        
        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(BottomBar, self)._update_style_properties(props)

        if props is None:
            return

        theme = Theme.get_default()

        for key, value in props.iteritems():
            if key == 'back-button-image':
                self.back_button.set_from_file(theme.get_resource(value))


class Crumb(Widget):
    
    def __init__(self):
        super(Crumb, self).__init__()

        self.text = Text()
        self.add(self.text)
        self.text.bg_a = 0
        self.text.weight = pgm.TEXT_WEIGHT_BOLD
        self.text.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self.text.visible = True

        self.background = Image()
        self.add(self.background)
        self.background.bg_a = 0
        self.background.layout = pgm.IMAGE_FILLED
        self.background.visible = True
        
        #create animation stuff
        self.animated = implicit.AnimatedObject \
                                                (self)
        self.settings = {'duration': 500,
                         'transformation': implicit.DECELERATE,
                         'end_callback': None}
        self.animated.setup_next_animations(**self.settings)
        self.animated.mode = implicit.REPLACE

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(Crumb, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'font-family':
                self.text.font_family = value
            elif key == 'text-height':
                self.text.height = value
                self.text.y = (1.0-self.text.height)/2.0

class BreadCrumbs(Widget):
    """
    DOCME

    @ivar logo: image visually located at the right end of the bread crumbs
    @type logo: L{elisa.plugins.pigment.graph.image.Image}
    """

    __gsignals__ = {'crumb-clicked': (gobject.SIGNAL_RUN_LAST,
                                      gobject.TYPE_BOOLEAN,
                                      (gobject.TYPE_PYOBJECT,))}

    def __init__(self):
        super(BreadCrumbs, self).__init__()
        self._crumbs = []
        self.previous_crumb = None

        self.background = Image()
        self.add(self.background)
        self.background.bg_a = 0
        self.background.layout = pgm.IMAGE_FILLED
        self.background.visible = True

        # placeholder for logo, optionally used by plugins
        self.logo = Image()
        self.add(self.logo)
        self.logo.size = (0.1, 1.0)
        self.logo.position = (1.0-self.logo.width, 0.0, 0.0)
        self.logo.bg_color = (0, 0, 0, 0)
        self.logo.opacity = 200
        self.logo.visible = True

        self.master = Image()
        self.add(self.master)
        self.master.visible = False

        #create animation stuff
        self.animated = implicit.AnimatedObject \
                                                (self)
        self.settings = {'duration': 200,
                    'transformation': implicit.DECELERATE,
                    'end_callback': None}
        self.animated.setup_next_animations(**self.settings)
        self.animated.mode = implicit.REPLACE

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(BreadCrumbs, self)._update_style_properties(props)

        if props is None:
            return

        theme = Theme.get_default()

        for key, value in props.iteritems():
            if key == 'crumb-width':
                self._crumb_width = value
            elif key == 'crumb-arrow-width':
                self._crumb_arrow_width = value
            elif key == 'crumb-height':
                self._crumb_height = value
            elif key == 'text-x-offset':
                self._text_x_offset = value
            elif key == 'text-width':
                self._text_width = value
            elif key == 'bread-crumbs-background':
                self.background.set_from_file(theme.get_resource(value))
            elif key == 'crumb-background':
                self.master.set_from_file(theme.get_resource(value))

    def push_crumb(self, controller):
        if hasattr(controller, 'display_name'):
            text = controller.display_name
        else:
            text = _('unknown')
        crumb = Crumb()
        crumb.text.label = text
        crumb.background.connect('clicked', lambda *args: self.emit('crumb-clicked', crumb))
        self._crumbs.append(crumb)
        self.add(crumb)
        crumb.background.set_from_image(self.master)
        index = len(self._crumbs) - 1
        crumb.height = self._crumb_height
        crumb.width = self._crumb_width
        start_x = (self._crumb_width - self._crumb_arrow_width) * ( index - 1 )
        end_x = (self._crumb_width - self._crumb_arrow_width) * index
        z = 0.01 - 0.0001 * index
        crumb.x, crumb.y , crumb.z =  start_x, 0.0, z
        crumb.text.x = self._text_x_offset
        crumb.text.width = self._text_width
        crumb.visible = True
        
        #do not animate the first one
        if index == 0:
            crumb.x = end_x
        else:
            crumb.animated.x = end_x
      
    def pop_crumb(self):
        if len(self._crumbs) == 0:
            return
        
        if self.previous_crumb != None:
            self.previous_crumb.animated.stop_animations()
        
        self.previous_crumb = self._crumbs.pop()
        index = len(self._crumbs) - 1
        end_x = (self._crumb_width - self._crumb_arrow_width) * index
        
        def _remove_crumb(p):
            self.previous_crumb.animated.update_animation_settings \
                                                    (end_callback=None)
            #catch exception if the function remove_all_crumbs is called during 
            #the loop 
            try:
                self.remove(self.previous_crumb)
            except:
                pass
            self.previous_crumb = None
              
        self.previous_crumb.animated.update_animation_settings \
                                                    (end_callback=_remove_crumb)
        self.previous_crumb.animated.x = end_x
        self.previous_crumb.animated.setup_next_animations \
                                                (**self.previous_crumb.settings)
        
    def remove_all_crumbs(self):
        self._crumbs = []
        for w in self.get_children():
            #catch exception if the callback _remove_crumb is called during 
            #the loop 
            try:
                self.remove(w)
            except:
                pass

class TopBarWithBreadCrumbs(Widget):
    
    def __init__(self):
        super(TopBarWithBreadCrumbs, self).__init__()
        
        self.topbar = TopBar()
        self.add(self.topbar)
        self.topbar.z = 0.02
        self.topbar.visible = True
        self._display_bread_crumb = True
                
        self.bread_crumb = BreadCrumbs()
        self.add(self.bread_crumb)
        self.bread_crumb.z = 0
        self.bread_crumb.visible = True
        self.bread_crumb.opacity = 0

        #create animation stuff
        self.animated = implicit.AnimatedObject(self)
        settings = {'duration': 500,
                    'transformation': implicit.DECELERATE,
                    'end_callback': None}
        self.animated.setup_next_animations(**settings)
        self.animated.mode = implicit.REPLACE
        
        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(TopBarWithBreadCrumbs, self)._update_style_properties(props)

        if props is None:
            return
        
        my_props = ['top-bar-height', 'bread-crumb-height']
        if set(my_props).issubset(props.keys()):
            self.topbar.height = props['top-bar-height']
            self.bread_crumb.height = props['bread-crumb-height']

            if self._display_bread_crumb == True:
                self.bread_crumb.y = self.topbar.height
            else:
                self.bread_crumb.y = self.topbar.height - value
        
    def show_bread_crumb(self):
        self._display_bread_crumb = True
        self.bread_crumb.animated.y = self.topbar.height
        self.bread_crumb.animated.opacity = 255
        
    def hide_bread_crumb(self):
        self._display_bread_crumb = False
        def _remove_all_crumbs(p):
            self.bread_crumb.animated.update_animation_settings \
                                                    (end_callback=None)
            self.bread_crumb.remove_all_crumbs()
        
        self.bread_crumb.animated.update_animation_settings \
                                               (end_callback=_remove_all_crumbs)
        self.bread_crumb.animated.y = self.topbar.height - \
                                                        self.bread_crumb.height
        self.bread_crumb.animated.opacity = 0
        self.bread_crumb.animated.setup_next_animations \
                                               (**self.bread_crumb.settings)
    
class PoblesecBrowserController(BrowserController):

    def set_frontend(self, frontend):
        super(PoblesecBrowserController, self).set_frontend(frontend)
        
        #create the top bar with bread crumb
        self.topbar_with_bread_crumb = TopBarWithBreadCrumbs()
        self.widget.add(self.topbar_with_bread_crumb)
        self.topbar_with_bread_crumb.bread_crumb.connect('crumb-clicked',
                                                         self._bread_crumb_clicked)
        self.topbar_with_bread_crumb.visible = True

        #hide it
        delta = self.topbar_with_bread_crumb.topbar.height + \
                        self.topbar_with_bread_crumb.bread_crumb.height
        self.topbar_with_bread_crumb.y = -delta
        self.topbar_with_bread_crumb.opacity = 0

        #create the bottom bar
        self.bottombar = BottomBar(self.history)
        self.widget.add(self.bottombar)
        self.bottombar.visible = True

        #hide it
        self.bottombar.height = 0.075
        self.bottombar.y = 1 + self.bottombar.height
        self.bottombar.opacity = 0
        
        bb = self.topbar_with_bread_crumb.topbar.button_bar
        bb.play_button.connect('clicked', self._bt_clicked_cb, bb.PLAY)
        bb.stop_button.connect('clicked', self._bt_clicked_cb, bb.STOP)
        bb.list_button.connect('clicked', self._bt_clicked_cb, bb.LIST)
        bb.coverflow_button.connect('clicked', self._bt_clicked_cb, bb.COVERF)
        bb.grid_button.connect('clicked', self._bt_clicked_cb, bb.GRID)
        
        self.history.connect('push_controller', self._push_controller_signal)
        self.history.connect('pop_controller', self._pop_controller_signal)

        bb.connect('focus', self._button_bar_focus_changed)

    def _bread_crumb_clicked(self, bread_crumbs, crumb):
        if not self.history:
            return

        while bread_crumbs._crumbs[-1] != crumb:
            self.history.go_back()

    def _button_bar_focus_changed(self, button_bar, focus):
        if focus:
            button_bar.set_index(0)
        else:
            button_bar.set_index(-1)

    def _bt_clicked_cb(self, drawable, x, y, z, button, time, pressure, index):
        if not self.has_focus():
            return

        self.fire_button_bar(index)
    
    def _place_controller(self, controller):
        if self.history.index == 0:
            controller.widget.x, controller.widget.y = (0.0, 0.0)
            controller.widget.width, controller.widget.height = (1.0, 1.0)
        else:
            y = self.topbar_with_bread_crumb.topbar.height \
                            + self.topbar_with_bread_crumb.bread_crumb.height 
            controller.widget.x, controller.widget.y = (0.0, y)
            controller.widget.width, controller.widget.height = (1.0, 1.0 - y)

    def _go_section_start(self, *args):
        if self.history:
            while self.history.index > 1:
                self.history.go_back()

    def set_section_logo(self, logo_resource):
        """
        Load L{logo_resource} into the bread crumbs bar. That logo will be
        visually located at the right end of the bar.
        If logo_resource is None, unload previously loaded resource.

        @param logo_resource: resource pointing to the logo
        @type logo_resource:  str
        """
        image = self.topbar_with_bread_crumb.bread_crumb.logo
        if logo_resource != None:
            self.frontend.load_from_theme(logo_resource, image)
        else:
            image.clear()

    def _push_controller_signal(self, history, previous, current_dfr):
        current_dfr.addCallback(self._new_controller_pushed, previous)

    def _new_controller_pushed(self, current, previous):
        if self.history.index > 0:
            self._update_crumbs_logo(previous, current)

            if self.history.index == 1:
                topbar = self.topbar_with_bread_crumb.topbar
                topbar.text.label = current.display_name
                topbar.image.connect('clicked', self._go_section_start)
                topbar.text.connect('clicked', self._go_section_start)
                image_path = previous.model[previous.menu.selected_item_index].icon
                self.frontend.load_from_theme(image_path, topbar.image)
                 
            self.topbar_with_bread_crumb.animated.y = 0
            self.topbar_with_bread_crumb.animated.opacity = 255
            
            if self.history.index > 1:
                self.topbar_with_bread_crumb.bread_crumb.push_crumb(current)
                self.topbar_with_bread_crumb.show_bread_crumb()

            self.bottombar.animated.y = 1 - self.bottombar.height
            self.bottombar.animated.opacity = 255
 
        self.topbar_with_bread_crumb.topbar.button_bar.set_index(-1)
        return current

    def _update_crumbs_logo(self, previous_controller, current_controller):
        # if the new controller defines a logo attribute it is loaded replacing
        # the currently displayed one
        current_logo = getattr(current_controller, "crumbs_logo", None)
        previous_logo = getattr(previous_controller, "crumbs_logo", None)

        if current_logo != previous_logo:
            self.set_section_logo(current_logo)

    def _pop_controller_signal(self, history, previous, current):
        if self.history.index < 1:
            delta = self.topbar_with_bread_crumb.topbar.height + \
                        self.topbar_with_bread_crumb.bread_crumb.height
            self.topbar_with_bread_crumb.animated.y = -delta
            self.topbar_with_bread_crumb.animated.opacity = 0

            self.bottombar.animated.y = 1 + self.bottombar.height
            self.bottombar.animated.opacity = 0
            
        #I do not pop the last crumb because I don't want any animation
        #this last crumb will be removed after hide_bread_crumb() animation
        if self.history.index > 1:
            self.topbar_with_bread_crumb.bread_crumb.pop_crumb()
            self._update_crumbs_logo(previous, current)

        if self.history.index <= 1:
            self.topbar_with_bread_crumb.hide_bread_crumb()

        self.topbar_with_bread_crumb.topbar.button_bar.set_index(-1)

    def fire_button_bar(self, index):
        bb = self.topbar_with_bread_crumb.topbar.button_bar
        if index == bb.PLAY:
            controllers = self.frontend.retrieve_controllers('/poblesec')
            main = controllers[0]
            player = main.current_player.player
            if len(player.playlist) > 0 and player.current_index != -1:
                # switch only to the player if there is something to show/do
                main.show_current_player()
                player.play()
        elif index == bb.STOP:
            controllers = self.frontend.retrieve_controllers('/poblesec')
            main = controllers[0]
            main.stop_all_players()
        elif index == bb.LIST:
            # we assume current is a ListSwitcher
            switcher = self.history.current
            switcher.switch_mode(switcher.modes[0])
            self.history.current.widget.focus = True
        elif index == bb.COVERF:
            # we assume current is a ListSwitcher
            switcher = self.history.current
            switcher.switch_mode(switcher.modes[1])
            self.history.current.widget.focus = True
        elif index == bb.GRID:
            # we assume current is a ListSwitcher
            switcher = self.history.current
            switcher.switch_mode(switcher.modes[2])
            self.history.current.widget.focus = True

    def handle_input(self, manager, input_event):
        button_bar = self.topbar_with_bread_crumb.topbar.button_bar

        if self.history.current.has_focus() == True and \
                self.history.current.handle_input(manager, input_event) == True:
            return True
        
        #If I'm not in the main menu
        if self.history.index > 0:
   
            if input_event.value == EventValue.KEY_MENU and \
                                                    self.history != None:
                self.history.go_back()
                return True
            index = button_bar.index
            
            if input_event.value == EventValue.KEY_GO_LEFT:
                button_bar.set_index(index-1)
                if (index-1) < 0:
                    self.history.current.widget.focus = True
                return True
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                if not button_bar.focus:
                    button_bar.focus = True
                else:
                    button_bar.set_index(index+1)
                return True
            elif input_event.value == EventValue.KEY_GO_UP:
                if not button_bar.focus:
                    button_bar.focus = True
                return True
            elif input_event.value == EventValue.KEY_GO_DOWN and index > -1:
                self.history.current.widget.focus = True
                return True
            elif input_event.value == EventValue.KEY_OK and button_bar.focus:
                self.fire_button_bar(index)
                return True
                
        return False
