# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Olivier Tilloy <olivier@fluendo.com>

from elisa.core.component import Component

from twisted.internet import task


class HierarchyController(Component):

    """
    Generic controller that should be used as a base class for all custom
    controllers that define a browsable hierarchy.

    The model of the controller is usually populated in its initialize method,
    while the behaviour when an item of the list representing a given level of
    the hierarchy is clicked is defined by the node_clicked method.

    Every deferred call done in the context of this controller should be
    registered in the internal dictionary of deferreds so as to keep track of
    them and allow their cancellation if needed (e.g. when cleaning the
    controller). Each value of this internal dictionary is a list, meaning that
    several deferred calls can be associated to one key.

        >>> self.register_deferred(key, deferred)

    To cancel all the deferred calls associated to a given key, do:

        >>> self.cancel_deferreds(key)

    @ivar deferreds: a dictionary of currently pending deferred calls
    @type deferreds: C{dict} of C{list} of
                     {elisa.core.utils.cancellable_defer.CancellableDeferred}
    @ivar actions: a list of contextual actions for this controller.
    @type actions: C{list} of L{elisa.plugins.poblesec.actions.Action}
    """

    def initialize(self):
        self.deferreds = {}
        self.actions = self.make_actions()

        # This call to super has to be done after the initialization of
        # self.actions, because it will take us into the hierarchy of the other
        # superclass of the class inheriting from us - if present. Which may rely on the
        # fact that self.actions are already up and filled.
        # E.g.: see elisa.plugins.poblesec.base.preview_list.MenuItemPreviewListController
        deferred = super(HierarchyController, self).initialize()

        return deferred

    def node_clicked(self, widget, item):
        """
        Callback invoked when an item of the list representing a given level of
        the hierarchy is clicked.

        @param widget: the selected list item widget in the view
        @type widget:  L{elisa.plugins.pigment.widgets.widget.Widget}
        @param item:   the selected list item in the controller's model
        @type item:    L{elisa.core.components.model.Model}
        """
        raise NotImplementedError()

    def make_actions(self):
        """
        Build contextual actions for this controller.

        @return: the actions for the controller
        @rtype:  C{list} of L{elisa.plugins.poblesec.actions.Action}
        """
        return []

    def register_deferred(self, key, deferred):
        """
        Register a deferred call to be associated to a given key.
        """
        # TODO: add a callback/errback to the deferred that removes it from the
        # dictionary once fired or errbacked?
        try:
            self.deferreds[key].append(deferred)
        except KeyError:
            self.deferreds[key] = [deferred]

    def cancel_deferreds(self, key):
        """
        Cancel all the currently pending deferred calls associated to one given
        key in the internal dictionary.

        @param key: the key in the internal deferreds dictionary
        @type key:  any immutable type
        """
        if not key in self.deferreds:
            return

        for deferred in self.deferreds[key]:
            if not deferred.called:
                deferred.cancel()

        del self.deferreds[key]

    def cancel_all_deferreds(self):
        """
        Cancel all the currently pending deferred calls.
        """
        def cancel_all():
            for key in self.deferreds.keys():
                self.cancel_deferreds(key)
                yield None

        return task.coiterate(cancel_all())

    def clean(self):
        return self.cancel_all_deferreds()
