# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL.EXCEPTION" in the root of this distribution
# including a special exception to use Elisa with Fluendo's plugins and
# about affiliation parameters.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>


from elisa.core.components.resource_provider import ResourceProvider
from elisa.plugins.http_client.http_client import ElisaHttpClient
from elisa.plugins.http_client.extern.client_http import ClientRequest

from elisa.core.media_uri import MediaUri

from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.base.models.media import RawDataModel
from elisa.plugins.yesfm.constants import SERVER_NAME, SERVER_PORT, \
                                          API_KEY, API_PATH
from elisa.plugins.yesfm.models import ArtistModel, ArtistsSearchResultModel,\
                                       AlbumModel, AlbumsSearchResultModel, \
                                       TrackModel, TracksSearchResultModel, \
                                       PlayableModel, \
                                       LoginModel, \
                                       GeneralSearchResultModel

from elisa.core.utils import defer
from twisted.internet import task
from twisted.web2 import responsecode
from twisted.web2.stream import BufferedStream, MemoryStream

from simplejson.decoder import JSONDecoder

import mimetools

"""
The Yesfm Rest-JSON-API-v1 Resource Provider
"""


ALWAYSMAP = {'id' : 'yesfmid', 'radio' : 'radioid', 'radiotype' : 'radiotype',
             'rating' : 'rating', 'score' : 'score'}

"""
Map the different models to a tuple that describes how to parse the data into
the model. The tuple contains two dictionaries: the first one is mapping the
attribute (that is a list) to a class that the data should be parsed into. The
second maps the keys of the dictionary to the attribute of the model.
"""
PARSING_MAP = {
    # the different possible data
    ArtistModel: ( {'albums' : AlbumModel, 'relatedartists' : ArtistModel},
                   {'name' : 'name'}),
                   # missing: biography
    AlbumModel: (  {'tracks' : TrackModel, 'artist': ArtistModel},
                   {'title' : 'album', 'volumes' : 'volumes', 'year' : 'year',
                    'volumes' : 'volumes', 'publisher': 'publisher', }),
    TrackModel: ( {'album' : AlbumModel},
                  {'volume' : 'volume', 'trackno': 'track_number',
                   'length' : 'duration', 'title': 'title'}),
    PlayableModel: ({}, {'audiouri' : 'yesfm_uri'}),

    # Search results
    AlbumsSearchResultModel :  ( {'result' : AlbumModel},
                                 {'totalresultsreturned' : 'count'}),
    ArtistsSearchResultModel : ( {'result' : ArtistModel},
                                 {'totalresultsreturned' : 'count'}),
    TracksSearchResultModel : ( {'result' : TrackModel},
                                 {'totalresultsreturned' : 'count'}),

    # a bit tricky and special: the general search
    GeneralSearchResultModel : ( {'artists': ArtistsSearchResultModel,
                            'albums': AlbumsSearchResultModel,
                            'tracks' : TracksSearchResultModel,}, {}),
                            #missing: radio, users
    }


"""
Map uri-path to the corresponding model
"""

PATH_TO_MODEL = {'artist' : ArtistModel, 'album': AlbumModel,
                 'track': TrackModel, 
                 'artists': ArtistsSearchResultModel}

"""
Map the different kind of searches to the right models
"""
SEARCH_TO_MODEL = {'artist' : ArtistsSearchResultModel,
                   'track' : TracksSearchResultModel,
                   'album' : AlbumsSearchResultModel}

class NotLoggedIn(Exception): pass

class ErrorInResponse(Exception): pass

class YesfmResource(ResourceProvider):
    """
    This resource provider makes the Content of Yes.FM available inside elisa.
    It uses the public JSON-RPC-API. See #FIXME: ADD PATH for more informations.
    """
    
    supported_uri = '^http://(%s/($|%s)|cdn.yes.fm)' % (SERVER_NAME, API_PATH)

    default_config = {'userkey': ''}
    config_doc = {'userkey' : 'the login key against the API. please use the' \
                              ' GUI to perform a User-Password Login'}

    def __init__(self):
        super(YesfmResource, self).__init__()
        self.session_key = None
        self.decoder = JSONDecoder()
        self._data_connections = {}

    def initialize(self):
        dfr = super(YesfmResource, self).initialize()
        dfr.addCallback(self._local_initialize)
        return dfr

    def _local_initialize(self, result):
        self._http_client = ElisaHttpClient(SERVER_NAME, SERVER_PORT)
        userkey = self.config['userkey'] or None
        self._login_dfr = self.login(userkey=userkey)
        self._login_dfr.addCallbacks(self._set_session_from_login,
                self._init_login_error)
        return self

    def clean(self):
        if self._login_dfr is not None:
            try:
                self._login_dfr.cancel()
            except defer.AlreadyCalledError:
                pass
        def close_connections(result):
            dfrs = [self._http_client.close()]
            for client in self._data_connections.values():
                dfrs.append(client.close())

            return defer.DeferredList(dfrs, consumeErrors=True)

        dfr = super(YesfmResource, self).clean()
        dfr.addCallback(close_connections)
        return dfr

    def login(self, user=None, password=None, userkey=None):
        """
        """
        data = {}

        login_model = LoginModel()

        if user is not None and password is not None:
            self.debug("Username and password specified.")
            data['authtype'] = 'user'
            data['username'] = user
            data['password'] = password

        elif userkey is not None:
            self.debug("Using userkey")
            data['authtype'] = 'userkey'
            data['userkey'] = userkey

        else:
            self.debug("anonymous login")
            data['authtype'] = 'anon'

        boundary = mimetools.choose_boundary()
        lines = []
        for key, value in data.iteritems():
            lines.append('--' + boundary.encode('utf-8'))
            lines.append('Content-Disposition: form-data; name="%s";' % key)
            lines.append('')
            lines.append(value.encode('utf-8'))

        request = ClientRequest('POST', '/v1/auth/login', None,
                                MemoryStream('\r\n'.join(lines)))
        request.headers.setRawHeaders('Content-Type',
                ['multipart/form-data; boundary=%s' % boundary])
        dfr = self._send_request(request)
        dfr.addCallback(self._parse_response)
        dfr.addCallback(self.decoder.decode)
        dfr.addCallback(self._login_response, login_model)

        dfr.addErrback(self._login_error, login_model)

        return dfr

    
    def get(self, uri, context_model):

        if uri.host != SERVER_NAME:
            return self._get_raw(uri)

        if not self.session_key:
            # login did not yet happen or failed
            # FIXME: would be better to queue them in the case that login is
            # still pending
            return None, defer.fail(NotLoggedIn(uri))

        model, rip_out = self.get_model_for_uri_path(uri.path)
        if not model:
            return None, defer.fail(NotImplementedError(uri))
        
        dfr = self._send_simple_request(uri)
        dfr.addCallback(self._parse_response)
        dfr.addCallback(self.decoder.decode)
        if rip_out is not None:
            dfr.addCallback(self._rip_out, rip_out)
        dfr.addCallback(self.parse_data_into_model, model)

        return model, dfr

    def post(self, uri, **postdata):
        """
        Post certain data to the YesfmResource provider. Currently only two uris
        are supported:
            /v1/auth/login - to try a login. the keyword for that need to be:
                             user and password. returns you a
                             L{elisa.plugins.search.models.LoginModel}
            /              - allows you to set 'sessionkey' and/or 'userkey'.
                             Returns nothing.

        """

        path = uri.path

        if path == '/v1/auth/login':
            dfr = self.login(**postdata)

        elif path == '/':
            try:
                self.session_key = postdata['sessionkey']
            except KeyError:
                pass
            try:
                self.config['userkey'] = postdata['userkey']
            except KeyError:
                pass

            dfr = defer.succeed(None)
        else:
            dfr = defer.fail(NotImplementedError())


        return dfr
            

    def _get_raw(self, uri):
        host = uri.host

        try:
            client = self._data_connections[host]
        except KeyError:
            # no client for that host yet, create one
            client = ElisaHttpClient(host, uri.port or 80)
            self._data_connections[host] = client

        model = RawDataModel()

        # the data servers seem to be picky about having the host in the url, so
        # we have to remove it and give a fullpath only
        u_uri = "%s?%s" % (u_uri.path, u_uri.get_params_string())

        # passing unicode is not allowed
        u_uri = u_uri.encode('utf-8')

        dfr = client.request(u_uri)
        dfr.addCallback(self._parse_response)
        dfr.addCallback(self._parse_raw, model)
        return model, dfr

    def get_model_for_uri_path(self, uri_path):
        """
        Create the model corresponding to the given URI path.
        
        *Attention*: the uri_path is meant to start with /v1/. The next word
        after that decides about the model. See L{PATH_TO_MODEL} to find out for
        which one. A sepcial case is the 'search' keyword as it has sub-classes.
        For search/artist/.. a L{ArtistsSearchResultModel} will be created. For
        search/albums/.. a L{AlbumsSearchResultModel} will be created.

        @param uri_path: the path to the resource on the API-Server
        @type uri_path:  C{str}
        """
        splitted_path = uri_path.split('/')
        
        splitted_path.pop(0) # in from of the first slash is nothing

        # this is done for the version 1 of the API:
        assert splitted_path.pop(0) == 'v1'

        start = splitted_path.pop(0)

        model_class = None

        if start == 'search':
            if len(splitted_path) == 1:
                # there is only one kind part left. we have a general search.
                return GeneralSearchResultModel(), 'result'

            # select the right search
            search_type = splitted_path.pop(0)
            try:
                return SEARCH_TO_MODEL[search_type](), None
            except KeyError:
                pass
        try:
            return PATH_TO_MODEL[start](), start
        except KeyError:
            if start == 'trackaudio':
                # audio track is a bit special:
                return PlayableModel(), None
        return None, None

    def parse_data_into_model(self, data, model):
        """
        Parse the data into the given model.

        @type data:     dictionary
        @param model:   the model to parse the data into
        @type model:    Subclass of L{elisa.core.compontents.model.Model}
        """

        # callback methods
        def setter(model, key, mapped_key, value):
            setattr(model, mapped_key, value)

        def to_obj_parser(model, key, values, klass):
            if not values:
                # the values we should parse are actually an empty list or even
                # worse a NoneType. We should not create any model(s) in that
                # case
                return

            if isinstance(values, list):
                # we have a list of them, so parse each one of them
                dfr = task.coiterate(self._parse_list_of_data(values, \
                       getattr(model, key), klass))
            else:
                obj = klass()
                setattr(model, key, obj)
                dfr = self.parse_data_into_model(values, obj)
            return dfr
        
        def parse_attrs(result, attribute_map):
            dfr = task.coiterate(self._parse_simple(model, \
                    data, attribute_map, setter))
            return dfr

        def ret_model(result, model):
            return model
    
        list_map, attribute_map = PARSING_MAP[type(model)]

        if 'picture' in data:
            model.picture = self._parse_picture(data['picture'])

        dfr = task.coiterate(self._parse_simple(model, list_map,\
                data, to_obj_parser))

        dfr.addCallback(parse_attrs, attribute_map)
        dfr.addCallback(ret_model, model)

        return dfr

    # internal parsing methods:

    def _parse_list_of_data(self, ref_list, list_to_fill, klass):
        for data in ref_list:
            obj = klass()
            list_to_fill.append(obj)
            yield self.parse_data_into_model(data, obj)

    def _rip_out(self, data, ripper):
        try:
            return data[ripper]
        except KeyError:
            return data

    def _parse_picture(self, data):
        model = ImageModel()
        for size in ('small', 'medium', 'large'):
            uri = data[size]
            if uri is not None:
                uri = MediaUri(data[size])
                model.references.append(uri)
                
        return model

    def _parse_simple(self, model, data, key_map, callback):
        for key, value in data.iteritems():
            try:
                mapped = key_map[key]
            except KeyError:
                try:
                    mapped = ALWAYSMAP[key]
                except KeyError:
                    yield None
                    continue

            yield callback(model, key, mapped, value)
    
    # internal connection helper:
    def _send_simple_request(self, uri):
        req = ClientRequest('GET', str(uri.path), None, None)
        return self._send_request(req)

    def _send_request(self, request):

        request.headers.setRawHeaders('X-yesfm-APIkey', [API_KEY])

        if self.session_key:
            request.headers.setRawHeaders('Authorization', \
                    ['yesfmSession auth=%s' % self.session_key])

        return self._http_client.request_full(request)


    def _parse_response(self, response):
        if response.code != responsecode.OK:
            raise ErrorInResponse(response.code)

        buf_stream = BufferedStream(response.stream)

        read_dfr = defer.maybeDeferred(buf_stream.readExactly)
        return read_dfr

    def _parse_raw(self, response_data, model):
        model.data = response_data
        model.size = len(response_data)
        return model

    # response helper:
    def _login_error(self, failure, model):
        failure.trap(ErrorInResponse)

        if failure.value == 403:
            # the API says 403 means the login didn't work
            model.success = False

        return model

    def _set_session_from_login(self, model):
        self._login_dfr = None
        if model.success:
            self.session_key = model.sessionkey
            self.debug("login done. Session key: '%s'" % self.session_key)
        else:
            self.warning("Login failed. Please check your userkey in the"\
                        " configuration file or try to login with username"\
                        " and password again to renew the userkey.")
        return model

    def _init_login_error(self, failure):
        self._login_dfr = None
        self.warning("Login failed: %s" % failure)

    def _login_response(self, response_data, model, store=False):
        model.success = True

        session = response_data['session']
        for key in ('sessionkey', 'usertype', 'userkey'):
            try:
                value = str(session[key])
            except KeyError:
                self.debug("Login result misses %s key" % key)
                continue

            setattr(model, key, value)

        return model
