# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.tests.elisa_test_case import ElisaTestCase
from elisa.core.utils import dist

import os

class TestDist(ElisaTestCase):

    def test_text_from_mappings_valid(self):
        """Test serialising mappings with a valid input"""
        mappings = [("key1", "value1"),
                    ("key2", "value2a"),
                    ("key2", "value2b")]
        serialised = ["key1 = value1", "key2 = value2a", "key2 = value2b", ""]
        serialised = os.linesep.join(serialised)

        result = dist.text_from_mappings(mappings)
        self.assertEquals(result, serialised)

    def test_text_from_mappings_empty(self):
        """Test serialising mappings with an empty input"""
        mappings = []
        result = dist.text_from_mappings(mappings)
        self.assertEquals(result, "")

    def test_text_from_mappings_empty_key(self):
        """Test serialising mappings with an input containing an empty key
        Expected behaviour is to be very fault tolerant and let strange cases
        pass.
        """
        # empty keys are accepted
        mappings = [("key1", "value1"),
                    ("key2", "value2a"),
                    ("key2", "value2b"),
                    ("", "empty")]
        serialised = ["key1 = value1", "key2 = value2a", "key2 = value2b",
                      " = empty", ""]
        serialised = os.linesep.join(serialised)

        result = dist.text_from_mappings(mappings)
        self.assertEquals(result, serialised)

    def test_text_from_mappings_extra_value(self):
        """Test serialising mappings with an input containing a triplet"""
        mappings = [("key1", "value1", "extra_value"),
                    ("key2", "value2a"),
                    ("key2", "value2b")]

        self.failUnlessRaises(ValueError, dist.text_from_mappings, mappings)

    def test_text_from_mappings_none(self):
        """Test serialising None object"""
        result = dist.text_from_mappings(None)
        self.assertEquals(result, "")

    def test_mappings_from_text_valid(self):
        """Test deserialising mappings with a valid input"""
        mappings = [("key1", "value1"),
                    ("key2", "value2a"),
                    ("key2", "value2b")]
        serialised = ["key1 = value1", "key2 = value2a", "key2 = value2b", ""]
        serialised = os.linesep.join(serialised)

        result = dist.mappings_from_text(serialised)
        self.assertEquals(result, mappings)

    def test_mappings_from_text_empty(self):
        """Test deserialising mappings with an empty input"""
        serialised = ""
        result = dist.mappings_from_text(serialised)
        self.assertEquals(result, [])

    def test_mappings_from_text_empty_key(self):
        """Test deserialising mappings with an input containing an empty key
        Expected behaviour is to be very fault tolerant and let strange cases
        pass.
        """
        # empty keys are accepted
        serialised = ["key1 = value1", "key2 = value2a", "key2 = value2b",
                      " = empty", ""]
        serialised = os.linesep.join(serialised)
        mappings = [("key1", "value1"),
                    ("key2", "value2a"),
                    ("key2", "value2b"),
                    ("", "empty")]

        result = dist.mappings_from_text(serialised)
        self.assertEquals(result, mappings)

    def test_mappings_from_text_none(self):
        """Test deserialising None object"""
        result = dist.mappings_from_text(None)
        self.assertEquals(result, [])

    def test_mappings_from_text_multiple(self):
        """Test deserialising a text with multiple assignments"""
        serialised = "key1 = value1 = blabla" + os.linesep

        self.failUnlessRaises(ValueError, dist.mappings_from_text, serialised)

    def test_text_from_mappings_valid_roundtrip(self):
        """Test a complete round trip starting with a mappings"""
        mappings = [("key1", "value1"),
                    ("key2", "value2a"),
                    ("key2", "value2b")]
        text = dist.text_from_mappings(mappings)
        result = dist.mappings_from_text(text)
        self.assertEquals(result, mappings)
