# -*- mode: python; coding: utf-8 -*-
#
# Pigment Python image wrapping example
#
# Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
#
# Author: Loïc Molinari <loic@fluendo.com>

import sys, math, pgm, gobject

# Translate the mapping matrix of the image
def update_pass_cb(viewport, image, translation):
    image.mapping_matrix *= translation
    return True

# Start animating the mapping matrix when the pixbuf is loaded
def on_file_loaded(image, viewport, transformation):
    viewport.connect('update-pass', update_pass_cb, image, transformation)

# Terminate the mainloop on a delete event
def on_delete(viewport, event):
    pgm.main_quit()

# Entry point
def main(args):
    # GObject threads initialization
    gobject.threads_init()

    # OpenGL viewport creation
    gl = pgm.viewport_factory_make('opengl')
    gl.connect('delete-event', on_delete)
    gl.size = (800, 164)
    gl.title = 'Image wrapping'

    # Canvas creation
    cvs = pgm.Canvas()
    cvs.size = (800, 164)
    gl.set_canvas(cvs)

    # Image creation
    img = pgm.Image()
    img.size = (700, 64)
    img.position = (50, 50, 0)
    img.bg_color = (255, 255, 255, 0)
    img.wrapping = (pgm.IMAGE_REPEAT, pgm.IMAGE_REPEAT)
    img.layout = pgm.IMAGE_FILLED

    # Definition of the mapping matrix applying a scale on the x axis to correct
    # the aspect-ratio of the image content distorted by the filled layout and
    # the size of the image
    mapping = pgm.mat4x4_new_identity()
    mapping.scale(800.0 / 64.0, 1.0, 0.0)
    img.mapping_matrix = mapping

    # Create the translation matrix and load the pattern
    t = pgm.mat4x4_new_identity()
    t.translate(-0.005, 0.0, 0.0)
    img.connect('file-loaded', on_file_loaded, gl, t)
    img.set_from_file('examples/pictures/line-pattern.png')

    # Add the image to the canvas, show the objects and start the main loop
    cvs.add(pgm.DRAWABLE_MIDDLE, img)
    img.show()
    gl.show()
    pgm.main()

if __name__ == '__main__':
    sys.exit(main(sys.argv[1:]))
