/*
 *  Copyright (C) 2005 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 *   This code is based on touchpad.cpp in Ksynaptics-0.2.0. 
 *   Copyright (C) 2004 Nadeem Hasan <nhasan@kde.org>
 *   Copyright (C) 2004 Stefan Kombrink <katakombi@web.de>
 */
#include <string.h>
#include "gsynaptics.h"

typedef struct _GSynapticsPrivate GSynapticsPrivate;
struct _GSynapticsPrivate
{
	SynapticsSHM *synshm;
	guint         synclient : 1;
};

#define G_SYNAPTICS_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), G_TYPE_SYNAPTICS, GSynapticsPrivate))

#define SYNSHM(obj) G_SYNAPTICS_GET_PRIVATE(obj)->synshm

static int finger_low[] = { 53, 38, 25, 18, 10 };
static const double coastingSpeedThreshold = 38;

static void     g_synaptics_class_init     (GSynapticsClass *klass);
static void     g_synaptics_init           (GSynaptics      *object);
static void     g_synaptics_dispose        (GObject         *object);
static void     g_synaptics_finalize       (GObject         *object);

static GObjectClass *parent_class = NULL;


static gdouble
g_synaptics_get_value_from_synclient (const gchar *key)
{
	gdouble value;
	gchar *output;

	g_spawn_command_line_sync ("synclient -l", &output, NULL, NULL, NULL);
	
	if (output)
	{
		gchar *pos = strstr (output, key);

		if (pos)
		{
			gchar *p1 = strchr (pos, '=');
			gchar *p2 = NULL;		
			if (p1)
			{
				p2 = strchr (p1, '\n');
			}

			if (p2)
			{
				gchar *str = g_strndup (p1 + 1, p2 - p1);
				gchar *str2 = g_strdup (g_strchug (str));
				value = g_ascii_strtod (str2, NULL);
				g_free (str);
				g_free (str2);
			}
		}
		g_free (output);
	}

	return value;
}

GType
g_synaptics_get_type (void)
{
	static GType object_type = 0;

	if (!object_type)
	{
		static const GTypeInfo object_info =
		{
			sizeof (GSynapticsClass),
			NULL,		/* base_init */
			NULL,		/* base_finalize */
			(GClassInitFunc) g_synaptics_class_init,
			NULL,		/* class_finalize */
			NULL,		/* class_data */
			sizeof (GSynaptics),
			0,		/* n_preallocs */
			(GInstanceInitFunc) g_synaptics_init,
		};

		object_type = g_type_register_static (G_TYPE_OBJECT, "GSynaptics",
				&object_info, 0);
	}

	return object_type;
}

static void
g_synaptics_class_init (GSynapticsClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
  
	gobject_class->dispose = g_synaptics_dispose;
	gobject_class->finalize = g_synaptics_finalize;

	g_type_class_add_private (gobject_class, sizeof(GSynapticsPrivate));
}

static void
g_synaptics_init (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);

	priv->synclient = FALSE;

	{
		gchar *find;
		find = g_find_program_in_path("synclient");
		
		if (find)
		{
			gint status;
			gchar *output, *error;

			g_spawn_command_line_sync ("synclient -l", &output, &error, &status, NULL);

			if (status == 0)
			{
				g_warning("Using synclient");
				priv->synclient = TRUE;
			}
			if (output)
				g_free (output);
			if (error)
				g_free (error);

			g_free (find);
		}
	}
}

static void
g_synaptics_dispose (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
g_synaptics_finalize (GObject *object)
{
	G_OBJECT_CLASS (parent_class)->finalize (object);
}

GSynaptics *
g_synaptics_new (void)
{
	GSynaptics *synaptics;

	synaptics = g_object_new (G_TYPE_SYNAPTICS, NULL);

	return synaptics;
}

gboolean
g_synaptics_is_available (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);

	return (priv->synclient || priv->synshm);
}

static gboolean
g_synaptics_is_valid (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);

	return SYNSHM(synaptics) != NULL || priv->synclient; 
}

gboolean
g_synaptics_is_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{
		return !g_synaptics_get_value_from_synclient ("TouchpadOff");
	}
	else
	{
		return !SYNSHM(synaptics)->touchpad_off;
	}
}

void
g_synaptics_edges (GSynaptics *synaptics)
{
}

gboolean
g_synaptics_is_tapping_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{
		return g_synaptics_get_value_from_synclient ("MaxTapTime") > 0;
	}
	else
	{
		return SYNSHM(synaptics)->tap_time > 0;
	}
}

gint
g_synaptics_tap_time (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return 0;

	if (priv->synclient)
	{
		return (gint)g_synaptics_get_value_from_synclient ("MaxTapTime");
	}
	else
	{
		return SYNSHM(synaptics)->tap_time;
	}
}

#if 0
gint
g_synaptics_tap_action                 (GSynaptics *synaptics, TapType type)
{
}
#endif

gint
g_synaptics_sensitivity (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	gint i;
	if (!g_synaptics_is_valid(synaptics))
		return 0;

	if (priv->synclient)
	{
		for (i = 0; i < 5; i++)
		{
			if (g_synaptics_get_value_from_synclient ("FingerLow") >= finger_low[i])
				return i;
		}
	}
	else
	{
		for (i = 0; i < 5; i++)
		{
			if (SYNSHM(synaptics)->finger_low >= finger_low[i])
				return i;
		}
	}

	return i - 1;
}


gboolean
g_synaptics_is_horizontal_scroll_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{
		return g_synaptics_get_value_from_synclient ("HorizScrollDelta") > 0;
	}
	else
	{
		return SYNSHM(synaptics)->scroll_dist_horiz > 0;
	}
}

gint
g_synaptics_horizontal_scroll_delta (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return 0;

	if (priv->synclient)
	{
		return (gint)g_synaptics_get_value_from_synclient ("HorizScrollDelta");
	}
	else
	{
		return SYNSHM(synaptics)->scroll_dist_horiz;
	}
}

gboolean
g_synaptics_is_vertical_scroll_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{

		return g_synaptics_get_value_from_synclient ("VertScrollDelta") > 0;
	}
	else
	{
		return SYNSHM(synaptics)->scroll_dist_vert > 0;
	}
}

gint
g_synaptics_vertical_scroll_delta (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return 0;

	if (priv->synclient)
	{
		return (gint)g_synaptics_get_value_from_synclient ("VertScrollDelta");
	}
	else
	{
		return SYNSHM(synaptics)->scroll_dist_vert;
	}
}

gboolean
g_synaptics_is_edge_motion_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{
		return g_synaptics_get_value_from_synclient ("EdgeMotionUseAlways") ? TRUE : FALSE;
	}
	else
	{
		return SYNSHM(synaptics)->edge_motion_use_always;
	}
}

gboolean
g_synaptics_is_coasting_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{
		return g_synaptics_get_value_from_synclient ("CoastEnable") ? TRUE : FALSE;
	}
	else
	{
		return SYNSHM(synaptics)->coasting_speed > 0.1;
	}
}

gboolean
g_synaptics_is_circular_scroll_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{
		return g_synaptics_get_value_from_synclient ("CircularScrolling") ? TRUE : FALSE;
	}
	else
	{
		return SYNSHM(synaptics)->circular_scrolling;
	}
}

gint
g_synaptics_circular_scroll_delta (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return 0;

	if (priv->synclient)
	{
		return (gint)(g_synaptics_get_value_from_synclient ("CircScrollDelta") * 1000);
	}
	else
	{
		return (gint)(SYNSHM(synaptics)->scroll_dist_circ * 1000);
	}
}

ScrollTrigger
g_synaptics_circular_scroll_trigger (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return NoTrigger;

	if (priv->synclient)
	{
		return (ScrollTrigger)g_synaptics_get_value_from_synclient ("CircScrollTrigger");
	}
	else
	{
		return (ScrollTrigger)SYNSHM(synaptics)->circular_trigger;
	}
}

Button
g_synaptics_button_for_tap (GSynaptics *synaptics, TapType tap)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return None;

	if (priv->synclient)
	{
		return (Button)g_synaptics_get_value_from_synclient ("ButtonForTap");
	}
	else
	{
		return (Button)SYNSHM(synaptics)->tap_action[tap];
	}
}

gboolean
g_synaptics_are_fast_taps_enabled (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return FALSE;

	if (priv->synclient)
	{
		return (gboolean)g_synaptics_get_value_from_synclient ("FastTaps");
	}
	else
	{
		return SYNSHM(synaptics)->fast_taps;
	}
}

gint
g_synaptics_abs_coord_x (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return 0;

	if (priv->synclient)
	{
		return (gint)g_synaptics_get_value_from_synclient ("AbsCoordX");
	}
	else
	{
		return SYNSHM(synaptics)->x;
	}
}

gint
g_synaptics_abs_coord_y (GSynaptics *synaptics)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return 0;

	if (priv->synclient)
	{
		return (gint)g_synaptics_get_value_from_synclient ("AbsCoordY");
	}
	else
	{
		return SYNSHM(synaptics)->y;
	}
}


void
g_synaptics_set_enabled (GSynaptics *synaptics, gint enable)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient TouchpadOff=%d",
			       		   enable ? 1 : 0);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->touchpad_off = enable;
	}
}

void
g_synaptics_set_edges (GSynaptics *synaptics)
{
	if (!g_synaptics_is_valid(synaptics))
		return;
}

void
g_synaptics_set_tap_time (GSynaptics *synaptics, gint time)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient MaxTapTime=%d", time);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->tap_time = time;
	}
}

#if 0
void
g_synaptics_set_tap_action (GSynaptics *synaptics, TapType type, int action)
{
	if (!g_synaptics_is_valid(synaptics))
		return;

	SYNSHM(synaptics)->tap_action = action;
}
#endif

void
g_synaptics_set_sensitivity (GSynaptics *synaptics, gint value)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

  	if ( value < 0 && value > 4)
		return;
	
	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient FingerLow=%d FingerHigh=%d",
			       		   finger_low[value], finger_low[value] + 5);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
#if 0
	if ( !SynConfig::treatAsALPS())
	{
#endif
		SYNSHM(synaptics)->finger_low = finger_low[value];
		SYNSHM(synaptics)->finger_high = finger_low[value] + 5;
#if 0
	}
	else
	{
		SYNSHM(synaptics)->finger_low = finger_low[value] - 11;
		SYNSHM(synaptics)->finger_high = finger_low[value] - 10;
	}
#endif
	}
}

void
g_synaptics_set_horizontal_scroll_delta (GSynaptics *synaptics, gint delta)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient HorizScrollDelta=%d",
			       		   delta);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->scroll_dist_horiz = delta;
	}
}

void
g_synaptics_set_vertical_scroll_delta (GSynaptics *synaptics, gint delta)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient VertScrollDelta=%d",
			       		   delta);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->scroll_dist_vert = delta;
	}
}

void
g_synaptics_set_circular_scroll_enabled (GSynaptics *synaptics, gboolean enable)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient CircularScrolling=%d",
			       		   enable ? 1 : 0);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)-> circular_scrolling = enable;
	}
}

void
g_synaptics_set_edge_motion_enabled (GSynaptics *synaptics, gboolean enable)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient EdgeMotionUseAlways=%d",
			       		   enable ? 1 : 0);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->edge_motion_use_always = enable;
	}
}

void
g_synaptics_set_coasting_enabled (GSynaptics *synaptics, gboolean enable)
{
	gdouble thresh = coastingSpeedThreshold;
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (!enable)
		thresh = 0;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient CoastingSpeedThreashold=%f",
			       		   thresh);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->coasting_speed = thresh;
	}
}

void
g_synaptics_set_circular_scroll_delta (GSynaptics *synaptics, gint delta)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient CircScrollDelta=%f",
			       		   (gdouble)delta / 1000);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->scroll_dist_circ = ((double)delta) / 1000;
	}
}

void
g_synaptics_set_circular_scroll_trigger (GSynaptics *synaptics, ScrollTrigger t)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;

	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient CircScrollTrigger=%d",
			       		   t);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->circular_trigger = t;
	}
}

void
g_synaptics_set_button_for_tap (GSynaptics *synaptics, TapType tap, Button button)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;
}

void
g_synaptics_set_fast_taps (GSynaptics *synaptics, gboolean enable)
{
	GSynapticsPrivate *priv = G_SYNAPTICS_GET_PRIVATE (synaptics);
	if (!g_synaptics_is_valid(synaptics))
		return;
	
	if (priv->synclient)
	{
		gchar *command;
		command = g_strdup_printf ("synclient FastTaps=%d",
			       		   enable ? 1 : 0);
		g_spawn_command_line_async (command, NULL);
		g_free (command);
	}
	else
	{
		SYNSHM(synaptics)->fast_taps = enable;
	}
}
