/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2008  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2006-2007  Bastien Nocera <hadess@hadess.net>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "client.h"

#include "general.h"
#include "dialog.h"

static BluetoothClient *client;

static GtkTreeModel *adapter_model;

static DBusGConnection *connection;
static DBusGProxy *manager;

static GtkWidget *notebook = NULL;

static gboolean class_visibility = TRUE;

static void update_tab_label(GtkWidget *child, const char *name)
{
	GtkWidget *label;

	gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(notebook), child,
				name && name[0] != '\0' ? name : _("Adapter"));

	label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), child);
	gtk_label_set_max_width_chars(GTK_LABEL(label), 20);
	gtk_label_set_ellipsize(GTK_LABEL(label), PANGO_ELLIPSIZE_END);
}

static GList *adapter_list = NULL;

struct adapter_data {
	char *path;
	int attached;
	GtkWidget *child;
	GtkWidget *button_connect;
	GtkWidget *button_visible;
	GtkWidget *button_limited;
	GtkWidget *timeout_label;
	GtkWidget *timeout_scale;
	GtkWidget *entry;
	GtkWidget *class_box;
	GtkWidget *class_label;
	GtkWidget *class_combo;
	GtkWidget *button_delete;
	GtkWidget *button_trusted;
	GtkWidget *button_disconnect;
	GtkTreeSelection *selection;
	int name_changed;
	gboolean enable_limited;
};

static void adapter_free(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	adapter_list = g_list_remove(adapter_list, adapter);

	g_free(adapter->path);
	g_free(adapter);
}

static void adapter_disable(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;
	gint page;

	adapter->attached = 0;

	if (adapter->child != NULL) {
		page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
							adapter->child);

		gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);
	}

	adapter->child = NULL;
}

static gint adapter_compare(gconstpointer a, gconstpointer b)
{
	const struct adapter_data *adapter = a;
	const char *path = b;

	return g_ascii_strcasecmp(adapter->path, path);
}

static void mode_callback(GtkWidget *button, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	const char *mode;
	DBusGProxy *object;
	gboolean sensitive;

	if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button)) == FALSE)
		return;

	if (button == adapter->button_connect) {
		sensitive = FALSE;
		mode = "connectable";
	} else if (button == adapter->button_visible) {
		sensitive = TRUE;
		mode = "discoverable";
	} else if (button == adapter->button_limited) {
		sensitive = adapter->enable_limited;
		mode = "limited";
	} else
		return;

	object = dbus_g_proxy_new_for_name(connection, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetMode", NULL,
			G_TYPE_STRING, mode, G_TYPE_INVALID, G_TYPE_INVALID);

	gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_label), sensitive);
	gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_scale), sensitive);
}

static void scale_callback(GtkWidget *scale, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	gdouble value;
	guint32 timeout;
	DBusGProxy *object;

	value = gtk_range_get_value(GTK_RANGE(scale));

	if (value == 31)
		timeout = 0;
	else
		timeout = value * 60;

	object = dbus_g_proxy_new_for_name(connection, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetDiscoverableTimeout", NULL,
			G_TYPE_UINT, timeout, G_TYPE_INVALID, G_TYPE_INVALID);
}

static gchar *format_callback(GtkWidget *scale,
				gdouble value, gpointer user_data)
{
	if (value > 30)
		return g_strdup(_("never"));
	else if (value == 1)
		return g_strdup(_("1 minute"));
	else
		return g_strdup_printf(_("%g minutes"), value);
}

static void name_callback(GtkWidget *editable, gpointer user_data)
{
	struct adapter_data *adapter = user_data;

	adapter->name_changed = 1;
}

static gboolean focus_callback(GtkWidget *editable,
				GdkEventFocus *event, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	const gchar *text;
	DBusGProxy *object;

	if (!adapter->name_changed)
		return FALSE;

	text = gtk_entry_get_text(GTK_ENTRY(editable));

	object = dbus_g_proxy_new_for_name(connection, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetName", NULL,
			G_TYPE_STRING, text, G_TYPE_INVALID, G_TYPE_INVALID);

	adapter->name_changed = 0;

	return FALSE;
}

static void class_callback(GtkWidget *combobox, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	const char *minor;
	gint index;
	DBusGProxy *object;

	index = gtk_combo_box_get_active(GTK_COMBO_BOX(combobox));

	switch (index) {
	case 0:
		minor = "uncategorized";
		break;
	case 1:
		minor = "desktop";
		break;
	case 2:
		minor = "laptop";
		break;
	case 3:
		minor = "server";
		break;
	case 4:
		minor = "handheld";
		break;
	default:
		return;
	}

	object = dbus_g_proxy_new_for_name(connection, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "SetMinorClass", NULL,
			G_TYPE_STRING, minor, G_TYPE_INVALID, G_TYPE_INVALID);
}

static void update_buttons(struct adapter_data *adapter, gboolean bonded,
					gboolean trusted, gboolean connected)
{
	if (trusted == FALSE)
		gtk_button_set_label(GTK_BUTTON(adapter->button_trusted),
							_("Set _Trusted"));
	else
		gtk_button_set_label(GTK_BUTTON(adapter->button_trusted),
							_("Remove _Trust"));

	gtk_widget_set_sensitive(adapter->button_delete, bonded);

	gtk_widget_set_sensitive(adapter->button_disconnect, connected);
}

static void select_callback(GtkTreeSelection *selection, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean selected;
	gboolean bonded = FALSE, trusted = FALSE, connected = FALSE;

	selected = gtk_tree_selection_get_selected(selection, &model, &iter);

	if (selected == TRUE)
		gtk_tree_model_get(model, &iter, COLUMN_BONDED, &bonded,
					COLUMN_TRUSTED, &trusted,
					COLUMN_CONNECTED, &connected, -1);

	update_buttons(adapter, bonded, trusted, connected);

	if (selected == TRUE)
		gtk_widget_show(adapter->button_trusted);
	else
		gtk_widget_hide(adapter->button_trusted);
}

static void row_callback(GtkTreeModel *model, GtkTreePath  *path,
					GtkTreeIter *iter, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	gboolean bonded = FALSE, trusted = FALSE, connected = FALSE;

	if (gtk_tree_selection_iter_is_selected(adapter->selection,
							iter) == FALSE)
		return;

	gtk_tree_model_get(model, iter, COLUMN_BONDED, &bonded,
					COLUMN_TRUSTED, &trusted,
					COLUMN_CONNECTED, &connected, -1);

	update_buttons(adapter, bonded, trusted, connected);
}

static void delete_callback(GtkWidget *button, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *address;

	if (gtk_tree_selection_get_selected(adapter->selection,
						&model, &iter) == FALSE)
		return;

	gtk_tree_model_get(model, &iter, COLUMN_ADDRESS, &address, -1);

	if (show_confirm_dialog() == TRUE)
		bluetooth_client_remove_bonding(client, adapter->path, address);

	g_free(address);
}

static void trusted_callback(GtkWidget *button, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *address = NULL;
	gboolean trusted = FALSE;

	if (gtk_tree_selection_get_selected(adapter->selection,
						&model, &iter) == FALSE)
		return;

	gtk_tree_model_get(model, &iter, COLUMN_ADDRESS, &address,
					COLUMN_TRUSTED, &trusted, -1);

	if (trusted == FALSE)
		bluetooth_client_set_trusted(client, adapter->path, address);
	else
		bluetooth_client_remove_trust(client, adapter->path, address);

	g_free(address);
}

static void disconnect_callback(GtkWidget *button, gpointer user_data)
{
	struct adapter_data *adapter = user_data;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *address;

	if (gtk_tree_selection_get_selected(adapter->selection,
						&model, &iter) == FALSE)
		return;

	gtk_tree_model_get(model, &iter, COLUMN_ADDRESS, &address, -1);

	bluetooth_client_disconnect(client, adapter->path, address);

	g_free(address);

	gtk_widget_set_sensitive(button, FALSE);
}

static void type_to_icon(GtkTreeViewColumn *column, GtkCellRenderer *cell,
			GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	guint type;

	gtk_tree_model_get(model, iter, COLUMN_TYPE, &type, -1);

	switch (type) {
	case BLUETOOTH_TYPE_PHONE:
		g_object_set(cell, "icon-name", "stock_cell-phone", NULL);
		break;
	case BLUETOOTH_TYPE_MODEM:
		g_object_set(cell, "icon-name", "modem", NULL);
		break;
	case BLUETOOTH_TYPE_COMPUTER:
		g_object_set(cell, "icon-name", "computer", NULL);
		break;
	case BLUETOOTH_TYPE_NETWORK:
		g_object_set(cell, "icon-name", "network-wireless", NULL);
		break;
	case BLUETOOTH_TYPE_HEADSET:
		g_object_set(cell, "icon-name", "stock_headphones", NULL);
		break;
	case BLUETOOTH_TYPE_KEYBOARD:
		g_object_set(cell, "icon-name", "input-keyboard", NULL);
		break;
	case BLUETOOTH_TYPE_MOUSE:
		g_object_set(cell, "icon-name", "input-mouse", NULL);
		break;
	case BLUETOOTH_TYPE_CAMERA:
		g_object_set(cell, "icon-name", "camera-photo", NULL);
		break;
	case BLUETOOTH_TYPE_PRINTER:
		g_object_set(cell, "icon-name", "printer", NULL);
		break;
	default:
		g_object_set(cell, "icon-name", "bluetooth", NULL);
		break;
	}
}

static void name_to_text(GtkTreeViewColumn *column, GtkCellRenderer *cell,
			GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gchar *address, *name;

	gtk_tree_model_get(model, iter, COLUMN_ADDRESS, &address,
						COLUMN_NAME, &name, -1);

	if (name == NULL) {
		name = g_strdup(address);
		g_strdelimit(name, ":", '-');
	}

	g_object_set(cell, "text", name ? name : address, NULL);

	g_free(name);
	g_free(address);
}

static void connected_to_icon(GtkTreeViewColumn *column, GtkCellRenderer *cell,
			GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gboolean connected;

	gtk_tree_model_get(model, iter, COLUMN_CONNECTED, &connected, -1);

	if (connected == TRUE)
		g_object_set(cell, "icon-name", GTK_STOCK_CONNECT, NULL);

	g_object_set(cell, "visible", connected, NULL);
}

static void trusted_to_icon(GtkTreeViewColumn *column, GtkCellRenderer *cell,
			GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gboolean trusted;

	gtk_tree_model_get(model, iter, COLUMN_TRUSTED, &trusted, -1);

	if (trusted == TRUE)
		g_object_set(cell, "icon-name", GTK_STOCK_ABOUT, NULL);

	g_object_set(cell, "visible", trusted, NULL);
}

static void bonded_to_icon(GtkTreeViewColumn *column, GtkCellRenderer *cell,
			GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	gboolean bonded;

	gtk_tree_model_get(model, iter, COLUMN_BONDED, &bonded, -1);

	if (bonded == TRUE)
		g_object_set(cell, "icon-name",
				GTK_STOCK_DIALOG_AUTHENTICATION, NULL);

	g_object_set(cell, "visible", bonded, NULL);
}

GtkWidget *create_tree(GtkTreeModel *model, gboolean icons)
{
	GtkWidget *tree;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	tree = gtk_tree_view_new();
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(tree), FALSE);
	//gtk_tree_view_set_show_expanders(GTK_TREE_VIEW(tree), FALSE);
	gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(tree), TRUE);
	gtk_widget_set_size_request(tree, -1, 100);

	gtk_tree_view_set_model(GTK_TREE_VIEW(tree), model);

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title(column, "Device");
	gtk_tree_view_column_set_expand(GTK_TREE_VIEW_COLUMN(column), TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree), column);

	if (icons == TRUE) {
		renderer = gtk_cell_renderer_pixbuf_new();
		gtk_tree_view_column_set_spacing(column, 4);
		gtk_tree_view_column_pack_start(column, renderer, FALSE);
		gtk_tree_view_column_set_cell_data_func(column, renderer,
						type_to_icon, NULL, NULL);
	}

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func(column, renderer,
						name_to_text, NULL, NULL);

	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(tree), -1,
				"Connected", gtk_cell_renderer_pixbuf_new(),
					connected_to_icon, NULL, NULL);

	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(tree), -1,
				"Trusted", gtk_cell_renderer_pixbuf_new(),
					trusted_to_icon, NULL, NULL);

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_visible(column, FALSE);
	gtk_tree_view_insert_column_with_data_func(GTK_TREE_VIEW(tree), -1,
				"Bonded", gtk_cell_renderer_pixbuf_new(),
					bonded_to_icon, NULL, NULL);

	column = gtk_tree_view_column_new();
	gtk_tree_view_column_set_visible(column, FALSE);
	gtk_tree_view_insert_column(GTK_TREE_VIEW(tree), column, -1);
	gtk_tree_view_set_expander_column(GTK_TREE_VIEW(tree), column);

	return tree;
}

static void create_adapter(struct adapter_data *adapter)
{
	DBusGProxy *object;
	const char *name = NULL, *mode = NULL, *major = NULL, *minor = NULL;
	const guint32 timeout;
	const gchar **array = NULL;

	GtkWidget *mainbox;
	GtkWidget *vbox;
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *button;
	GtkWidget *scale;
	GtkWidget *entry;
	GtkWidget *combobox;
	GtkWidget *buttonbox;
	GtkWidget *scrolled;
	GtkWidget *tree;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GSList *group = NULL;
	gdouble value;
	gint index;

	object = dbus_g_proxy_new_for_name(connection, "org.bluez",
					adapter->path, "org.bluez.Adapter");

	dbus_g_proxy_call(object, "ListAvailableModes", NULL, G_TYPE_INVALID,
				G_TYPE_STRV, &array, G_TYPE_INVALID);

	adapter->enable_limited = FALSE;

	if (array) {
		while (*array) {
			if (g_ascii_strcasecmp(*array, "limited") == 0)
				adapter->enable_limited = TRUE;
			array++;
		}
	}

	dbus_g_proxy_call(object, "GetMode", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &mode, G_TYPE_INVALID);
	dbus_g_proxy_call(object, "GetDiscoverableTimeout", NULL, G_TYPE_INVALID,
				G_TYPE_UINT, &timeout, G_TYPE_INVALID);
	dbus_g_proxy_call(object, "GetName", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &name, G_TYPE_INVALID);
	dbus_g_proxy_call(object, "GetMajorClass", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &major, G_TYPE_INVALID);
	dbus_g_proxy_call(object, "GetMinorClass", NULL, G_TYPE_INVALID,
				G_TYPE_STRING, &minor, G_TYPE_INVALID);

	mainbox = gtk_vbox_new(FALSE, 18);
	gtk_container_set_border_width(GTK_CONTAINER(mainbox), 12);

	gtk_notebook_prepend_page(GTK_NOTEBOOK(notebook), mainbox, NULL);

	update_tab_label(mainbox, name);

	adapter->child = mainbox;

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_box_pack_start(GTK_BOX(mainbox), vbox, FALSE, FALSE, 0);

	label = create_label(_("Mode of operation"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	button = gtk_radio_button_new_with_label(group,
					_("Other devices can connect"));
	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	if (mode && g_ascii_strcasecmp(mode, "off") == 0)
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	if (mode && g_ascii_strcasecmp(mode, "connectable") == 0)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	adapter->button_connect = button;

	g_signal_connect(G_OBJECT(button), "toggled",
					G_CALLBACK(mode_callback), adapter);

	button = gtk_radio_button_new_with_label(group,
				_("Visible and connectable for other devices"));
	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	if (mode && g_ascii_strcasecmp(mode, "off") == 0)
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	if (mode && g_ascii_strcasecmp(mode, "discoverable") == 0)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	adapter->button_visible = button;

	g_signal_connect(G_OBJECT(button), "toggled",
					G_CALLBACK(mode_callback), adapter);

	button = gtk_radio_button_new_with_label(group,
				_("Limited discoverable and connectable"));
	group = gtk_radio_button_get_group(GTK_RADIO_BUTTON(button));
	gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);

	if (mode && g_ascii_strcasecmp(mode, "off") == 0)
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	if (mode && g_ascii_strcasecmp(mode, "limited") == 0)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	if (adapter->enable_limited == FALSE)
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);

	adapter->button_limited = button;

	g_signal_connect(G_OBJECT(button), "toggled",
					G_CALLBACK(mode_callback), adapter);

	label = gtk_label_new(_("Make adapter invisible after:"));
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	adapter->timeout_label = label;

	scale = gtk_hscale_new_with_range(1, 31, 1);
	gtk_scale_set_digits(GTK_SCALE(scale), 0);
	gtk_scale_set_value_pos(GTK_SCALE(scale), GTK_POS_BOTTOM);

	if (timeout == 0)
		value = 31;
	else if (timeout < 60)
		value = 1;
	else if (timeout > 60 * 30)
		value = 30;
	else
		value = timeout / 60;

	gtk_range_set_value(GTK_RANGE(scale), value);
	gtk_range_set_update_policy(GTK_RANGE(scale), GTK_UPDATE_DISCONTINUOUS);
	gtk_box_pack_start(GTK_BOX(vbox), scale, FALSE, FALSE, 0);

	adapter->timeout_scale = scale;

	g_signal_connect(G_OBJECT(scale), "value-changed",
					G_CALLBACK(scale_callback), adapter);
	g_signal_connect(G_OBJECT(scale), "format-value",
					G_CALLBACK(format_callback), NULL);

	if (mode && g_ascii_strcasecmp(mode, "discoverable") != 0 &&
				g_ascii_strcasecmp(mode, "limited") != 0) {
		gtk_widget_set_sensitive(GTK_WIDGET(label), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(scale), FALSE);
	}

	if (mode && g_ascii_strcasecmp(mode, "limited") == 0 &&
					adapter->enable_limited == FALSE) {
		gtk_widget_set_sensitive(GTK_WIDGET(label), FALSE);
		gtk_widget_set_sensitive(GTK_WIDGET(scale), FALSE);
	}

	hbox = gtk_hbox_new(FALSE, 12);
	gtk_box_pack_start(GTK_BOX(mainbox), hbox, FALSE, TRUE, 0);

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

	label = create_label(_("Adapter name"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	entry = gtk_entry_new();
	gtk_entry_set_max_length(GTK_ENTRY(entry), 248);
	gtk_widget_set_size_request(entry, 240, -1);
	gtk_box_pack_start(GTK_BOX(vbox), entry, TRUE, TRUE, 0);

	if (name != NULL)
		gtk_entry_set_text(GTK_ENTRY(entry), name);

	adapter->entry = entry;

	g_signal_connect(G_OBJECT(entry), "changed",
					G_CALLBACK(name_callback), adapter);
	g_signal_connect(G_OBJECT(entry), "focus-out-event",
					G_CALLBACK(focus_callback), adapter);

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_widget_set_no_show_all(vbox, TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

	adapter->class_box = vbox;

	label = create_label(_("Class of device"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	adapter->class_label = label;

	combobox = gtk_combo_box_new_text();
	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox),
						_("Unspecified"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox),
						_("Desktop workstation"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox),
						_("Laptop computer"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox),
						_("Server-class computer"));
	gtk_combo_box_append_text(GTK_COMBO_BOX(combobox),
						_("Handheld device"));
	gtk_box_pack_start(GTK_BOX(vbox), combobox, FALSE, FALSE, 0);

	if (major && minor && g_ascii_strcasecmp(major, "computer") == 0) {
		if (g_ascii_strcasecmp(minor, "uncategorized") == 0)
			index = 0;
		else if (g_ascii_strcasecmp(minor, "desktop") == 0)
			index = 1;
		else if (g_ascii_strcasecmp(minor, "laptop") == 0)
			index = 2;
		else if (g_ascii_strcasecmp(minor, "server") == 0)
			index = 3;
		else if (g_ascii_strcasecmp(minor, "handheld") == 0)
			index = 4;
		else
			index = -1;

		gtk_combo_box_set_active(GTK_COMBO_BOX(combobox), index);
	} else
		gtk_widget_set_sensitive(GTK_WIDGET(combobox), FALSE);

	adapter->class_combo = combobox;

	g_signal_connect(G_OBJECT(combobox), "changed",
					G_CALLBACK(class_callback), adapter);

	vbox = gtk_vbox_new(FALSE, 6);
	gtk_box_pack_start(GTK_BOX(mainbox), vbox, TRUE, TRUE, 0);

	label = create_label(_("Bonded devices"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrolled),
							GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(vbox), scrolled);

	model = bluetooth_client_get_model_bonded_list(client, adapter->path);
	g_signal_connect(G_OBJECT(model), "row-changed",
				G_CALLBACK(row_callback), adapter);
	tree = create_tree(model, FALSE);
	g_object_unref(model);

	selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(tree));
	gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

	g_signal_connect(G_OBJECT(selection), "changed",
				G_CALLBACK(select_callback), adapter);

	adapter->selection = selection;

	gtk_container_add(GTK_CONTAINER(scrolled), tree);

	buttonbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox),
						GTK_BUTTONBOX_START);
	gtk_box_set_spacing(GTK_BOX(buttonbox), 6);
	gtk_box_set_homogeneous(GTK_BOX(buttonbox), FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), buttonbox, FALSE, FALSE, 0);

	button = gtk_button_new_from_stock(GTK_STOCK_DELETE);
	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	gtk_container_add(GTK_CONTAINER(buttonbox), button);

	g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(delete_callback), adapter);

	adapter->button_delete = button;

	button = gtk_button_new_with_mnemonic(NULL);
	gtk_widget_set_no_show_all(button, TRUE);
	gtk_container_add(GTK_CONTAINER(buttonbox), button);

	g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(trusted_callback), adapter);

	adapter->button_trusted = button;

	button = gtk_button_new_from_stock(GTK_STOCK_DISCONNECT);
	gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
	gtk_container_add(GTK_CONTAINER(buttonbox), button);
	gtk_button_box_set_child_secondary(GTK_BUTTON_BOX(buttonbox),
								button, TRUE);

	g_signal_connect(G_OBJECT(button), "clicked",
				G_CALLBACK(disconnect_callback), adapter);

	adapter->button_disconnect = button;

	gtk_widget_show_all(mainbox);

	if (class_visibility == TRUE) {
		gtk_widget_show(adapter->class_combo);
		gtk_widget_show(adapter->class_label);
		gtk_widget_show(adapter->class_box);
	}
}

static void mode_changed(DBusGProxy *object,
				const char *mode, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		GtkWidget *button = NULL;
		gboolean sensitive;

		if (!adapter->attached)
			return;

		sensitive = g_ascii_strcasecmp(mode, "off") == 0 ? FALSE : TRUE;

		gtk_widget_set_sensitive(GTK_WIDGET(adapter->button_connect),
								sensitive);
		gtk_widget_set_sensitive(GTK_WIDGET(adapter->button_visible),
								sensitive);
		if (adapter->enable_limited == TRUE)
			gtk_widget_set_sensitive(GTK_WIDGET(adapter->button_limited), sensitive);
		else
			gtk_widget_set_sensitive(GTK_WIDGET(adapter->button_limited), FALSE);

		if (g_ascii_strcasecmp(mode, "connectable") == 0) {
			sensitive = FALSE;
			button = adapter->button_connect;
		} else if (g_ascii_strcasecmp(mode, "discoverable") == 0) {
			sensitive = TRUE;
			button = adapter->button_visible;
		} else if (g_ascii_strcasecmp(mode, "limited") == 0) {
			sensitive = adapter->enable_limited;
			button = adapter->button_limited;
		}

		gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_label),
								sensitive);
		gtk_widget_set_sensitive(GTK_WIDGET(adapter->timeout_scale),
								sensitive);

		if (!button)
			return;

		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	}
}

static void timeout_changed(DBusGProxy *object,
				const guint32 timeout, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		gdouble value;

		if (timeout == 0)
			value = 31;
		else if (timeout < 60)
			value = 1;
		else if (timeout > 60 * 30)
			value = 30;
		else
			value = timeout / 60;

		gtk_range_set_value(GTK_RANGE(adapter->timeout_scale), value);
	}
}

static void name_changed(DBusGProxy *object,
				const char *name, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		GtkWidget *child = adapter->child;

		if (!adapter->attached)
			return;

		update_tab_label(child, name);

		gtk_entry_set_text(GTK_ENTRY(adapter->entry),
						name ? name : "");

		adapter->name_changed = 0;
	}
}

static void minor_changed(DBusGProxy *object,
				const char *minor, gpointer user_data)
{
	GList *list;
	const char *path;

	path = dbus_g_proxy_get_path(object);

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		gint index;

		if (!adapter->attached)
			return;

		if (g_ascii_strcasecmp(minor, "uncategorized") == 0)
			index = 0;
		else if (g_ascii_strcasecmp(minor, "desktop") == 0)
			index = 1;
		else if (g_ascii_strcasecmp(minor, "laptop") == 0)
			index = 2;
		else
			index = -1;

		gtk_combo_box_set_active(GTK_COMBO_BOX(adapter->class_combo),
									index);
	}
}

static struct adapter_data *add_adapter(const char *path)
{
	DBusGProxy *object;
	GList *list;
	struct adapter_data *adapter;

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;

		adapter->attached = 1;

		create_adapter(adapter);

		return adapter;
	}

	adapter = g_try_malloc0(sizeof(*adapter));
	if (!adapter)
		return NULL;

	adapter->path = g_strdup(path);
	adapter->attached = 1;

	adapter_list = g_list_append(adapter_list, adapter);

	object = dbus_g_proxy_new_for_name(connection, "org.bluez",
						path, "org.bluez.Adapter");

	dbus_g_proxy_add_signal(object, "ModeChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "ModeChanged",
				G_CALLBACK(mode_changed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "DiscoverableTimeoutChanged",
					G_TYPE_UINT, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "DiscoverableTimeoutChanged",
				G_CALLBACK(timeout_changed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "NameChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "NameChanged",
				G_CALLBACK(name_changed), NULL, NULL);

	dbus_g_proxy_add_signal(object, "MinorClassChanged",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "MinorClassChanged",
				G_CALLBACK(minor_changed), NULL, NULL);

	create_adapter(adapter);

	return adapter;
}

static gboolean adapter_added(GtkTreeModel *model, GtkTreePath  *path,
					GtkTreeIter *iter, gpointer user_data)
{
	gchar *adapter_path;

	gtk_tree_model_get(model, iter, COLUMN_PATH, &adapter_path, -1);

	if (adapter_path == NULL)
		return FALSE;

	add_adapter(adapter_path);

	g_free(adapter_path);

	return FALSE;
}

static void adapter_removed(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	GList *list;

	list = g_list_find_custom(adapter_list, path, adapter_compare);
	if (list && list->data) {
		struct adapter_data *adapter = list->data;
		gint page;

		adapter->attached = 0;

		page = gtk_notebook_page_num(GTK_NOTEBOOK(notebook),
							adapter->child);

		gtk_notebook_remove_page(GTK_NOTEBOOK(notebook), page);

		adapter->child = NULL;
	}
}

void setup_adapter(DBusGConnection *conn)
{
	client = bluetooth_client_new();

	adapter_model = bluetooth_client_get_model_adapter_list(client);

	g_signal_connect(G_OBJECT(adapter_model), "row-inserted",
				G_CALLBACK(adapter_added), NULL);

	connection = dbus_g_connection_ref(conn);

	manager = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Manager");
	if (manager == NULL)
		return;

	dbus_g_proxy_add_signal(manager, "AdapterRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(manager, "AdapterRemoved",
				G_CALLBACK(adapter_removed), NULL, NULL);

	gtk_tree_model_foreach(adapter_model, adapter_added, NULL);
}

void assign_adapter(GtkWidget *widget)
{
	notebook = g_object_ref(widget);
}

void cleanup_adapter(void)
{
	g_list_foreach(adapter_list, adapter_free, NULL);

	if (manager != NULL)
		g_object_unref(manager);

	dbus_g_connection_unref(connection);

	g_object_unref(notebook);

	g_object_unref(adapter_model);

	g_object_unref(client);
}

void disable_adapter(void)
{
	g_list_foreach(adapter_list, adapter_disable, NULL);
}

void set_trusted(const char *adapter, const char *address)
{
	DBusGProxy *proxy;
	const char *path;

	if (manager == NULL)
		return;

	dbus_g_proxy_call(manager, "FindAdapter", NULL,
				G_TYPE_STRING, adapter, G_TYPE_INVALID,
				G_TYPE_STRING, &path, G_TYPE_INVALID);

	proxy = dbus_g_proxy_new_for_name(connection, "org.bluez",
					path, "org.bluez.Adapter");

	dbus_g_proxy_call(proxy, "SetTrusted", NULL,
				G_TYPE_STRING, address, G_TYPE_INVALID,
							G_TYPE_INVALID);

	g_object_unref(proxy);
}

static void class_hide(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	if (adapter->attached == 0)
		return;

	gtk_widget_hide(adapter->class_box);
	gtk_widget_hide(adapter->class_label);
	gtk_widget_hide(adapter->class_combo);
}

void hide_class_widget(void)
{
	class_visibility = FALSE;

	g_list_foreach(adapter_list, class_hide, NULL);
}

static void class_show(gpointer data, gpointer user_data)
{
	struct adapter_data *adapter = data;

	if (adapter->attached == 0)
		return;

	gtk_widget_show(adapter->class_combo);
	gtk_widget_show(adapter->class_label);
	gtk_widget_show(adapter->class_box);
}

void show_class_widget(void)
{
	class_visibility = TRUE;

	g_list_foreach(adapter_list, class_show, NULL);
}
