/*
 * Copyright (C) 2007,2008 daniel g. siegel <dgsiegel@gmail.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#include <cheese-config.h>
#endif

#include <glib.h>
#include <stdlib.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "cheese-gconf.h"

#define CHEESE_GCONF_PREFIX   "/apps/cheese"

G_DEFINE_TYPE (CheeseGConf, cheese_gconf, G_TYPE_OBJECT)

#define CHEESE_GCONF_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), CHEESE_TYPE_GCONF, CheeseGConfPrivate))

typedef struct
{
  GConfClient *client;

} CheeseGConfPrivate;

static void
cheese_gconf_get_property (GObject *object, guint prop_id, GValue *value,
                           GParamSpec *pspec)
{
  CheeseGConf *self;
  self = CHEESE_GCONF (object);
  CheeseGConfPrivate *priv = CHEESE_GCONF_GET_PRIVATE (self);

  char *effects;
  GSList *list, *tmp;

  switch (prop_id) 
  {
    case GCONF_PROP_COUNTDOWN:
      g_value_set_boolean (value, gconf_client_get_bool (priv->client,
                                                         CHEESE_GCONF_PREFIX "/countdown",
                                                         NULL));
      break;
    case GCONF_PROP_WEBCAM:
      g_value_set_string (value, gconf_client_get_string (priv->client,
                                                          CHEESE_GCONF_PREFIX "/webcam",
                                                          NULL));
      break;
    case GCONF_PROP_SELECTED_EFFECTS:
      effects = NULL;
      list = gconf_client_get_list (priv->client,
                                    CHEESE_GCONF_PREFIX "/selected_effects",
                                    GCONF_VALUE_STRING,
                                    NULL);
      tmp = list;
      while (tmp != NULL)
      {
        if (effects == NULL)
          effects = tmp->data;
        else
          effects = g_strjoin (",", effects, tmp->data, NULL);

        tmp = g_slist_next(tmp);
      }
      g_value_set_string (value, effects);

      g_slist_free (list);
      g_slist_free (tmp);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
cheese_gconf_set_property (GObject *object, guint prop_id, const GValue *value,
                           GParamSpec *pspec)
{
  CheeseGConf *self;
  self = CHEESE_GCONF (object);
  CheeseGConfPrivate *priv = CHEESE_GCONF_GET_PRIVATE (self);

  gchar **effects = NULL;
  GSList *list = NULL;
  int i;

  switch (prop_id) 
  {
    case GCONF_PROP_COUNTDOWN:
      gconf_client_set_bool (priv->client,
                             CHEESE_GCONF_PREFIX "/countdown",
                             g_value_get_boolean (value),
                             NULL);
      break;
    case GCONF_PROP_WEBCAM:
      gconf_client_set_string (priv->client,
                               CHEESE_GCONF_PREFIX "/webcam",
                               g_value_get_string (value),
                               NULL);
      break;
    case GCONF_PROP_SELECTED_EFFECTS:

      if (g_value_get_string (value) == NULL) {
        list = NULL;
      }
      else
      {
        effects = g_strsplit(g_value_get_string (value), ",", 12);
        for (i = 0; effects[i] != NULL; i++)
        {
          list = g_slist_append (list, effects[i]);
        }
      }
      gconf_client_set_list (priv->client,
                               CHEESE_GCONF_PREFIX "/selected_effects",
                               GCONF_VALUE_STRING,
                               list,
                               NULL);
      g_slist_free (list);
      g_strfreev (effects);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
cheese_gconf_finalize (GObject *object)
{
  CheeseGConf *gconf;

  gconf = CHEESE_GCONF (object);
  CheeseGConfPrivate *priv = CHEESE_GCONF_GET_PRIVATE (gconf);  

  g_object_unref (priv->client);
  G_OBJECT_CLASS (cheese_gconf_parent_class)->finalize (object);
}

static void
cheese_gconf_class_init (CheeseGConfClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = cheese_gconf_finalize;

  object_class->get_property = cheese_gconf_get_property;
  object_class->set_property = cheese_gconf_set_property;

  g_object_class_install_property (object_class, GCONF_PROP_COUNTDOWN,
                                   g_param_spec_boolean ("gconf_prop_countdown",
                                                         NULL,
                                                         NULL,
                                                         FALSE,
                                                         G_PARAM_READWRITE));
  g_object_class_install_property (object_class, GCONF_PROP_WEBCAM,
                                   g_param_spec_string ("gconf_prop_webcam",
                                                        NULL,
                                                        NULL,
                                                        "",
                                                        G_PARAM_READWRITE));
  g_object_class_install_property (object_class, GCONF_PROP_SELECTED_EFFECTS,
                                   g_param_spec_string ("gconf_prop_selected_effects",
                                                        NULL,
                                                        NULL,
                                                        "",
                                                        G_PARAM_READWRITE));

  g_type_class_add_private (klass, sizeof (CheeseGConfPrivate));
}

static void
cheese_gconf_init (CheeseGConf *gconf)
{
  CheeseGConfPrivate* priv = CHEESE_GCONF_GET_PRIVATE (gconf);
  priv->client = gconf_client_get_default();
}

CheeseGConf * 
cheese_gconf_new ()
{
  CheeseGConf *gconf;

  gconf = g_object_new (CHEESE_TYPE_GCONF, NULL);  
  return gconf;
}
