# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>
#          Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core import common
from elisa.core.media_uri import MediaUri

from elisa.core.utils.i18n import install_translation

# models
from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.database.models import *
from elisa.plugins.poblesec.link import Link

# controllers and their default widgets
from elisa.plugins.poblesec.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.artists_browser import Grid, Coverflow, \
    VerticalWithPreview, TracksVerticalWithPreview, \
    AlbumsController, ArtistsController, TracksController

from elisa.plugins.poblesec.section import SectionMenuController
from elisa.plugins.poblesec.list_preview import PreviewListController
from elisa.plugins.poblesec.coverflow import CoverflowListController
from elisa.plugins.poblesec.grid import GridController

from elisa.plugins.poblesec.artists_browser import album_cover_retriever

from twisted.internet import defer, task

from storm.expr import Not

_ = install_translation('database')

import datetime


def music_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/music_library'
    link.label = _('Music Library')
    link.icon = 'elisa.plugins.poblesec.music_library'
    controller.model.append(link)

    return defer.succeed(None)


def music_lib_artists_decorator(controller):
    artists = Link()
    artists.controller_path = '/poblesec/database/artists'
    artists.label = _('Artists')
    artists.icon = 'elisa.plugins.poblesec.by_artist'
    controller.model.append(artists)
    return defer.succeed(None)


def music_lib_albums_decorator(controller):
    albums = Link()
    albums.controller_path = '/poblesec/database/albums'
    albums.label = _('Albums')
    albums.icon = 'elisa.plugins.poblesec.by_album'
    controller.model.append(albums)
    return defer.succeed(None)


def music_lib_tracks_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/tracks'
    tracks.label = _('Tracks')
    tracks.icon = 'elisa.plugins.poblesec.by_track'
    controller.model.append(tracks)
    return defer.succeed(None)


def music_lib_genres_decorator(controller):
    genres = Link()
    genres.controller_path = '/poblesec/database/genres'
    genres.label = _('Genres')
    genres.icon = 'elisa.plugins.poblesec.by_genre'
    controller.model.append(genres)
    return defer.succeed(None)


def music_lib_decades_decorator(controller):
    decades = Link()
    decades.controller_path = '/poblesec/database/decades'
    decades.label = _('Decades')
    decades.icon = 'elisa.plugins.poblesec.by_decade'
    controller.model.append(decades)
    return defer.succeed(None)


class AlbumsDbController(AlbumsController):
    tracks_controller_path = '/poblesec/database/tracks'

    def _start(self, result, artist):
        self.artist = artist

        if artist is None:
            # show all albums
            dfr = common.application.store.find(MusicAlbum)

            def got_albums(albums):
                self.model.extend(albums)
                return self

            def sort_albums(result_set):
                result_set.order_by(MusicAlbum.name)
                return result_set.all()

            dfr.addCallback(sort_albums)
            dfr.addCallback(got_albums)

        else:
            # show albums of given artist
            dfr = artist.tracks.find()


            def add_albums(albums):
                self.model.extend(albums)
                return self

            def got_albums(result):
                result.order_by(MusicAlbum.name)
                dfr = result.all()
                dfr.addCallback(add_albums)
                return dfr

            def got_tracks(album_names):
                dfr = common.application.store.find(MusicAlbum,
                        MusicAlbum.name.is_in(album_names))
                dfr.addCallback(got_albums)
                return dfr

            dfr.addCallback(lambda rs: rs.values(MusicTrack.album_name))
            dfr.addCallback(got_tracks)
        return dfr

class ArtistsDbController(ArtistsController):
    albums_controller_path = '/poblesec/database/albums'

    def _start(self, result):
        dfr = common.application.store.find(Artist)

        def got_artists(artists):
            artists = sorted(artists, key=lambda a: a.name.lower())
            self.model.extend(artists)
            return self

        def sort_artists(result_set):
            result_set.order_by(Artist.name)
            return result_set.all()

        dfr.addCallback(sort_artists)
        dfr.addCallback(got_artists)
        return dfr

class TracksDbController(TracksController):


    def _get_title(self, item):
        title = "%s" % item.title

        if self.album and item.track_number:
            # showing one album: if possible show the track number
            num = str(item.track_number).zfill(2)
            title = "%s. %s" % (num, title)

        return title

    def _get_path_for_item(self, item):
        if self.album:
            return self.album.cover_uri
        if hasattr(item, 'cover_uri'):
            return item.cover_uri

    def _load_sublabel(self, item, widget):

        def values(result):
            return result.values(Artist.name)

        def set_distinct(result_set):
            result_set.config(distinct=True)
            return result_set

        def set_values(result_list):
            result_list_len = len(result_list)

            if result_list_len == 0:
                # nothing found
                return
            elif result_list_len > 1:
                result = ','.join(result_list)
            else:
                result = result_list[0]


            widget.sublabel.label = result

        dfr = item.artists.find()
        dfr.addCallback(set_distinct)
        dfr.addCallback(values)
        dfr.addCallback(set_values)

    def _retrieve(self, widget, item):
        if not album_cover_retriever:
            return

        def got_album(album, item, widget):
            if not album:
                return

            if album.cover_uri != None:
                # already a cover found
                item.cover_uri = album.cover_uri
                self._update_item(item, item)
                return
    
            dfr = album_cover_retriever.get_image(album)
            if dfr:
                dfr.addCallback(self._update_item, item)
                widget.dfr = dfr
            return dfr

        dfr = item.album
        dfr.addCallback(got_album, item, widget)
        
    def _start(self, result, album):
        self.album = album

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def sort_tracks_number_title(result_set):
            result_set.order_by(MusicTrack.track_number, MusicTrack.title)
            return result_set.all()

        def sort_tracks_title(result_set):
            result_set.order_by(MusicTrack.title)
            return result_set.all()

        store = common.application.store

        if album:
            dfr = store.find(MusicTrack, MusicTrack.album_name == album.name)
            dfr.addCallback(sort_tracks_number_title)
        else:
            dfr = store.find(MusicTrack, MusicTrack.title != None)
            dfr.addCallback(sort_tracks_title)

        dfr.addCallback(got_tracks)
        return dfr

    def play_audio(self, model):
        controllers = self.frontend.retrieve_controllers('/poblesec/music_player')
        player_controller = controllers[0]

        # enqueue and play the clicked item
        playable_model = PlayableModel()
        playable_model.uri = MediaUri('file://' + model.file_path)
        playable_model.title = self._get_title(model)
        player_controller.player.play_model(playable_model)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_music_player()
        self.stop_loading_animation()

        # enqueue all the following items of the container
        index = self.model.index(model)
        for item in self.model[index+1:]:
            if not isinstance(item, MusicTrack):
                continue
            playable_model = PlayableModel()
            playable_model.uri = MediaUri('file://' + item.file_path)
            player_controller.player.enqueue_to_playlist(playable_model)


# the most important genres!
GENRES = [u"Acid Jazz", u"Alternative", u"Alternative & Punk",
          u"Alternative Pop/Rock", u"Alternative Rock", u"AlternRock",
          u"Ambient", u"Blues", u"Blues/R&B", u"Books & Spoken",
          u"Children's Music", u"Classic Rock", u"Classical", u"Comedy",
          u"Dance", u"Easy Listening", u"Electronic", u"Electronica & Dance",
          u"Electronice/Dance", u"Entertainment", u"Folk", u"Funk",
          u"Games", u"Garage/Punk Rock Revival", u"General Alternative",
          u"Hard Rock", u"Hip Hop", u"Hip Hop/Rap", u"Hip-Hop",
          u"Hip-Hop/Rap", u"Holiday", u"House", u"Indi", u"Industrial",
          u"Jazz", u"Latin", u"Metal", u"Music Videos", u"New Age",
          u"Other", u"Pop", u"Punk", u"R&B", u"R&B/Soul", u"Rap", u"Raggae",
          u"Religious", u"Rock", u"Rock & Roll", u"Soundtrack", u"Techno",
          u"Trance", u"World"]


class GenresController(SectionMenuController, PreviewListController):
    # FIXME: This kind of specialized controller (vertical list) doesn't allow
    #        view mode switching.

    def initialize(self, all_genres=False):
        dfr = super(GenresController, self).initialize()

        dfr.addCallback(self._load, all_genres)
        return dfr

    def _load(self, result, all_genres):

        def got_genres(genres, all_genres):
            def iterate(genres, models, all_genres):
                if not all_genres:
                    all = Link()
                    all.controller_path = '/poblesec/database/genres'
                    all.label = _("All genres")
                    all.controller_args = {'all_genres': True}
                    all.icon = 'elisa.plugins.poblesec.by_genre' 
                    models.append(all)

                for genre in genres:
                    link = Link()
                    link.controller_path = '/poblesec/database/genre'
                    link.label = genre
                    link.controller_args = {'genre' : genre}
                    link.icon = 'elisa.plugins.poblesec.by_genre'
                    models.append(link)
                    yield link

            def iterate_done(result, models):
                self.model.extend(models)
                return self

            models = []
            dfr = task.coiterate(iterate(genres, models, all_genres))
            dfr.addCallback(iterate_done, models)
            return dfr

        def sort_by_title(result_set):
            result_set.order_by(MusicTrack.genre)
            return result_set

        def set_distinct(result_set):
            result_set.config(distinct=True)
            return result_set

        def get_genres(result_set):
            return result_set.values(MusicTrack.genre)

        store = common.application.store

        if all_genres:
            dfr = store.find(MusicTrack, MusicTrack.genre != None)
        else:
            dfr = store.find(MusicTrack, MusicTrack.genre.is_in(GENRES))

        dfr.addCallback(set_distinct)
        dfr.addCallback(sort_by_title)
        dfr.addCallback(get_genres)
        dfr.addCallback(got_genres, all_genres)

        return dfr


class GenreController(TracksDbController):

    # FIXME: shouldn't we display a list of albums rather than a quite useless
    #        list of tracks?

    def initialize(self, genre=None):
        dfr = super(GenreController, self).initialize()

        dfr.addCallback(self._load, genre)
        return dfr

    def _start(self, *args):
        return

    def _load(self, result, genre):

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def sort_by_title(result_set):
            result_set.order_by(MusicTrack.genre)
            return result_set.all()

        store = common.application.store
        if genre:
            dfr = store.find(MusicTrack, MusicTrack.genre == unicode(genre))
        else:
            dfr = store.find(MusicTrack, Not(MusicTrack.genre.is_in(GENRES)))

        dfr.addCallback(sort_by_title)
        dfr.addCallback(got_tracks)

        return dfr

    def _get_title(self, item):
        return item.title


class DecadesController(SectionMenuController):

    def initialize(self):
        dfr = super(DecadesController, self).initialize()

        dfr.addCallback(self._load)
        return dfr

    def _load(self, result):

        def iterate(models):
            # FIXME: What about other decades (10's, 20's, 30's, 40's, ...)?
            #        And what about other centuries?
            for decade in (50, 60, 70, 80, 90):
                link = Link()
                link.controller_path = '/poblesec/database/time'
                link.label = "%ss" % decade
                link.controller_args = {'begin' : datetime.datetime(1900+decade,1,1), 
                               'end' : datetime.datetime(1909+decade,12,31)}
                link.icon = 'elisa.plugins.poblesec.by_decade'
                models.append(link)
                yield link

        def iterate_done(result, models):
            self.model.extend(models)
            return self

        models = []
        dfr = task.coiterate(iterate(models))
        dfr.addCallback(iterate_done, models)
        return dfr

class TimeController(AlbumsDbController):

    def initialize(self, begin=None, end=None):
        dfr = super(TimeController, self).initialize()

        dfr.addCallback(self._load, begin, end)
        return dfr

    def _start(self, *args):
        return

    def _load(self, result, begin, end):

        def got_albums(albums):
            if not albums:
                # FIXME: We should never raise generic exceptions, let's find
                #        a suitable specific exception for this case.
                raise Exception("Missing Data")
            self.model.extend(albums)
            return self

        def sort_by_title(result_set):
            result_set.order_by(MusicAlbum.name)
            return result_set.all()

        store = common.application.store
        dfr = store.find(MusicAlbum, (MusicAlbum.release_date >= begin) & \
                                     (MusicAlbum.release_date <= end))

        dfr.addCallback(sort_by_title)
        dfr.addCallback(got_albums)

        return dfr

# Artists
class ArtistsDbVerticalWithPreview(ArtistsDbController, VerticalWithPreview):
    pass

class ArtistsDbCoverflow(ArtistsDbController, Coverflow):
    pass

class ArtistsDbGrid(ArtistsDbController, Grid):
    pass

class ArtistsListSwitcher(ListSwitcherController):
    modes = [ArtistsDbVerticalWithPreview,
             ArtistsDbCoverflow,
             ArtistsDbGrid]
    default_mode = ArtistsDbVerticalWithPreview


# Albums
class AlbumsDbVerticalWithPreview(AlbumsDbController, VerticalWithPreview):
    pass

class AlbumsDbCoverflow(AlbumsDbController, Coverflow):
    pass

class AlbumsDbGrid(AlbumsDbController, Grid):
    pass

class AlbumsListSwitcher(ListSwitcherController):
    modes = [AlbumsDbVerticalWithPreview,
             AlbumsDbCoverflow,
             AlbumsDbGrid]
    default_mode = AlbumsDbCoverflow


# Tracks
class TracksDbVerticalWithPreview(TracksDbController, TracksVerticalWithPreview):
    pass

class TracksDbCoverflow(TracksDbController, Coverflow):
    pass

class TracksDbGrid(TracksDbController, Grid):
    pass

class TracksListSwitcher(ListSwitcherController):
    modes = [TracksDbVerticalWithPreview,
             TracksDbCoverflow,
             TracksDbGrid]
    default_mode = TracksDbVerticalWithPreview


# Time
class TimeVerticalWithPreview(TimeController, VerticalWithPreview):
    pass

class TimeCoverflow(TimeController, Coverflow):
    pass

class TimeGrid(TimeController, Grid):
    pass

class TimeListSwitcher(ListSwitcherController):
    modes = [TimeVerticalWithPreview,
             TimeCoverflow,
             TimeGrid]
    default_mode = TimeCoverflow


# Genre
class GenreVerticalWithPreview(GenreController, TracksVerticalWithPreview):
    pass

class GenreCoverflow(GenreController, Coverflow):
    pass

class GenreGrid(GenreController, Grid):
    pass

class GenreListSwitcher(ListSwitcherController):
    modes = [GenreVerticalWithPreview,
             GenreCoverflow,
             GenreGrid]
    default_mode = GenreVerticalWithPreview
