# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core import common

from elisa.plugins.poblesec.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.artists_browser import TracksController, \
    Grid, Coverflow, TracksVerticalWithPreview

from elisa.plugins.database.controller import TracksDbController
from elisa.plugins.poblesec.link import Link

from elisa.plugins.database.models import MusicTrack, File
from storm.expr import Desc

from twisted.internet import defer

# i18n
from elisa.core.utils.i18n import install_translation
_ = install_translation('database')


def most_played_tracks_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/played'
    link.label = _('Most Played Tracks')
    link.controller_args = {'filter' : MusicTrack, 'limit': 20,}
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)


def last_played_tracks_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/played'
    link.label = _('Recently Played Tracks')
    link.controller_args = {'filter' : MusicTrack, 'limit':20, 
                            'order_by' : 'last_played'}
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)


def recently_added_tracks_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/added'
    link.label = _('Recently Added Tracks')
    link.controller_args = {'filter' : MusicTrack, 'limit' : 20}
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)

def playlists_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/playlists'
    link.label = _('Playlists')
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)


class PlayedController(TracksDbController):

    def initialize(self, filter=None, order_by=None, limit=10):
        dfr = super(PlayedController, self).initialize()

        dfr.addCallback(self._load, filter, order_by, limit)
        return dfr

    def _start(self, *args):
        return

    def _load(self, result, filter, order_by, limit):
        self.filter = filter

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def get_tracks(result_set):
            return result_set.all()

        def set_limit(result_set, limit):
            result_set.config(limit=limit)
            return result_set

        def sort_by_count(result_set):
            result_set.order_by(Desc(File.playcount))
            return result_set

        def sort_by_date(result_set):
            result_set.order_by(Desc(File.last_played))
            return result_set

        store = common.application.store

        dfr = store.find(filter, filter.file_path == File.path, \
                File.playcount != 0)

        if order_by == "last_played":
            dfr.addCallback(sort_by_date)
        else:
            dfr.addCallback(sort_by_count)

        dfr.addCallback(set_limit, limit)
        dfr.addCallback(get_tracks)
        dfr.addCallback(got_tracks)
        return dfr


class AddedController(TracksDbController):

    def initialize(self, filter=None, limit=10):
        dfr = super(AddedController, self).initialize()

        dfr.addCallback(self._load, filter, limit)
        return dfr

    def _start(self, *args):
        return

    def _load(self, result, filter, limit):
        self.filter = filter

        def got_tracks(tracks):
            self.model.extend(tracks)
            return self

        def get_tracks(result_set):
            return result_set.all()

        def set_limit(result_set, limit):
            result_set.config(limit=limit)
            return result_set

        def sort_by_added(result_set):
            # this is not exactly the time it was added but last changed
            result_set.order_by(Desc(File.modification_time))
            return result_set

        store = common.application.store

        dfr = store.find(filter, filter.file_path == File.path)

        dfr.addCallback(sort_by_added)

        dfr.addCallback(set_limit, limit)
        dfr.addCallback(get_tracks)
        dfr.addCallback(got_tracks)
        return dfr


# Added 
class AddedVerticalWithPreview(AddedController, TracksVerticalWithPreview):
    pass

class AddedCoverflow(AddedController, Coverflow):
    pass

class AddedGrid(AddedController, Grid):
    pass

class AddedListSwitcher(ListSwitcherController):
    modes = [AddedVerticalWithPreview,
             AddedCoverflow,
             AddedGrid]
    default_mode = AddedVerticalWithPreview


# Played 
class PlayedVerticalWithPreview(PlayedController, TracksVerticalWithPreview):
    pass

class PlayedCoverflow(PlayedController, Coverflow):
    pass

class PlayedGrid(PlayedController, Grid):
    pass

class PlayedListSwitcher(ListSwitcherController):
    modes = [PlayedVerticalWithPreview,
             PlayedCoverflow,
             PlayedGrid]
    default_mode = PlayedVerticalWithPreview
