# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
models the media scanner is returning
"""

from elisa.core.components.model import Model

from twisted.internet.defer import Deferred

from Queue import Queue

class Statistic(Model):
    """
    The statistics and states of the media scanner. This is for others than
    the media scanner read-only.

    @ivar running:              is the media scanner currently running
    @type running:              bool
    @ivar currently_scanning:   the scanner is currently scanning
    @type currently_scanning:   L{ScanResource}
    @ivar scanned:              all the already scanned resources (since elisa
                                start)
    @type scanned:              C{list} of L{ScanResource}s
    @ivar queue:                queue of pending resources to scan
    @type queue:                C{Queue.Queue} of L{ScanResource}s

    """
    def __init__(self):
        super(Statistic, self).__init__()
        self.running = False
        self.currently_scanning = None
        self.scanned = []
        self.queue = Queue()

class ScanResource(Model):
    """
    A resource to scan.
    @ivar root_uri:         the root to scan
    @type root_uri:         L{elisa.core.media_uri.MediaUri}
    @ivar state:            the state it is currently in. See the C{SCANNING*}
                            constants in the class descrition
    @type state:            int

    @ivar files_total:      the total number of files we found in this resource
    @type files_total:      int
    @ivar files_scanned:    number of files we have scanned the metadata
    @type files_scanned:    int
    @ivar files_failed:     the files that the metadata getting failed and the
                            reason (as a Failure instance)
    @type files_failed:     C{list} of C{tuple}

    @ivar last_scan:        when the resource was scanned the last time
    @type last_scan:        float
    @ivar duration:         the time it took to scan this resource
    @type duration:         float

    @ivar defer:            triggered when the scanning is done
    @type defer:            L{twisted.internet.defer.Deferred}
    """
    SCANNING_NONE = 0
    SCANNING_FS = 1
    SCANNING_METADATA = 2
    SCANNING_DONE = 3

    def __init__(self, root=None):
        super(ScanResource, self).__init__()
        self.root_uri = root
        self.state = self.SCANNING_NONE
        self.pending_dfrs = []

        self.files_total = 0
        self.files_scanned = 0
        self.files_failed = []

        self.defer = Deferred()

        self.last_scan = 0 
        self.duration = 0
