# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


from elisa.core import common
from elisa.core import media_uri

from elisa.plugins.base.models.audio import TrackModel
from elisa.plugins.base.models.media import PlayableModel

from elisa.plugins.poblesec.link import Link
from elisa.plugins.poblesec.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.artists_browser import AlbumsController
from elisa.plugins.poblesec.artists_browser import ArtistsController
from elisa.plugins.poblesec.artists_browser import TracksController
from elisa.plugins.poblesec.artists_browser import Grid, Coverflow, VerticalWithPreview

from twisted.internet import task

class AlbumsIpodController(AlbumsController):
    tracks_controller_path = '/poblesec/music/ipod/tracks'

    def _get_path_for_item(self, item):
        cover_uri = getattr(item, 'cover_uri', None)
        return cover_uri

    def _start(self, result, artist):
        self.artist = artist
        resource, dfr = common.application.resource_manager.get(artist.elisa_uri)

        def got_albums(albums_list_model):
            albums = albums_list_model.albums

            def fix_models():
                for album_model in albums:
                    album_model.name = album_model.album
                    uri = media_uri.MediaUri(self.artist.elisa_uri)
                    uri.set_param('action', 'tracks')
                    uri.set_param('search', 'songs')
                    uri.set_param('album', album_model.name)
                    album_model.elisa_uri = uri
                    yield album_model

            def models_fixed(r, albums):
                albums = sorted(albums, key=lambda a: a.name.lower())
                self.model.extend(albums)
                return self

            d = task.coiterate(fix_models())
            d.addCallback(models_fixed, albums)
            return d

        dfr.addCallback(got_albums)
        return dfr

class ArtistsIpodController(ArtistsController):
    albums_controller_path = '/poblesec/music/ipod/albums'

    def _get_path_for_item(self, item):
        image_uri = getattr(item, 'image_uri', None)
        return image_uri

    def _start(self, result):
        resource, dfr = common.application.resource_manager.get(self.uri)

        def got_artists(model):
            artists = model.artists

            def fix_models():
                for artist_model in artists:
                    uri = media_uri.MediaUri(self.uri)
                    uri.set_param('action', 'tracks')
                    uri.set_param('search', 'albums')
                    uri.set_param('artist', artist_model.name)
                    artist_model.elisa_uri = uri
                    yield artist_model

            def models_fixed(r, artists):
                artists = sorted(artists, key=lambda a: a.name.lower())
                self.model.extend(artists)
                return self

            d = task.coiterate(fix_models())
            d.addCallback(models_fixed, artists)
            return d

        dfr.addCallback(got_artists)
        return dfr

class TracksIpodController(TracksController):

    def _start(self, result, album):
        self.album = album

        def got_tracks(tracks_list_model):
            tracks = sorted(tracks_list_model.songs, key=lambda i: i.track_number)
            self.model.extend(tracks)
            return self

        resource, dfr = common.application.resource_manager.get(album.elisa_uri)
        dfr.addCallback(got_tracks)
        return dfr

    def play_audio(self, model):
        controllers = self.frontend.retrieve_controllers('/poblesec/music_player')
        player_controller = controllers[0]

        # enqueue and play the clicked item
        playable_model = PlayableModel()
        playable_model.uri = model.playable_uri
        playable_model.title = self._get_title(model)
        player_controller.player.play_model(playable_model)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_music_player()

        # enqueue all the following items of the container
        index = self.model.index(model)
        for item in self.model[index+1:]:
            if not isinstance(item, TrackModel):
                continue
            playable_model = PlayableModel()
            playable_model.uri = item.playable_uri
            playable_model.title = self._get_title(item)
            player_controller.player.enqueue_to_playlist(playable_model)

# artists
class ArtistsVerticalWithPreview(ArtistsIpodController, VerticalWithPreview):
    pass
class ArtistsCoverflow(ArtistsIpodController, Coverflow):
    pass
class ArtistsGrid(ArtistsIpodController, Grid):
    pass

class ArtistsListSwitcher(ListSwitcherController):
    modes = [ArtistsVerticalWithPreview, ArtistsCoverflow, ArtistsGrid]

class ArtistsListSwitcherVertical(ArtistsListSwitcher):
    default_mode = ArtistsVerticalWithPreview

class ArtistsListSwitcherCoverflow(ArtistsListSwitcher):
    default_mode = ArtistsCoverflow

class ArtistsListSwitcherGrid(ArtistsListSwitcher):
    default_mode = ArtistsGrid

# albums
class AlbumsVerticalWithPreview(AlbumsIpodController, VerticalWithPreview):
    pass
class AlbumsCoverflow(AlbumsIpodController, Coverflow):
    pass
class AlbumsGrid(AlbumsIpodController, Grid):
    pass

class AlbumsListSwitcher(ListSwitcherController):
    modes = [AlbumsVerticalWithPreview, AlbumsCoverflow, AlbumsGrid]

class AlbumsListSwitcherVertical(AlbumsListSwitcher):
    default_mode = AlbumsVerticalWithPreview

class AlbumsListSwitcherCoverflow(AlbumsListSwitcher):
    default_mode = AlbumsCoverflow

class AlbumsListSwitcherGrid(AlbumsListSwitcher):
    default_mode = AlbumsGrid

# tracks
class TracksVerticalWithPreview(TracksIpodController, VerticalWithPreview):
    pass

class TracksCoverflow(TracksIpodController, Coverflow):
    pass
class TracksGrid(TracksIpodController, Grid):
    pass

class TracksListSwitcher(ListSwitcherController):
    modes = [TracksVerticalWithPreview, TracksCoverflow, TracksGrid]

class TracksListSwitcherVertical(TracksListSwitcher):
    default_mode = TracksVerticalWithPreview

class TracksListSwitcherCoverflow(TracksListSwitcher):
    default_mode = TracksCoverflow

class TracksListSwitcherGrid(TracksListSwitcher):
    default_mode = TracksGrid
