# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.widgets.button import Button
from elisa.plugins.pigment.widgets.const import *
from elisa.plugins.pigment.widgets.theme import Theme

import gobject

import pgm
from pgm.timing import implicit
from pgm.timing.implicit import AnimatedObject

from pkg_resources import resource_filename

from xml.etree import ElementTree
import logging
import pprint
import copy
import sys

# TODO: implement style properties
# TODO: implement image/widget cursor
# FIXME: use a real logging system --> port the widgets to Loggable
# FIXME: better default style

class LayoutParsingError(Exception):
    pass

class UnknownLayerError(Exception):
    pass

class Key(object):
    """One key as found in the layout file. Private API."""

    def __init__ (self, **kwargs):
        try:
            self.type = kwargs.get('type', 'char')
            self.value = kwargs.get('value')
            self.display = kwargs.get('display')
            self.image_resource = kwargs.get('image-resource')
            self.width = float(kwargs.get('width', 1))
            self.layers = map(str.strip, kwargs.get('layers', '').split(','))
        except ValueError, e:
            raise LayoutParsingError(e)

        self.row = None

    @property
    def label(self):
        """The actual label to display on the key."""
        return str(self.display or self.value or '')

    def __repr__(self):
        return pprint.pformat(self.__dict__)


class Row(object):
    """A row of keys in the layout file. Private API."""

    def __init__(self):
        self.keys = []

    def append_key(self, key):
        """Add a key in the row, adding a back reference in the key."""
        self.keys.append(key)
        key.row = self


class Layer(object):
     """A layer of the keybord in the layout file. Private API."""

     def __init__(self, id):
         self.id = id
         self.rows = []


class KeyboardButton(Button):
    """Key widget.

       Not using Button directly allows to style keyboard buttons indipendently
       from others.
    """

    def _update_style_properties(self, props=None):
        super(KeyboardButton, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'padding':
                pass


class KeyboardCursor(Widget):
    """Keyboard cursor."""


class OnScreenKeyboard(Widget):
    """On-screen keyboard widget.

    @ivar selected_key: the currently selected key, as (row, column)
    @type selected_key: tuple of int
    @ivar layers: the list of swappable layers' ids defined for the keyboard
    @type layers: list of strings
    """

    __gsignals__ = {'key-press': (gobject.SIGNAL_RUN_LAST,
                                  gobject.TYPE_BOOLEAN,
                                  (str,)),
                    'key-release': (gobject.SIGNAL_RUN_LAST,
                                    gobject.TYPE_BOOLEAN,
                                    (str,))}

    def __init__(self, layout_file=None, layer="layer1"):
        super(OnScreenKeyboard, self).__init__()

        self._background = Image()
        self.add(self._background, forward_signals=False)
        self._background.visible = True

        self.selected_key = None

        self._layout_file = layout_file \
                            or resource_filename('elisa.plugins.pigment.widgets',
                                                 'mockup_keyboard.xml')
        self._layers = {}
        self._layer_widgets = {}
        self._fixed_layer = None
        self._selected_layer = layer
        self._keys = []
        self._fixed_keys = []
        self._cursor = None
        self._animated_cursor = None

        self._parse_layout()
        self._render()

        self._set_cursor()

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(OnScreenKeyboard, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'text-color':
                for key in self._get_key_buttons():
                    # FIXME: using internal API of Button
                    key._label.fg_color = value
            elif key == 'text-height':
                for key in self._get_key_buttons():
                    # FIXME: using internal API of Button
                    key._label.height = value
                    key._label.y = 0.5 - (value / 2.0)
                self.regenerate()
            elif key == 'background-color':
                self._background.bg_color = value
                self.opacity = value[-1]

    def _set_cursor(self):
        if self._cursor:
            self.remove(self._cursor)

        self._cursor = Image()
        self.add(self._cursor, forward_signals=False)
        self._cursor.bg_color = (0, 255, 0, 200)
        self._cursor.position = (0.0, 0.0, 0.0)
        self._cursor.size = (0.0, 0.0)
        self._cursor.visible = True
        self._animated_cursor = AnimatedObject(self._cursor)
        settings = {'duration': 300,
                    'transformation': implicit.DECELERATE,
                    'resolution': 5}
        self._animated_cursor.setup_next_animations(**settings)

    def _parse_layout(self):
        keyboard = ElementTree.parse(self._layout_file)

        for layer_el in keyboard.getiterator('layer'):
            layer_id = layer_el.attrib.get('id')
            if not layer_id:
                print "Skipping layer without 'id' attribute"
                continue
            layer = Layer(layer_id)
            for row_el in layer_el.getiterator('row'):
                row = Row()
                for key_el in row_el.getiterator('key'):
                    key = Key(**key_el.attrib)
                    row.append_key(key)
                layer.rows.append (row)

            if layer_id == 'fixed':
                self._fixed_layer = layer
            else:
                self._layers[layer_id] = layer

    def _render(self):
        for layer_id, layer_widget in self._layer_widgets.items():
            self.remove(layer_widget)
            self._layer_widgets.pop(layer_id)

        self._render_layer('fixed')
        self._render_layer(self._selected_layer)

    def _render_layer(self, id):
        if id == 'fixed':
            if self._fixed_layer == None:
                print "No fixed layer"
                return
            else:
                layer = self._fixed_layer
                self._fixed_keys = []
                keys = self._fixed_keys
        else:
            layer = self._layers.get(id)
            self._keys = []
            # copy the keys of the fixed layer
            for r in self._fixed_keys:
                 self._keys.append(copy.copy(r))
            keys = self._keys
            if not layer:
                print "No layer named", id
                return

        if len(layer.rows) == 0:
            print "No rows in the layer"
            return

        print "Rendering layer", id
        layer_widget = Widget()
        self.add(layer_widget, forward_signals=False)
        layer_widget.size = (1.0, 1.0)

        theme = Theme.get_default()

        row_height = 1.0 / len(layer.rows)
        for i, row in enumerate(layer.rows):
            try:
                keys[i]
            except:
                keys.append([])
                assert len(keys) == (i + 1)

            if len(row.keys) == 0:
                continue

            key_width = 1.0 / sum([k.width for k in row.keys])
            key_x = 0
            for j, key in enumerate(row.keys):
                # FIXME: this stops the row! why?!
                if key.type == 'placeholder':
                    key_x += key.width
                    continue

                key_widget = KeyboardButton()

                keys[i].append(key_widget)

                key_widget.connect("clicked", self._set_selected_key)
                # HACK: store the value of the key
                key_widget._value = key.value
                if key.type == 'layer_switch':
                    key_widget.connect("clicked", self._swap_layers, key.layers)
                elif key.type == 'char':
                    key_widget.connect("clicked", self._emit_text, key.value)

                layer_widget.add(key_widget, forward_signals=False)

                width = key_width * key.width
                height = row_height - (key_widget.style.padding * 2)
                key_widget.size = (width, height)
                x = key_x
                y = (row_height * i) + key_widget.style.padding
                z = 0.0
                key_widget.position = (x, y, z)
                key_widget.bg_color = (0, 0, 0, 0)
                key_widget.fg_color = (255, 255, 255, 255)
                key_widget.label = key.label

                # FIXME: using internal API of Button
                y = (1.0 - self.style.text_height) / 2.0
                key_widget._label.position = (0.0, y, 0.0)
                key_widget._label.size = (1.0, self.style.text_height)
                key_widget._label.font_height = 1.0
                #key_widget._label.bg_color = (255, 0, 0, 100)
                key_widget._label.fg_color = self.style.text_color
                key_widget._label.weight = pgm.TEXT_WEIGHT_BOLD
                key_widget._label.alignment = pgm.TEXT_ALIGN_CENTER

                if key.image_resource:
                    image_file = theme.get_resource(key.image_resource)
                    if image_file:
                        label_width = key_widget.style.label_width
                        image_width = key_widget.style.image_width

                        image = Image()
                        key_widget.add(image)
                        image.set_from_file(image_file)
                        image.size = (image_width, self.style.text_height)
                        image.bg_color = (0, 0, 0, 0)
                        image.y = 0.5 - (self.style.text_height / 2.0)
                        image.visible = True

                        if key_widget.label == '':
                            image.x = (1.0 - image_width) / 2.0
                        else:
                            image.x = (1.0 - label_width - image_width) / 2.0

                            # FIXME: using internal API of Button
                            key_widget._label.width = label_width
                            key_widget._label.x = image.x + image.width + key_widget.style.spacing
                            key_widget._label.alignment = pgm.TEXT_ALIGN_LEFT

                key_x += key_widget.width
                key_widget.width = (key_width * key.width) - (key_widget.style.padding * 2)

                key_widget.visible = True

        layer_widget.opacity = 0
        layer_widget.visible = True

        widget_animated = AnimatedObject(layer_widget)
        widget_animated.update_animation_settings(duration=400)
        widget_animated.opacity = 255

        self._layer_widgets[id] = layer_widget

    def _get_key_buttons(self):
        for row in self._keys:
            for key in row:
                yield key

    def _set_selected_key(self, key, *placeholders):
        for i, row in enumerate(self._keys):
            for j, item in enumerate(row):
                if self._keys[i][j] == key:
                    self.selected_key = (i, j)
                    self._animated_cursor.position = (key.x, key.y, 0.0)
                    self._animated_cursor.size = (key.width, key.height)
                    return True
        return True

    def _swap_layers(self, key, x, y, z, type, time, data, layers):
        try:
            current = layers.index(self.get_layer())
        except ValueError:
            print "Cannot swap layers: current layer is unknown"
            raise UnknownLayerError()

        self.set_layer(layers[(current + 1) % len(layers)])

    def _emit_text(self, key, x, y, z, type, time, data, value):
        self.emit("key-press", value)

    @property
    def layers(self):
        """Get the list of swappable layers' ids defined for the keyboard.

        @rtype: list of strings
        """
        return self._layers.keys()

    def get_language(self):
        """Get the language of the keyabord.

        @returns: the ISO 639-1 code of the language
        @rtype: string
        """

    def get_layer(self):
        """Get the current active layer.

        @returns: the layer id
        @rtype: string
        """
        return self._selected_layer

    def set_layer(self, id):
        """Set the active layer.

        @param id: the id of the layer to set
        @type id: string
        """
        for layer_id, layer_widget in self._layer_widgets.items():
            if layer_id != 'fixed':
                self._selected_layer = id
                widget_animated = AnimatedObject(layer_widget)

                # Attention! Pass the correct context here.
                def end_animation(timer, layer_widget=layer_widget, layer_id=layer_id):
                    self.remove(layer_widget)
                    self._layer_widgets.pop(layer_id)
                    self._render_layer(id)
                    self._set_cursor()

                widget_animated.update_animation_settings(duration=400,
                                                          end_callback=end_animation)
                widget_animated.opacity = 0

    def move_cursor(self, direction):
        """Move the cursor in the specified direction.

        @param direction: where to move the cursor
        @type direction: one of L[elisa.plugins.pigment.widgets.const.[LEFT|RIGHT|TOP|BOTTOM]}
        """
        if not self.selected_key:
            row, column = (0, 0)
        else:
            row = self.selected_key[0]
            column = self.selected_key[1]
            current = self._keys[row][column]

            if direction == LEFT:
                if column == 0:
                    column = len(self._keys[row]) - 1
                else:
                    column -= 1
            elif direction == RIGHT:
                if column >= len(self._keys[row]) - 1:
                    column = 0
                else:
                    column += 1
            elif direction == TOP:
                if row == 0:
                    row = len(self._keys) - 1
                else:
                    while len(self._keys[row - 1]) == 0:
                        row -= 1
                    row -= 1
                deltas = map(lambda w: abs(current.x - w.x), self._keys[row])
                column = deltas.index(min(deltas))
            elif direction == BOTTOM:
                if row >= len(self._keys) - 1:
                    row = 0
                else:
                    while len(self._keys[row + 1]) == 0:
                        row += 1
                    row += 1
                deltas = map(lambda w: abs(current.x - w.x), self._keys[row])
                column = deltas.index(min(deltas))

        self._set_selected_key(self._keys[row][column])
        self._keys[row][column].focus = True

    def do_key_press_event(self, viewport, event, widget):
        """Default handler for the key-press-event"""

        if event.keyval == pgm.keysyms.Return:
            key = self._keys[self.selected_key[0]][self.selected_key[1]]
            key.emit("pressed", 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
            key.emit("clicked", 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
            key.emit("released", 0, 0, 0, pgm.BUTTON_LEFT, 0)
        elif event.keyval == pgm.keysyms.Left:
            self.move_cursor(LEFT)
        elif event.keyval == pgm.keysyms.Right:
            self.move_cursor(RIGHT)
        elif event.keyval == pgm.keysyms.Up:
            self.move_cursor(TOP)
        elif event.keyval == pgm.keysyms.Down:
            self.move_cursor(BOTTOM)
        else:
            found = None
            for key in self._get_key_buttons():
                value = unicode((key._value or '').lower())
                pgm_value = pgm.keyval_to_unicode(event.keyval)
                unicode_value = unichr(pgm_value)
                if value == unicode_value:
                    self._set_selected_key(key)
                    key.focus = True
                    break

    def do_key_release_event(self, viewport, event, widget):
        """Default handler for the key-release-event"""
        self.emit("key-release")

    @classmethod
    def _demo_widget(cls, *args, **kwargs):
        if sys.argv > 1:
            widget = cls(*sys.argv[1:])
        else:
            widget = cls()

        widget.visible = True

        def on_key_press(keyboard, text):
            print "OnScreenKeyboard key-press:", text

        widget.connect("key-press", on_key_press)

        return widget

    @classmethod
    def _set_demo_widget_defaults(cls, widget, canvas, viewport):
        Widget._set_demo_widget_defaults(widget, canvas, viewport)
        widget.size = (4.0, 3.0)
        widget.regenerate()


if __name__ == '__main__':
    logger = logging.getLogger()
    logger.setLevel(logging.WARNING)

    keyboard = OnScreenKeyboard.demo()
    try:
        __IPYTHON__
    except NameError:
        pgm.main()

