# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.poblesec.filesystem_controller import \
                                            VerticalWithFastScrollAndPreview
from elisa.plugins.poblesec.list_preview import PreviewListController                                         
from elisa.plugins.poblesec.section import MenuItemWidget
from elisa.plugins.pigment.widgets.list_vertical import ListVertical
from elisa.plugins.poblesec.sliced_image import SlicedImageHorizontal
from elisa.plugins.poblesec.section import SectionMenuController
from elisa.plugins.poblesec.link import Link

from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.pigment.graph.image import Image
from elisa.plugins.pigment.widgets import notifying_list

from elisa.core.input_event import *
from elisa.core.media_uri import MediaUri
from elisa.core.common import application

from pgm.timing import implicit
import pgm, os, gobject


from elisa.core.utils.i18n import install_translation

_ = install_translation('poblesec')

class ActionSelector(Widget):

    def __init__(self):
        super(ActionSelector, self).__init__()
        self._cursors = {}
        self._update_style_properties(self._style.get_items())

    def create_cursor(self, name):
        cursor = SlicedImageHorizontal()
        self.add(cursor)
        cursor.opacity = 0
        cursor.visible = True

        animated_cursor = implicit.AnimatedObject(cursor)
        settings = {'duration': 200,
                    'transformation': implicit.DECELERATE}
        animated_cursor.setup_next_animations(**settings)
        animated_cursor.mode = implicit.REPLACE

        # store the cursor
        self._cursors[name] = animated_cursor

        # if it's the only cursor show it
        if len(self._cursors) == 1:
            cursor.opacity = 255

        self._update_cursor_style(name)

    def set_left_cap(self, image_path):
        for name, animated_cursor in self._cursors.iteritems():
            animated_cursor.static_object.left_cap.set_from_file(image_path)

    def set_right_cap(self, image_path):
        for name, animated_cursor in self._cursors.iteritems():
            animated_cursor.static_object.right_cap.set_from_file(image_path)

    def set_body(self, image_path):
        for name, animated_cursor in self._cursors.iteritems():
            animated_cursor.static_object.body.set_from_file(image_path)

    def select_cursor(self, name):
        for local_name, animated_cursor in self._cursors.iteritems():
            if local_name != name:
                animated_cursor.opacity = 0

        self._cursors[name].opacity = 255

    def _update_style_properties(self, props=None):
        super(ActionSelector, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            # separate the first fragment from the rest of the key
            # it is considered as being the name of the cursor
            # what's left is considered as being the cursor's attribute
            cursor_name, attribute = key.split('-', 1)
            if self._cursors.has_key(cursor_name):
                cursor = self._cursors[cursor_name].static_object
                setattr(cursor, attribute, value)

    def _update_cursor_style(self, name):
        cursor = self._cursors[name].static_object

        for key, value in self._style.get_items().iteritems():
            # separate the first fragment from the rest of the key
            # it is considered as being the name of the cursor
            # what's left is considered as being the cursor's attribute
            cursor_name, attribute = key.split('-', 1)
            if cursor_name == name:
                setattr(cursor, attribute, value)


class ActionListSelector(ListVertical):

    action_selected = False
    widget_signals = {'clicked' : 'item-clicked', 'action': 'action-clicked'}


    __gsignals__ = {
        'action-clicked': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        }

    def select_list_item(self):
        if self._selector is None:
            return
        self.action_selected = False
        self._selector.select_cursor("item")

    def select_action_item(self):
        if self._selector is None:
            return
        self.action_selected = True
        self._selector.select_cursor("action")


class MenuItemWidgetWithAction(MenuItemWidget):

    __gsignals__ = {
        'action': (gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
                (gobject.TYPE_PYOBJECT,)),
        }

    def _create_widgets(self):
        super(MenuItemWidgetWithAction, self)._create_widgets()

        #FIXME use a button instead of two drawable
        self.action_image = Image()
        self.action_image.bg_a = 0
        self.action_image.connect('clicked', self._action_clicked)
        self.add(self.action_image, forward_signals=False)

        self.action_image.visible = True

    def _action_clicked(self, *args):
        self.emit('action', None)

    def _update_style_properties(self, props=None):
        super(MenuItemWidgetWithAction, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'action-x':
                self.action_image.x = value
            elif key == 'action-y':
                self.action_image.y = value
            elif key == 'action-width':
                self.action_image.width = value
            elif key == 'action-height':
                self.action_image.height = value

class MenuItem(object):

    media_type = None
    label = None
    icon = None
    uri = None

class DirectoryConfigureController(PreviewListController):
    
    node_widget = MenuItemWidgetWithAction
    list_widget = ActionListSelector
    
    def initialize(self, **args):
        dfr = super(DirectoryConfigureController, self).initialize()

        uri = args['uri']
        
        self.model = notifying_list.List()
        mi = MenuItem()
        mi.label = _("Add in Music Section")
        mi.media_type = "music"
        mi.icon = "elisa.plugins.poblesec.music_section"
        mi.uri = uri
        self.model.append(mi)
        
        mi = MenuItem()
        mi.label = _("Add in Video Section")
        mi.media_type = "video"
        mi.icon = "elisa.plugins.poblesec.video_section"
        mi.uri = uri
        self.model.append(mi)
        
        mi = MenuItem()
        mi.label = _("Add in Pictures Section")
        mi.media_type = "pictures"
        mi.icon = "elisa.plugins.poblesec.pictures_section"
        mi.uri = uri
        self.model.append(mi)
        
        return dfr
          
    def nodes_setup(self):
        super(DirectoryConfigureController,self).nodes_setup()
        #FIXME: why the renderer and the list are defined with class members
        #and not the selector ?
        self.selector = ActionSelector()
 
        self.selector.create_cursor("item")
        self.selector.create_cursor("action")

        self.nodes.connect('item-clicked', self.node_clicked)
        self.nodes.connect('action-clicked', self.action_clicked)

        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        self.selector.set_left_cap(left_cap)
        self.selector.set_right_cap(right_cap)
        self.selector.set_body(body)

        self.nodes.set_selector(self.selector)
    
    def node_renderer(self, item, widget):
        super(DirectoryConfigureController,self).node_renderer(item, widget)
        widget.label.label = item.label
        self.frontend.load_from_theme(item.icon, widget.icon)

        if item.uri.path in application.media_directories.directories[item.media_type]:
            lft = self.frontend.load_from_theme
            lft('elisa.plugins.poblesec.directory_settings_added', 
                                                        widget.action_image)
        else:
            lft = self.frontend.load_from_theme
            lft('elisa.plugins.poblesec.directory_settings_add', 
                                                        widget.action_image)

    def handle_input(self, manager, input_event):
        if  self.nodes.focus:
            if input_event.value == EventValue.KEY_GO_LEFT:
                if self.nodes.action_selected == True:
                    self.nodes.select_list_item()
                    return True
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                if self.nodes.action_selected == False:
                    self.nodes.select_action_item()
                    return True
            elif input_event.value == EventValue.KEY_OK:
                if self.nodes.action_selected == True:
                    index = self.nodes.selected_item_index
                    item = self.model[index]
                    self.toggle_for_model(item, index)
                    return True
    
        return super(DirectoryConfigureController,self).handle_input(manager, 
                                                                    input_event)

    def action_clicked(self, widget, model):
        self.toggle_for_model(model)

    def toggle_for_model(self, model, index=-1):
        application.media_directories.toggle_directory_installation \
                                    (model.uri, model.media_type)
        # that will trigger a refresh of the list widgets using
        # self.model
        if index == -1:
            index = self.model.index(model)
        self.model[index] = model



class MediaDirectoryList(SectionMenuController):
    
    def initialize(self, media_type):
        dfr = super(MediaDirectoryList, self).initialize()
        application.media_directories.refresh_directory_list(media_type)
        self.media_type = media_type
        self.widget.connect('focus', self._focus_cb)
        return dfr

    def set_frontend(self, frontend):
        super(MediaDirectoryList, self).set_frontend(frontend)
        self.list_media_directory(self.media_type)

    def list_media_directory(self, media_type):  
        application.media_directories.refresh_directory_list(media_type)
        dirs = application.media_directories.get_directories(media_type)

        for directory in dirs:
            link = Link()
            uri = MediaUri("file://%s" % directory)
            link.controller_path = "/poblesec/%s/filesystem" % media_type

            # translate the type of the section into a "media type"
            # (types defined in elisa.plugins.base.models.file.FileModel)
            translation = {'music': 'audio',
                           'video': 'video',
                           'pictures': 'image'}

            file_media_type = translation[media_type]
            link.controller_args = { 'uri': uri, 'media_type' : file_media_type }
            link.label = uri.filename
            link.icon = "elisa.plugins.poblesec.%s_folders" % media_type
            self.model.append(link)
        
        add_directory = Link()
        add_directory.controller_path = "/poblesec/settings/directory_settings"
        add_directory.label = _("Add folders")
        add_directory.icon = "elisa.plugins.poblesec.directory_settings_add_folder"
        add_directory.controller_args = {'media_type': media_type}
        self.model.append(add_directory)

    def _focus_cb(self, widget, focus):
        if focus:
            self.model[:] = []
            self.list_media_directory(self.media_type)

class DirectorySettingsController(VerticalWithFastScrollAndPreview):
    
    hide_files = True
    node_widget = MenuItemWidgetWithAction
    list_widget = ActionListSelector
    default_uri = MediaUri("file://"+os.path.expanduser("~"))
    media_type = None
        
    def initialize(self, uri=default_uri, media_type=media_type):
        dfr = super(DirectorySettingsController, self).initialize(uri)
        application.media_directories.refresh_directory_list(media_type)
        self.media_type = media_type
        return dfr

    def nodes_setup(self):
        super(DirectorySettingsController,self).nodes_setup()
        #FIXME: why the renderer and the list are defined with class members
        #and not the selector ?
        self.selector = ActionSelector()

        # connect to the signals
        self.nodes.connect('item-clicked', self.node_clicked)
        self.nodes.connect('action-clicked', self.action_clicked)

        self.selector.create_cursor("item")
        self.selector.create_cursor("action")

        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        self.selector.set_left_cap(left_cap)
        self.selector.set_right_cap(right_cap)
        self.selector.set_body(body)

        self.nodes.set_selector(self.selector)
    
    def node_renderer(self, item, widget):
        super(DirectorySettingsController,self).node_renderer(item, widget)
        
        application.media_directories.refresh_directory_list(self.media_type)
        if item.uri.path in application.media_directories.all_directories:
            lft = self.frontend.load_from_theme
            lft('elisa.plugins.poblesec.directory_settings_added', 
                                                        widget.action_image)
        else:
            lft = self.frontend.load_from_theme
            lft('elisa.plugins.poblesec.directory_settings_add', 
                                                        widget.action_image)

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            file_uri = model.uri

            # we assume there was only one controller created with that path
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            dfr = browser.history.append_controller(self.path, model.name,
                                                    uri=file_uri,
                                                    media_type=self.media_type)

    def action_clicked(self, widget, model):
        self.configure_directory(model)

    def configure_directory(self, model, index=-1):
        if index == -1:
            index = self.model.index(model)
        if self.media_type:
            application.media_directories.toggle_directory_installation \
                                                    (model.uri, self.media_type)
            self.model[index] = model
        else:
            file_uri = model.uri

            # we assume there was only one controller created with that path
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            path = '/poblesec/settings/directory_selection_settings'
            dfr = browser.history.append_controller(path, model.name, uri=file_uri)

    def handle_input(self, manager, input_event):
        
        if  self.nodes.focus:
            if input_event.value == EventValue.KEY_GO_LEFT:
                if self.nodes.action_selected == True:
                    self.nodes.select_list_item()
                    return True
            elif input_event.value == EventValue.KEY_GO_RIGHT:
                if self.nodes.action_selected == False:
                    self.nodes.select_action_item()
                    return True
            elif input_event.value == EventValue.KEY_OK:
                if self.nodes.action_selected == True:
                    index = self.nodes.selected_item_index
                    model = self.nodes.model[index]
                    self.configure_directory(model, index)
                    return True
    
        return super(DirectorySettingsController,self).handle_input(manager, 
                                                                    input_event)
