# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.plugins.pigment.widgets.list_vertical import ListVertical
from elisa.plugins.pigment.widgets.widget import Widget
from elisa.plugins.pigment.graph.text import Text
from elisa.plugins.poblesec.vertical_list import VerticalListController
from elisa.plugins.poblesec.sliced_image import SlicedImageHorizontal

from elisa.core.input_event import *

import pgm

class Shortcut(Widget):
    def __init__(self):
        super(Shortcut, self).__init__()

        self.letter = Text()
        self.add(self.letter)
        self.letter.width, self.letter.height = (1.0, 0.7)
        x = 0.0
        y = (1.0-self.letter.height)/2.0
        self.letter.x, self.letter.y = (x, y)
        self.letter.bg_color = (0, 0, 0, 0)
        self.letter.alignment = pgm.TEXT_ALIGN_CENTER
        self.letter.weight = pgm.TEXT_WEIGHT_BOLD
        self.letter.visible = True

        self._update_style_properties(self._style.get_items())

    def _update_style_properties(self, props=None):
        super(Shortcut, self)._update_style_properties(props)

        if props is None:
            return

        for key, value in props.iteritems():
            if key == 'font-family':
                self.letter.font_family = value

class FastScrollListController(VerticalListController):

    def set_frontend(self, frontend):
        super(FastScrollListController, self).set_frontend(frontend)

        self.fast_scroller = ListVertical(Shortcut, visible_range_size=13)
        self.fast_scroller.focus_on_click = False
        self.widget.add(self.fast_scroller)
        self.fast_scroller.width, self.fast_scroller.height = (0.05, 0.8)
        y = (1.0-self.fast_scroller.height)/2.0
        self.fast_scroller.x, self.fast_scroller.y = (0.0, y)
        self.fast_scroller.visible = True

        def shortcut_changed(widget, shortcut):
            self._shortcut_selected(shortcut)

        self.fast_scroller.connect('item-clicked', shortcut_changed)
        self.fast_scroller.set_model(self.shortcuts)
        self.fast_scroller.set_renderer(self._shortcut_renderer)

        selector = SlicedImageHorizontal()
        theme = self.frontend.get_theme()
        left_cap = theme.get_resource('elisa.plugins.poblesec.selector_left_cap')
        right_cap = theme.get_resource('elisa.plugins.poblesec.selector_right_cap')
        body = theme.get_resource('elisa.plugins.poblesec.selector_body')
        selector.left_cap.set_from_file(left_cap)
        selector.right_cap.set_from_file(right_cap)
        selector.body.set_from_file(body)

        self.fast_scroller.set_selector(selector)

    def _get_active_shortcuts(self):
        def remove_duplicates(alist):
            # remove duplicates from a list preserving the order
            set = {}
            return [set.setdefault(e,e) for e in alist if e not in set]
 
        letters = [self.item_to_letter(item) for item in self.model]
        return remove_duplicates(letters)

    def _get_current_shortcut(self):
        return self.shortcuts[self.fast_scroller.selected_item_index]

    def _set_current_shortcut(self, shortcut):
        self.fast_scroller.selected_item_index = self.shortcuts.index(shortcut)

    def _shortcut_renderer(self, shortcut, widget):
        if shortcut in self._get_active_shortcuts():
            widget.letter.fg_color = (100, 100, 100, 255)
        else:
            widget.letter.fg_color = (20, 20, 20, 255)
        widget.letter.label = shortcut.upper()

    def item_to_letter(self, item):
        return ""

    def _shortcut_selected(self, shortcut):
        # FIXME: stupid search: assumes that model is already alphabetically
        # sorted.
        for index, item in enumerate(self.model):
            if self.item_to_letter(item) == shortcut:
                self.nodes.selected_item_index = index
                break

    def handle_input(self, manager, input_event):
        if self.nothing_to_display_widget.visible:
            return False

        if input_event.value == EventValue.KEY_GO_LEFT:
            if self.fast_scroller.focus:
                return False
            else:
                self.fast_scroller.focus = True
                # select the letter corresponding to the current item in the
                # list
                current_item = self.model[self.nodes.selected_item_index]
                current_letter = self.item_to_letter(current_item)
                if current_letter in self.shortcuts:
                    self._set_current_shortcut(current_letter)
                return True
        elif input_event.value == EventValue.KEY_GO_RIGHT:
            if self.nodes.focus:
                return False
            else:
                self.nodes.focus = True
                return True

        if self.fast_scroller.focus:
            if input_event.value == EventValue.KEY_GO_UP:
                # scroll to the previous shortcut that is activated
                current_shortcut = self._get_current_shortcut()
                active_shorcuts = self._get_active_shortcuts()
                active_current_index = active_shorcuts.index(current_shortcut)
                try:
                    next_index = active_current_index-1
                    if next_index < 0:
                        return True
                    next_shortcut = active_shorcuts[active_current_index-1]
                    self._set_current_shortcut(next_shortcut)
                except IndexError:
                    # such a shortcut does not exist
                    pass
                return True
            elif input_event.value == EventValue.KEY_GO_DOWN:
                # scroll to the next shortcut that is activated
                current_shortcut = self._get_current_shortcut()
                active_shorcuts = self._get_active_shortcuts()
                active_current_index = active_shorcuts.index(current_shortcut)
                try:
                    next_index = active_current_index+1
                    next_shortcut = active_shorcuts[next_index]
                    self._set_current_shortcut(next_shortcut)
                except IndexError:
                    # such a shortcut does not exist
                    pass
                return True
            elif input_event.value == EventValue.KEY_OK:
                shortcut = self._get_current_shortcut()
                self._shortcut_selected(shortcut)
                return True

        return super(FastScrollListController, self).handle_input(manager,
                                                                  input_event)
