# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.input_event import *
from elisa.core.log import Loggable
from elisa.core.common import application
from elisa.core.utils import defer, cancellable_defer
from elisa.core.application import PICTURES_CACHE
from elisa.core.media_uri import MediaUri

from elisa.plugins.base.models.file import FileModel
from elisa.plugins.base.models.media import RawDataModel

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.widgets.list import List
from elisa.plugins.pigment.widgets import notifying_list
from elisa.plugins.pigment.widgets.list_horizontal import ListHorizontal
from elisa.plugins.poblesec.player_video import PlayerOsd as VideoPlayerOsd
from elisa.plugins.poblesec.player_video import PlayerOsdStatusMixin
from elisa.plugins.poblesec.long_loading_image import LongLoadingImage

from twisted.internet import reactor
import gobject
import os, os.path, md5

from pgm.utils import maths


class SlideshowPlayerOsdStatus(PlayerOsdStatusMixin):
    factor = 1
    def set_duration(self, length):
        self.progress_bar.items_number = length * self.factor
        self._formated_duration = length
        self._update_counter()

    def _second_to_readable(self, seconds):
        return seconds + 1

class PlayerOsd(VideoPlayerOsd):

    status_widget = SlideshowPlayerOsdStatus


class ListFading(List):

    preloaded = 1

    def __init__(self):
        List.__init__(self, widget_class=LongLoadingImage, visible_range_size=1)
        self.focus_on_click = False

    def _piecewise_interpolation(self, x, y, factor):
        # specific transformation to map the list items with the given
        # segments to interpolate for
        factor += 0.5
        t = self._visible_range_size
        x = map(lambda a: t*a, x)

        # clamp after lower and upper limits
        if factor < x[0]:
            return y[0]
        elif factor > x[-1]:
            return y[-1]
        else:
            # general case: looking for the segment where factor belongs
            i = 0
            while factor > x[i+1]:
                i += 1

            # factor must be between 0.0 and 1.0
            new_factor = (factor - x[i]) / (x[i+1] - x[i])
            return maths.lerp(y[i], y[i+1], new_factor)

    def do_item_clicked(self, item):
        # avoid default behaviour when an item is clicked: selecting that item
        return True

    def _prepare_widget(self, widget):
        widget.position = (0.0, 0.0, 0.0)
        widget.size = (1.0, 1.0)

    def _layout_widget(self, widget, position):
        widget.opacity = self.compute_opacity(position)

    def compute_opacity(self, index):
        x = [0.0, 0.5, 1.0]
        y = [0, 255, 0]

        return self._piecewise_interpolation(x, y, index)

    def _selected_to_range_start(self, selected):
        visible_range_start = selected
        return visible_range_start

    def _range_start_to_selected(self, visible_range_start):
        selected = visible_range_start
        selected = int(round(selected))
        selected = maths.clamp(selected, 0, len(self.model)-1)
        return selected

class Slideshow(gobject.GObject, Loggable):
    """
    @ivar status:   DOCME
    @type status:   one of [Slideshow.STOPPED, Slideshow.PLAYING, Slideshow.PAUSED]
    @ivar duration: delay between two pictures (in seconds)
    @type duration: float
    """

    STOPPED = 0
    PLAYING = 1
    PAUSED = 2

    __gsignals__ = {'current-index-changed': (gobject.SIGNAL_RUN_LAST,
                                              gobject.TYPE_NONE,
                                          (gobject.TYPE_INT, gobject.TYPE_INT)),
                    'status-changed': (gobject.SIGNAL_RUN_LAST,
                                      gobject.TYPE_BOOLEAN,
                                      (gobject.TYPE_INT,)),
                   }

    def __init__(self):
        super(Slideshow, self).__init__()
        self.logCategory = "slideshow"

        self.status = self.STOPPED
        self.duration = 5.0
        self._call_later = None
        self.playlist = notifying_list.List()
        self.current_index = -1

        self.background = ListFading()

        def on_selected_item_changed(background, item, prev_item):
            index = self.background.model.index(item)
            self._set_current_index(index)

        self.background.connect('selected-item-changed',
                                on_selected_item_changed)

        self.background.set_model(self.playlist)
        self.background.set_renderer(self._renderer)
        self.background.visible = True

        # pictures cache directory management
        try:
            os.makedirs(PICTURES_CACHE)
        except OSError:
            # cache directory already existing
            pass

    def _renderer(self, item, widget):
        def cancel_deferred(widget, name):
            if hasattr(widget, name):
                dfr = getattr(widget, name)
                if dfr != None and not dfr.called:
                    # Swallow the error raised by the cancellation
                    dfr.addErrback(lambda x: None)
                    dfr.cancel()
                delattr(widget, name)

        # should cancel previous dfr_thumbnail and dfr_full created for widget
        cancel_deferred(widget, "dfr_thumbnail")
        cancel_deferred(widget, "dfr_full")

        widget.clear()
        self._set_default_icon(widget)

        if item.references:
            dfr_thumbnail = self._retrieve_thumbnail(item)
            dfr_thumbnail.addCallback(self._load_thumbnail, widget)
            dfr_full = self._retrieve_full_resolution(item)
            dfr_full.addCallback(self._load_full_resolution, widget, dfr_thumbnail)
            # FIXME: storing the deferred in widget seems ugly
            setattr(widget, 'dfr_thumbnail', dfr_thumbnail)
            setattr(widget, 'dfr_full', dfr_full)
        else:
            # We have not got the list of URLs for the image yet, we may be
            # waiting for it (asynchronous filling, see Flickr).
            if hasattr(item, '_references_deferred'):
                # FIXME: Flickr specific hack
                def got_image_sizes(images):
                    delattr(item, '_references_deferred')
                    if self.current_index == self.playlist.index(item):
                        dfr_thumbnail = self._retrieve_thumbnail(item)
                        dfr_thumbnail.addCallback(self._load_thumbnail, widget)
                        dfr_full = self._retrieve_full_resolution(item)
                        dfr_full.addCallback(self._load_full_resolution, widget, dfr_thumbnail)
                        # FIXME: storing the deferred in widget seems ugly
                        setattr(widget, 'dfr_thumbnail', dfr_thumbnail)
                        setattr(widget, 'dfr_full', dfr_full)
                item._references_deferred.addCallback(got_image_sizes)

    def _set_default_icon(self, widget):
        icon = "elisa.plugins.poblesec.file_picture"
        self._load_from_theme(icon, widget.quick_image)

    def _retrieve_thumbnail(self, item):
        uri = item.references[0]
        self.debug("retrieving thumbnail image from %s" % uri)
        return self._retrieve_and_cache(uri)

    def _retrieve_full_resolution(self, item):
        uri = item.references[-1]
        self.debug("retrieving full resolution image from %s" % uri)
        return self._retrieve_and_cache(uri)

    def _retrieve_and_cache(self, uri):
        path = self._compute_cached_path(uri)

        if os.path.exists(path):
            self.debug("%s already cached at %s", uri, path)
            # already cached
            dfr = defer.Deferred()
            dfr.callback(path)
        else:
            def cache_it(result):
                if isinstance(result, RawDataModel):
                    self._cache_raw_data(result.data, path)
                    return path
                elif isinstance(result, FileModel):
                    return result.uri.path

            model, dfr = application.resource_manager.get(uri)
            dfr.addCallback(cache_it)

        return dfr

    def _load_thumbnail(self, path, widget):
        # downscaling to 1024 is our safety net here in case some broken
        # resource provider is used
        widget.quick_image.set_from_file(path, 1024)

    def _load_full_resolution(self, path, widget, dfr_thumbnail=None):
        if dfr_thumbnail != None and not dfr_thumbnail.called \
            and isinstance(dfr_thumbnail, cancellable_defer.CancellableDeferred):
            dfr_thumbnail.cancel()

        # FIXME: should downscale the image on loading to a size never bigger
        # than the viewport size nor 1024
        widget.image.set_from_file(path, 1024)

    def _compute_cached_path(self, uri):
        filename = md5.new(str(uri)).hexdigest() + '.' + uri.extension
        path = os.path.join(PICTURES_CACHE, filename)
        return path

    def _cache_raw_data(self, data, path):
        self.debug("caching image at %s", path)
        if not os.path.exists(PICTURES_CACHE):
            os.makedirs(PICTURES_CACHE, 0755)
        fd = open(path, 'wb')
        fd.write(data)
        fd.close()

    def _cancel_last_call_later(self):
        if self._call_later != None and self._call_later.active():
            self._call_later.cancel()

    def play(self):
        self.status = self.PLAYING
        self.emit('status-changed', self.status)
        self.next_image()

    def pause(self):
        self.status = self.PAUSED
        self.emit('status-changed', self.status)
        self._cancel_last_call_later()

    def stop(self):
        self.pause()
        self.clear_playlist()

    def enqueue_to_playlist(self, model):
        self.playlist.append(model)

    def jump_to_index(self, index):
        self.info("Loading image at index %r", index)
        self._set_current_index(index)
        self.background.animated = False
        self.background.selected_item_index = index
        self.background.animated = True
        self.status = self.PAUSED
        self.emit('status-changed', self.status)

    def play_at_index(self, index):
        self.info("Loading image at index %r", index)
        self._set_current_index(index)
        self.background.selected_item_index = index

    def _set_current_index(self, index):
        previous_index = self.current_index
        self.current_index = index
        self.emit('current-index-changed', index, previous_index)

    def clear_playlist(self):
        self.playlist[:] = []

    def next_image(self):
        self._cancel_last_call_later()

        index = self.current_index + 1
        if index < len(self.playlist):
            self.play_at_index(index)
            if self.status == self.PLAYING:
                self._call_later = reactor.callLater(self.duration,
                                                     self.next_image)
            return True
        else:
            self.info("Reached end of slideshow")
            return False

    def previous_image(self):
        self._cancel_last_call_later()

        index = self.current_index - 1
        if index < len(self.playlist) and index >= 0:
            self.play_at_index(index)
            if self.status == self.PLAYING:
                self._call_later = reactor.callLater(self.duration,
                                                     self.previous_image)
            return True
        else:
            self.info("Reached beginning of slideshow")
            return False


class SlideshowController(PigmentController):

    def __init__(self):
        super(SlideshowController, self).__init__()
        self.player = Slideshow()
        self.widget.add(self.player.background)
        self.player.background.position = (0.0, 0.0, -1.0)
        self.player.background.size = (1.0, 1.0)
        self.player.connect('current-index-changed', self._player_index_changed)
        self.player.playlist.notifier.connect('items-deleted',
                                              self._player_playlist_changed)
        self.player.playlist.notifier.connect('items-inserted',
                                              self._player_playlist_changed)
        self.player.connect('status-changed', self._player_status_changed)

        def on_item_clicked(widget, item):
            if self.has_focus():
                # don't increment widget.selected_item_index because
                # of the callLater in player. Better to use
                # next_image() which takes care to cancel that callLater
                self.player.next_image()
            return True

        self.player.background.connect('item-clicked', on_item_clicked)


        # On Screen Display
        self.player_osd = PlayerOsd()

        # hide the volume bar
        self.player_osd.volume.visible = False

        self.widget.add(self.player_osd)
        self.player_osd.bg_a = 0
        self.player_osd.opacity = 0
        self.player_osd.visible = True

        self.player_osd.exit.background.connect('clicked',
                                                self._exit_clicked_cb)
        self.player_osd.status.play_button.connect('clicked',
                                                   self._play_pause_clicked_cb)
        self.player_osd.status.pause_button.connect('clicked',
                                                    self._play_pause_clicked_cb)

        self.player_osd.status.connect \
                                ('progress-changed', self._progress_bar_cb)

        self.player_osd.volume.connect \
                                ('progress-changed', self._volume_bar_cb)
        self.player_osd.volume.play_button.connect \
                                ('clicked', self._volume_mute_cb)
                                
        self.player_osd.volume.pause_button.connect \
                                ('clicked', self._volume_mute_cb)


    def _exit_clicked_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        self.player_osd.hide()
        main.hide_current_player()
        return True

    def _play_pause_clicked_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.toggle_play_pause()
        return True

    def _volume_bar_cb(self, widget, position):
        self.player_osd.volume.play()
        self.mute_volume = -1
        self.player.set_volume(position / 10.0)

    def _volume_mute_cb(self, drawable, x, y, z, button, time, pressure):
        if not self.has_focus():
            return True

        self.toggle_mute()
        return True

    def _progress_bar_cb(self, widget, position):
        self.player.jump_to_index(position)

    def _player_index_changed(self, player, index, previous_index):
        # update title display
        item = self.player.playlist[index]
        try:
            if item.title == None:
                self.player_osd.status.text.label = ""
            else:
                self.player_osd.status.text.label = item.title
        except AttributeError:
            pass

        # update index display
        if not self.player_osd.status.seek_drag:
            self.player_osd.status.set_position(index)

    def _player_status_changed(self, player, status):
        if status == player.PLAYING:
            self.player_osd.status.pause()
        elif status == player.PAUSED:
            self.player_osd.status.play()
        elif status == player.STOPPED:
            self.player_osd.status.play()

    def _player_playlist_changed(self, *args):
        self.player_osd.status.set_duration(len(self.player.playlist))

    def set_frontend(self, frontend):
        super(SlideshowController, self).set_frontend(frontend)
        self._load_img_from_theme()
        self.frontend.viewport.connect('motion-notify-event',
                                       self._mouse_motion_cb)
        self.player._load_from_theme = self.frontend.load_from_theme

    def _mouse_motion_cb(self, viewport, event):
         if self.has_focus():
             self.player_osd.show()

    def _load_img_from_theme(self):
        lft = self.frontend.load_from_theme

        osd_status = self.player_osd.status
        lft('elisa.plugins.poblesec.osd_status_dock', 
                                      osd_status.dock)
        lft('elisa.plugins.poblesec.osd_status_play_button', 
                                      osd_status.play_button)
        lft('elisa.plugins.poblesec.osd_status_pause_button', 
                                      osd_status.pause_button)

        exit = self.player_osd.exit
        lft('elisa.plugins.poblesec.osd_exit_button', 
                                      exit.background)

    def toggle_play_pause(self):
        if self.player.status == self.player.PLAYING:
            self.player.pause()
            self.player_osd.show()
        else:
            self.player.play()
            self.player_osd.show()

    def handle_input(self, manager, input_event):
        if input_event.value == EventValue.KEY_MENU:
            if self.player_osd.is_visible == False:
                self.player_osd.show()
                return True
            else:
                self.player_osd.hide()
                self.player.pause()
                return False

        elif input_event.value == EventValue.KEY_OK or \
            input_event.value == EventValue.KEY_SPACE:
            self.toggle_play_pause()
            return True

        elif input_event.value == EventValue.KEY_GO_LEFT:
            self.player_osd.show()
            self.player.previous_image()

        elif input_event.value == EventValue.KEY_GO_RIGHT:
            self.player_osd.show()
            self.player.next_image()

        return False
