# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 2.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Guido Amoruso <guidonte@fluendo.com>

from twisted.trial.unittest import TestCase
from twisted.internet import defer

from elisa.core import common
from elisa.core.config import Config
from elisa.core.input_event import EventValue

from elisa.plugins.pigment.pigment_frontend import PigmentFrontend
from elisa.plugins.pigment.pigment_controller import PigmentController

from elisa.plugins.poblesec.browser_controller import BrowserController


class FakeController(PigmentController):
    def handle_input(self, manager, input_event):
        pass

class DummyController(PigmentController):
    pass

class FakePluginRegistry(object):
    def create_component(self, *args, **kwargs):
        return defer.succeed(DummyController())

class FakeApplication(object):
    config = Config()
    plugin_registry = FakePluginRegistry()

class BrokenFrontend(object):
    pass

class FakeTransition(object):
    def __init__(self, broken=False):
        self._broken = broken

    def __call__(self, controller):
        if self._broken:
            return defer.fail(controller)
        else:
            return defer.succeed(controller)

class FakeManager(object):
    pass

class FakeInputEvent(object):
    value = None


class TestBrowserController(TestCase):

    def setUp(self):
        common.application = FakeApplication()
        self.frontend = PigmentFrontend()
        self.frontend.add_controller("/fake/controller", FakeController)
        self.frontend.add_controller("/dummy/controller", DummyController)
        self.browser = BrowserController()

    def test_initialize(self):
        self.browser.initialize()
        self.failUnlessEqual(self.browser._home_path, self.browser.default_config['home'])

        home_path = '/dummy/controller'
        self.browser.initialize(home_path)
        self.failUnlessEqual(self.browser._home_path, home_path)

    def test_set_frontend(self):
        self.browser.initialize()

        dfr = self.browser.set_frontend(self.frontend)

        def on_append(result):
            self.failUnlessEqual(result.path, self.browser._home_path)

            self.failUnlessIdentical(self.browser.history.current, result)
            self.failUnlessEqual(self.browser.history.index, 0)
            self.failUnlessEqual(self.browser.history._controllers, [result])
            self.failUnlessEqual(len(self.browser.history._controllers), 1)
            self.failUnlessEqual(self.browser.history._creation_dfr.result, result)

        dfr.addCallback(on_append)

        return dfr

    def test_set_frontend_broken(self):
        self.browser.initialize()

        frontend = BrokenFrontend()
        self.failUnlessRaises(AttributeError, self.browser.set_frontend, (frontend))

        return None

    def test__push_controller_cb(self):
        self.browser.initialize()

        previous = FakeController()
        current = FakeController()

        self.browser._push_controller_cb(None, previous, current)
        self.failUnlessEqual(current.widget.focus, True)

    def test__pop_controller_cb(self):
        self.browser.initialize()

        previous = FakeController()
        current = FakeController()

        self.browser._pop_controller_cb(None, previous, current)
        self.failUnlessEqual(current.widget.focus, True)

    def test__hide_controller(self):
        self.browser.initialize()

        controller = FakeController()
        transition = FakeTransition()

        self.browser.widget.add(controller.widget)

        dfr = self.browser._hide_controller(controller, transition)

        def removed_controller(controller):
            self.failUnlessEqual(self.browser.widget.get_children(), [])
            self.failUnlessEqual(self.browser._hiding_controllers, {})

        dfr.addCallback(removed_controller)

        return dfr

    def test__hide_controller_broken(self):
        self.browser.initialize()

        controller = FakeController()
        transition = FakeTransition(broken=True)

        self.browser.widget.add(controller.widget)

        dfr = self.browser._hide_controller(controller, transition)

        def error_removing(err):
            self.failUnlessEqual(self.browser.widget.get_children(), [controller.widget])
            self.failUnlessEqual(self.browser._hiding_controllers, {})

        dfr.addErrback(error_removing)

        return dfr

    def test__show_controller(self):
        self.browser.initialize()

        controller = FakeController()
        transition = FakeTransition()

        dfr = self.browser._show_controller(controller, transition)

        self.failUnlessEqual(self.browser.widget.get_children(), [controller.widget])
        x, y = (controller.widget.x, controller.widget.y)
        width, height = (controller.widget.width, controller.widget.height)
        self.failUnlessEqual((x, y, width, height), (0, 0, 1.0, 1.0))

        return dfr

    def test__show_controller_broken(self):
        self.browser.initialize()

        controller = FakeController()
        transition = FakeTransition(broken=True)

        dfr = self.browser._show_controller(controller, transition)

        self.failUnlessEqual(self.browser.widget.get_children(), [controller.widget])
        x, y = (controller.widget.x, controller.widget.y)
        width, height = (controller.widget.width, controller.widget.height)
        self.failUnlessEqual((x, y, width, height), (0, 0, 1.0, 1.0))

        def error_showing(err):
            self.failUnlessEqual(self.browser.widget.get_children(), [controller.widget])
            self.failUnlessEqual(self.browser._hiding_controllers, {})

        dfr.addErrback(error_showing)

        return dfr

    def test_handle_input(self):
        self.browser.initialize()
        self.browser.set_frontend(self.frontend)

        manager = FakeManager()
        event = FakeInputEvent()
        self.browser.handle_input(manager, event)

        event.value = EventValue.KEY_GO_LEFT
        self.browser.handle_input(manager, event)

#        dfr = self.browser.history.append_controller("/fake/controller", "Fake")
#        dfr.addCallback(lambda res: self.browser.history.append_controller("/fake/controller", "Dummy"))
#
#        def handle_input(result):
#            self.browser.handle_input(manager, event)
#
#        dfr.addCallback(handle_input)
#
#        return dfr

